﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/workspaces/WorkSpaces_EXPORTS.h>
#include <aws/workspaces/model/ConnectionAliasAssociation.h>
#include <aws/workspaces/model/ConnectionAliasState.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace WorkSpaces {
namespace Model {

/**
 * <p>Describes a connection alias. Connection aliases are used for cross-Region
 * redirection. For more information, see <a
 * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/cross-region-redirection.html">
 * Cross-Region Redirection for Amazon WorkSpaces</a>.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/workspaces-2015-04-08/ConnectionAlias">AWS
 * API Reference</a></p>
 */
class ConnectionAlias {
 public:
  AWS_WORKSPACES_API ConnectionAlias() = default;
  AWS_WORKSPACES_API ConnectionAlias(Aws::Utils::Json::JsonView jsonValue);
  AWS_WORKSPACES_API ConnectionAlias& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_WORKSPACES_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The connection string specified for the connection alias. The connection
   * string must be in the form of a fully qualified domain name (FQDN), such as
   * <code>www.example.com</code>.</p>
   */
  inline const Aws::String& GetConnectionString() const { return m_connectionString; }
  inline bool ConnectionStringHasBeenSet() const { return m_connectionStringHasBeenSet; }
  template <typename ConnectionStringT = Aws::String>
  void SetConnectionString(ConnectionStringT&& value) {
    m_connectionStringHasBeenSet = true;
    m_connectionString = std::forward<ConnectionStringT>(value);
  }
  template <typename ConnectionStringT = Aws::String>
  ConnectionAlias& WithConnectionString(ConnectionStringT&& value) {
    SetConnectionString(std::forward<ConnectionStringT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the connection alias.</p>
   */
  inline const Aws::String& GetAliasId() const { return m_aliasId; }
  inline bool AliasIdHasBeenSet() const { return m_aliasIdHasBeenSet; }
  template <typename AliasIdT = Aws::String>
  void SetAliasId(AliasIdT&& value) {
    m_aliasIdHasBeenSet = true;
    m_aliasId = std::forward<AliasIdT>(value);
  }
  template <typename AliasIdT = Aws::String>
  ConnectionAlias& WithAliasId(AliasIdT&& value) {
    SetAliasId(std::forward<AliasIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current state of the connection alias.</p>
   */
  inline ConnectionAliasState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(ConnectionAliasState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline ConnectionAlias& WithState(ConnectionAliasState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Web Services account that owns the connection
   * alias.</p>
   */
  inline const Aws::String& GetOwnerAccountId() const { return m_ownerAccountId; }
  inline bool OwnerAccountIdHasBeenSet() const { return m_ownerAccountIdHasBeenSet; }
  template <typename OwnerAccountIdT = Aws::String>
  void SetOwnerAccountId(OwnerAccountIdT&& value) {
    m_ownerAccountIdHasBeenSet = true;
    m_ownerAccountId = std::forward<OwnerAccountIdT>(value);
  }
  template <typename OwnerAccountIdT = Aws::String>
  ConnectionAlias& WithOwnerAccountId(OwnerAccountIdT&& value) {
    SetOwnerAccountId(std::forward<OwnerAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The association status of the connection alias.</p>
   */
  inline const Aws::Vector<ConnectionAliasAssociation>& GetAssociations() const { return m_associations; }
  inline bool AssociationsHasBeenSet() const { return m_associationsHasBeenSet; }
  template <typename AssociationsT = Aws::Vector<ConnectionAliasAssociation>>
  void SetAssociations(AssociationsT&& value) {
    m_associationsHasBeenSet = true;
    m_associations = std::forward<AssociationsT>(value);
  }
  template <typename AssociationsT = Aws::Vector<ConnectionAliasAssociation>>
  ConnectionAlias& WithAssociations(AssociationsT&& value) {
    SetAssociations(std::forward<AssociationsT>(value));
    return *this;
  }
  template <typename AssociationsT = ConnectionAliasAssociation>
  ConnectionAlias& AddAssociations(AssociationsT&& value) {
    m_associationsHasBeenSet = true;
    m_associations.emplace_back(std::forward<AssociationsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectionString;

  Aws::String m_aliasId;

  ConnectionAliasState m_state{ConnectionAliasState::NOT_SET};

  Aws::String m_ownerAccountId;

  Aws::Vector<ConnectionAliasAssociation> m_associations;
  bool m_connectionStringHasBeenSet = false;
  bool m_aliasIdHasBeenSet = false;
  bool m_stateHasBeenSet = false;
  bool m_ownerAccountIdHasBeenSet = false;
  bool m_associationsHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkSpaces
}  // namespace Aws
