﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class DescribeDBProxyEndpointsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBProxyEndpointsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBProxyEndpoints"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the DB proxy whose endpoints you want to describe. If you omit
   * this parameter, the output includes information about all DB proxy endpoints
   * associated with all your DB proxies.</p>
   */
  inline const Aws::String& GetDBProxyName() const { return m_dBProxyName; }
  inline bool DBProxyNameHasBeenSet() const { return m_dBProxyNameHasBeenSet; }
  template <typename DBProxyNameT = Aws::String>
  void SetDBProxyName(DBProxyNameT&& value) {
    m_dBProxyNameHasBeenSet = true;
    m_dBProxyName = std::forward<DBProxyNameT>(value);
  }
  template <typename DBProxyNameT = Aws::String>
  DescribeDBProxyEndpointsRequest& WithDBProxyName(DBProxyNameT&& value) {
    SetDBProxyName(std::forward<DBProxyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a DB proxy endpoint to describe. If you omit this parameter, the
   * output includes information about all DB proxy endpoints associated with the
   * specified proxy.</p>
   */
  inline const Aws::String& GetDBProxyEndpointName() const { return m_dBProxyEndpointName; }
  inline bool DBProxyEndpointNameHasBeenSet() const { return m_dBProxyEndpointNameHasBeenSet; }
  template <typename DBProxyEndpointNameT = Aws::String>
  void SetDBProxyEndpointName(DBProxyEndpointNameT&& value) {
    m_dBProxyEndpointNameHasBeenSet = true;
    m_dBProxyEndpointName = std::forward<DBProxyEndpointNameT>(value);
  }
  template <typename DBProxyEndpointNameT = Aws::String>
  DescribeDBProxyEndpointsRequest& WithDBProxyEndpointName(DBProxyEndpointNameT&& value) {
    SetDBProxyEndpointName(std::forward<DBProxyEndpointNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter is not currently supported.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeDBProxyEndpointsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeDBProxyEndpointsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous request. If this
   * parameter is specified, the response includes only records beyond the marker, up
   * to the value specified by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBProxyEndpointsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so that the remaining results can be
   * retrieved.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBProxyEndpointsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBProxyName;

  Aws::String m_dBProxyEndpointName;

  Aws::Vector<Filter> m_filters;

  Aws::String m_marker;

  int m_maxRecords{0};
  bool m_dBProxyNameHasBeenSet = false;
  bool m_dBProxyEndpointNameHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
