﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/location/LocationService_EXPORTS.h>
#include <aws/location/model/MapConfiguration.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace LocationService {
namespace Model {
class DescribeMapResult {
 public:
  AWS_LOCATIONSERVICE_API DescribeMapResult() = default;
  AWS_LOCATIONSERVICE_API DescribeMapResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_LOCATIONSERVICE_API DescribeMapResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The map style selected from an available provider.</p>
   */
  inline const Aws::String& GetMapName() const { return m_mapName; }
  template <typename MapNameT = Aws::String>
  void SetMapName(MapNameT&& value) {
    m_mapNameHasBeenSet = true;
    m_mapName = std::forward<MapNameT>(value);
  }
  template <typename MapNameT = Aws::String>
  DescribeMapResult& WithMapName(MapNameT&& value) {
    SetMapName(std::forward<MapNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the map resource. Used to specify a
   * resource across all Amazon Web Services.</p> <ul> <li> <p>Format example:
   * <code>arn:aws:geo:region:account-id:map/ExampleMap</code> </p> </li> </ul>
   */
  inline const Aws::String& GetMapArn() const { return m_mapArn; }
  template <typename MapArnT = Aws::String>
  void SetMapArn(MapArnT&& value) {
    m_mapArnHasBeenSet = true;
    m_mapArn = std::forward<MapArnT>(value);
  }
  template <typename MapArnT = Aws::String>
  DescribeMapResult& WithMapArn(MapArnT&& value) {
    SetMapArn(std::forward<MapArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the data provider for the associated map tiles.</p>
   */
  inline const Aws::String& GetDataSource() const { return m_dataSource; }
  template <typename DataSourceT = Aws::String>
  void SetDataSource(DataSourceT&& value) {
    m_dataSourceHasBeenSet = true;
    m_dataSource = std::forward<DataSourceT>(value);
  }
  template <typename DataSourceT = Aws::String>
  DescribeMapResult& WithDataSource(DataSourceT&& value) {
    SetDataSource(std::forward<DataSourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the map tile style selected from a partner data provider.</p>
   */
  inline const MapConfiguration& GetConfiguration() const { return m_configuration; }
  template <typename ConfigurationT = MapConfiguration>
  void SetConfiguration(ConfigurationT&& value) {
    m_configurationHasBeenSet = true;
    m_configuration = std::forward<ConfigurationT>(value);
  }
  template <typename ConfigurationT = MapConfiguration>
  DescribeMapResult& WithConfiguration(ConfigurationT&& value) {
    SetConfiguration(std::forward<ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional description for the map resource.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  DescribeMapResult& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags associated with the map resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  DescribeMapResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  DescribeMapResult& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp for when the map resource was created in <a
   * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a>
   * format: <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetCreateTime() const { return m_createTime; }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  void SetCreateTime(CreateTimeT&& value) {
    m_createTimeHasBeenSet = true;
    m_createTime = std::forward<CreateTimeT>(value);
  }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  DescribeMapResult& WithCreateTime(CreateTimeT&& value) {
    SetCreateTime(std::forward<CreateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp for when the map resource was last update in <a
   * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a>
   * format: <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetUpdateTime() const { return m_updateTime; }
  template <typename UpdateTimeT = Aws::Utils::DateTime>
  void SetUpdateTime(UpdateTimeT&& value) {
    m_updateTimeHasBeenSet = true;
    m_updateTime = std::forward<UpdateTimeT>(value);
  }
  template <typename UpdateTimeT = Aws::Utils::DateTime>
  DescribeMapResult& WithUpdateTime(UpdateTimeT&& value) {
    SetUpdateTime(std::forward<UpdateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  DescribeMapResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_mapName;

  Aws::String m_mapArn;

  Aws::String m_dataSource;

  MapConfiguration m_configuration;

  Aws::String m_description;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::Utils::DateTime m_createTime{};

  Aws::Utils::DateTime m_updateTime{};

  Aws::String m_requestId;
  bool m_mapNameHasBeenSet = false;
  bool m_mapArnHasBeenSet = false;
  bool m_dataSourceHasBeenSet = false;
  bool m_configurationHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_createTimeHasBeenSet = false;
  bool m_updateTimeHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace LocationService
}  // namespace Aws
