﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/guardduty/GuardDuty_EXPORTS.h>
#include <aws/guardduty/model/IndicatorType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GuardDuty {
namespace Model {

/**
 * <p>Contains information about the indicators that include a set of signals
 * observed in an attack sequence.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/Indicator">AWS
 * API Reference</a></p>
 */
class Indicator {
 public:
  AWS_GUARDDUTY_API Indicator() = default;
  AWS_GUARDDUTY_API Indicator(Aws::Utils::Json::JsonView jsonValue);
  AWS_GUARDDUTY_API Indicator& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GUARDDUTY_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Specific indicator keys observed in the attack sequence. For description of
   * the valid values for key, see <a
   * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-summary.html#guardduty-extended-threat-detection-attack-sequence-finding-details">Attack
   * sequence finding details</a> in the <i>Amazon GuardDuty User Guide</i>.</p>
   */
  inline IndicatorType GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  inline void SetKey(IndicatorType value) {
    m_keyHasBeenSet = true;
    m_key = value;
  }
  inline Indicator& WithKey(IndicatorType value) {
    SetKey(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Values associated with each indicator key. For example, if the indicator key
   * is <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the
   * network. If the indicator key is <code>ATTACK_TACTIC</code>, then the value will
   * be one of the MITRE tactics. </p>
   */
  inline const Aws::Vector<Aws::String>& GetValues() const { return m_values; }
  inline bool ValuesHasBeenSet() const { return m_valuesHasBeenSet; }
  template <typename ValuesT = Aws::Vector<Aws::String>>
  void SetValues(ValuesT&& value) {
    m_valuesHasBeenSet = true;
    m_values = std::forward<ValuesT>(value);
  }
  template <typename ValuesT = Aws::Vector<Aws::String>>
  Indicator& WithValues(ValuesT&& value) {
    SetValues(std::forward<ValuesT>(value));
    return *this;
  }
  template <typename ValuesT = Aws::String>
  Indicator& AddValues(ValuesT&& value) {
    m_valuesHasBeenSet = true;
    m_values.emplace_back(std::forward<ValuesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Title describing the indicator.</p>
   */
  inline const Aws::String& GetTitle() const { return m_title; }
  inline bool TitleHasBeenSet() const { return m_titleHasBeenSet; }
  template <typename TitleT = Aws::String>
  void SetTitle(TitleT&& value) {
    m_titleHasBeenSet = true;
    m_title = std::forward<TitleT>(value);
  }
  template <typename TitleT = Aws::String>
  Indicator& WithTitle(TitleT&& value) {
    SetTitle(std::forward<TitleT>(value));
    return *this;
  }
  ///@}
 private:
  IndicatorType m_key{IndicatorType::NOT_SET};

  Aws::Vector<Aws::String> m_values;

  Aws::String m_title;
  bool m_keyHasBeenSet = false;
  bool m_valuesHasBeenSet = false;
  bool m_titleHasBeenSet = false;
};

}  // namespace Model
}  // namespace GuardDuty
}  // namespace Aws
