﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/events/CloudWatchEventsRequest.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvents {
namespace Model {

/**
 */
class ListPartnerEventSourceAccountsRequest : public CloudWatchEventsRequest {
 public:
  AWS_CLOUDWATCHEVENTS_API ListPartnerEventSourceAccountsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListPartnerEventSourceAccounts"; }

  AWS_CLOUDWATCHEVENTS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHEVENTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the partner event source to display account information
   * about.</p>
   */
  inline const Aws::String& GetEventSourceName() const { return m_eventSourceName; }
  inline bool EventSourceNameHasBeenSet() const { return m_eventSourceNameHasBeenSet; }
  template <typename EventSourceNameT = Aws::String>
  void SetEventSourceName(EventSourceNameT&& value) {
    m_eventSourceNameHasBeenSet = true;
    m_eventSourceName = std::forward<EventSourceNameT>(value);
  }
  template <typename EventSourceNameT = Aws::String>
  ListPartnerEventSourceAccountsRequest& WithEventSourceName(EventSourceNameT&& value) {
    SetEventSourceName(std::forward<EventSourceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token returned by a previous call to this operation. Specifying this
   * retrieves the next set of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListPartnerEventSourceAccountsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifying this limits the number of results returned by this operation. The
   * operation also returns a NextToken which you can use in a subsequent operation
   * to retrieve the next set of results.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline ListPartnerEventSourceAccountsRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_eventSourceName;

  Aws::String m_nextToken;

  int m_limit{0};
  bool m_eventSourceNameHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_limitHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
