﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datasync/DataSyncRequest.h>
#include <aws/datasync/DataSync_EXPORTS.h>

#include <utility>

namespace Aws {
namespace DataSync {
namespace Model {

/**
 */
class UpdateLocationFsxWindowsRequest : public DataSyncRequest {
 public:
  AWS_DATASYNC_API UpdateLocationFsxWindowsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateLocationFsxWindows"; }

  AWS_DATASYNC_API Aws::String SerializePayload() const override;

  AWS_DATASYNC_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the ARN of the FSx for Windows File Server transfer location that
   * you're updating.</p>
   */
  inline const Aws::String& GetLocationArn() const { return m_locationArn; }
  inline bool LocationArnHasBeenSet() const { return m_locationArnHasBeenSet; }
  template <typename LocationArnT = Aws::String>
  void SetLocationArn(LocationArnT&& value) {
    m_locationArnHasBeenSet = true;
    m_locationArn = std::forward<LocationArnT>(value);
  }
  template <typename LocationArnT = Aws::String>
  UpdateLocationFsxWindowsRequest& WithLocationArn(LocationArnT&& value) {
    SetLocationArn(std::forward<LocationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a mount path for your file system using forward slashes. DataSync
   * uses this subdirectory to read or write data (depending on whether the file
   * system is a source or destination location).</p>
   */
  inline const Aws::String& GetSubdirectory() const { return m_subdirectory; }
  inline bool SubdirectoryHasBeenSet() const { return m_subdirectoryHasBeenSet; }
  template <typename SubdirectoryT = Aws::String>
  void SetSubdirectory(SubdirectoryT&& value) {
    m_subdirectoryHasBeenSet = true;
    m_subdirectory = std::forward<SubdirectoryT>(value);
  }
  template <typename SubdirectoryT = Aws::String>
  UpdateLocationFsxWindowsRequest& WithSubdirectory(SubdirectoryT&& value) {
    SetSubdirectory(std::forward<SubdirectoryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the name of the Windows domain that your FSx for Windows File
   * Server file system belongs to.</p> <p>If you have multiple Active Directory
   * domains in your environment, configuring this parameter makes sure that DataSync
   * connects to the right file system.</p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  UpdateLocationFsxWindowsRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the user with the permissions to mount and access the files,
   * folders, and file metadata in your FSx for Windows File Server file system.</p>
   * <p>For information about choosing a user with the right level of access for your
   * transfer, see <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-fsx-location.html#create-fsx-windows-location-permissions">required
   * permissions</a> for FSx for Windows File Server locations.</p>
   */
  inline const Aws::String& GetUser() const { return m_user; }
  inline bool UserHasBeenSet() const { return m_userHasBeenSet; }
  template <typename UserT = Aws::String>
  void SetUser(UserT&& value) {
    m_userHasBeenSet = true;
    m_user = std::forward<UserT>(value);
  }
  template <typename UserT = Aws::String>
  UpdateLocationFsxWindowsRequest& WithUser(UserT&& value) {
    SetUser(std::forward<UserT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the password of the user with the permissions to mount and access
   * the files, folders, and file metadata in your FSx for Windows File Server file
   * system.</p>
   */
  inline const Aws::String& GetPassword() const { return m_password; }
  inline bool PasswordHasBeenSet() const { return m_passwordHasBeenSet; }
  template <typename PasswordT = Aws::String>
  void SetPassword(PasswordT&& value) {
    m_passwordHasBeenSet = true;
    m_password = std::forward<PasswordT>(value);
  }
  template <typename PasswordT = Aws::String>
  UpdateLocationFsxWindowsRequest& WithPassword(PasswordT&& value) {
    SetPassword(std::forward<PasswordT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_locationArn;

  Aws::String m_subdirectory;

  Aws::String m_domain;

  Aws::String m_user;

  Aws::String m_password;
  bool m_locationArnHasBeenSet = false;
  bool m_subdirectoryHasBeenSet = false;
  bool m_domainHasBeenSet = false;
  bool m_userHasBeenSet = false;
  bool m_passwordHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataSync
}  // namespace Aws
