﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/ViewInputContent.h>
#include <aws/connect/model/ViewStatus.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class UpdateViewContentRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API UpdateViewContentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateViewContent"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the Amazon Connect instance. You can find the instanceId in
   * the ARN of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  UpdateViewContentRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the view. Both <code>ViewArn</code> and <code>ViewId</code>
   * can be used.</p>
   */
  inline const Aws::String& GetViewId() const { return m_viewId; }
  inline bool ViewIdHasBeenSet() const { return m_viewIdHasBeenSet; }
  template <typename ViewIdT = Aws::String>
  void SetViewId(ViewIdT&& value) {
    m_viewIdHasBeenSet = true;
    m_viewId = std::forward<ViewIdT>(value);
  }
  template <typename ViewIdT = Aws::String>
  UpdateViewContentRequest& WithViewId(ViewIdT&& value) {
    SetViewId(std::forward<ViewIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the view status as either <code>SAVED</code> or
   * <code>PUBLISHED</code>. The <code>PUBLISHED</code> status will initiate
   * validation on the content.</p>
   */
  inline ViewStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(ViewStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline UpdateViewContentRequest& WithStatus(ViewStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>View content containing all content necessary to render a view except for
   * runtime input data and the runtime input schema, which is auto-generated by this
   * operation.</p> <p>The total uncompressed content has a maximum file size of
   * 400kB.</p>
   */
  inline const ViewInputContent& GetContent() const { return m_content; }
  inline bool ContentHasBeenSet() const { return m_contentHasBeenSet; }
  template <typename ContentT = ViewInputContent>
  void SetContent(ContentT&& value) {
    m_contentHasBeenSet = true;
    m_content = std::forward<ContentT>(value);
  }
  template <typename ContentT = ViewInputContent>
  UpdateViewContentRequest& WithContent(ContentT&& value) {
    SetContent(std::forward<ContentT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;

  Aws::String m_viewId;

  ViewStatus m_status{ViewStatus::NOT_SET};

  ViewInputContent m_content;
  bool m_instanceIdHasBeenSet = false;
  bool m_viewIdHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_contentHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
