﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/comprehend/Comprehend_EXPORTS.h>
#include <aws/comprehend/model/ClassifierMetadata.h>
#include <aws/comprehend/model/DocumentClassifierInputDataConfig.h>
#include <aws/comprehend/model/DocumentClassifierMode.h>
#include <aws/comprehend/model/DocumentClassifierOutputDataConfig.h>
#include <aws/comprehend/model/LanguageCode.h>
#include <aws/comprehend/model/ModelStatus.h>
#include <aws/comprehend/model/VpcConfig.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Comprehend {
namespace Model {

/**
 * <p>Provides information about a document classifier.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/comprehend-2017-11-27/DocumentClassifierProperties">AWS
 * API Reference</a></p>
 */
class DocumentClassifierProperties {
 public:
  AWS_COMPREHEND_API DocumentClassifierProperties() = default;
  AWS_COMPREHEND_API DocumentClassifierProperties(Aws::Utils::Json::JsonView jsonValue);
  AWS_COMPREHEND_API DocumentClassifierProperties& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_COMPREHEND_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that identifies the document classifier.</p>
   */
  inline const Aws::String& GetDocumentClassifierArn() const { return m_documentClassifierArn; }
  inline bool DocumentClassifierArnHasBeenSet() const { return m_documentClassifierArnHasBeenSet; }
  template <typename DocumentClassifierArnT = Aws::String>
  void SetDocumentClassifierArn(DocumentClassifierArnT&& value) {
    m_documentClassifierArnHasBeenSet = true;
    m_documentClassifierArn = std::forward<DocumentClassifierArnT>(value);
  }
  template <typename DocumentClassifierArnT = Aws::String>
  DocumentClassifierProperties& WithDocumentClassifierArn(DocumentClassifierArnT&& value) {
    SetDocumentClassifierArn(std::forward<DocumentClassifierArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The language code for the language of the documents that the classifier was
   * trained on.</p>
   */
  inline LanguageCode GetLanguageCode() const { return m_languageCode; }
  inline bool LanguageCodeHasBeenSet() const { return m_languageCodeHasBeenSet; }
  inline void SetLanguageCode(LanguageCode value) {
    m_languageCodeHasBeenSet = true;
    m_languageCode = value;
  }
  inline DocumentClassifierProperties& WithLanguageCode(LanguageCode value) {
    SetLanguageCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the document classifier. If the status is <code>TRAINED</code>
   * the classifier is ready to use. If the status is
   * <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you
   * should review the warnings returned in the <code>CreateDocumentClassifier</code>
   * response.</p> <p> If the status is <code>FAILED</code> you can see additional
   * information about why the classifier wasn't trained in the <code>Message</code>
   * field.</p>
   */
  inline ModelStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(ModelStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline DocumentClassifierProperties& WithStatus(ModelStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Additional information about the status of the classifier.</p>
   */
  inline const Aws::String& GetMessage() const { return m_message; }
  inline bool MessageHasBeenSet() const { return m_messageHasBeenSet; }
  template <typename MessageT = Aws::String>
  void SetMessage(MessageT&& value) {
    m_messageHasBeenSet = true;
    m_message = std::forward<MessageT>(value);
  }
  template <typename MessageT = Aws::String>
  DocumentClassifierProperties& WithMessage(MessageT&& value) {
    SetMessage(std::forward<MessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time that the document classifier was submitted for training.</p>
   */
  inline const Aws::Utils::DateTime& GetSubmitTime() const { return m_submitTime; }
  inline bool SubmitTimeHasBeenSet() const { return m_submitTimeHasBeenSet; }
  template <typename SubmitTimeT = Aws::Utils::DateTime>
  void SetSubmitTime(SubmitTimeT&& value) {
    m_submitTimeHasBeenSet = true;
    m_submitTime = std::forward<SubmitTimeT>(value);
  }
  template <typename SubmitTimeT = Aws::Utils::DateTime>
  DocumentClassifierProperties& WithSubmitTime(SubmitTimeT&& value) {
    SetSubmitTime(std::forward<SubmitTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time that training the document classifier completed.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  DocumentClassifierProperties& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the time when the training starts on documentation classifiers. You
   * are billed for the time interval between this time and the value of
   * TrainingEndTime. </p>
   */
  inline const Aws::Utils::DateTime& GetTrainingStartTime() const { return m_trainingStartTime; }
  inline bool TrainingStartTimeHasBeenSet() const { return m_trainingStartTimeHasBeenSet; }
  template <typename TrainingStartTimeT = Aws::Utils::DateTime>
  void SetTrainingStartTime(TrainingStartTimeT&& value) {
    m_trainingStartTimeHasBeenSet = true;
    m_trainingStartTime = std::forward<TrainingStartTimeT>(value);
  }
  template <typename TrainingStartTimeT = Aws::Utils::DateTime>
  DocumentClassifierProperties& WithTrainingStartTime(TrainingStartTimeT&& value) {
    SetTrainingStartTime(std::forward<TrainingStartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time that training of the document classifier was completed. Indicates
   * the time when the training completes on documentation classifiers. You are
   * billed for the time interval between this time and the value of
   * TrainingStartTime.</p>
   */
  inline const Aws::Utils::DateTime& GetTrainingEndTime() const { return m_trainingEndTime; }
  inline bool TrainingEndTimeHasBeenSet() const { return m_trainingEndTimeHasBeenSet; }
  template <typename TrainingEndTimeT = Aws::Utils::DateTime>
  void SetTrainingEndTime(TrainingEndTimeT&& value) {
    m_trainingEndTimeHasBeenSet = true;
    m_trainingEndTime = std::forward<TrainingEndTimeT>(value);
  }
  template <typename TrainingEndTimeT = Aws::Utils::DateTime>
  DocumentClassifierProperties& WithTrainingEndTime(TrainingEndTimeT&& value) {
    SetTrainingEndTime(std::forward<TrainingEndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The input data configuration that you supplied when you created the document
   * classifier for training.</p>
   */
  inline const DocumentClassifierInputDataConfig& GetInputDataConfig() const { return m_inputDataConfig; }
  inline bool InputDataConfigHasBeenSet() const { return m_inputDataConfigHasBeenSet; }
  template <typename InputDataConfigT = DocumentClassifierInputDataConfig>
  void SetInputDataConfig(InputDataConfigT&& value) {
    m_inputDataConfigHasBeenSet = true;
    m_inputDataConfig = std::forward<InputDataConfigT>(value);
  }
  template <typename InputDataConfigT = DocumentClassifierInputDataConfig>
  DocumentClassifierProperties& WithInputDataConfig(InputDataConfigT&& value) {
    SetInputDataConfig(std::forward<InputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Provides output results configuration parameters for custom classifier
   * jobs.</p>
   */
  inline const DocumentClassifierOutputDataConfig& GetOutputDataConfig() const { return m_outputDataConfig; }
  inline bool OutputDataConfigHasBeenSet() const { return m_outputDataConfigHasBeenSet; }
  template <typename OutputDataConfigT = DocumentClassifierOutputDataConfig>
  void SetOutputDataConfig(OutputDataConfigT&& value) {
    m_outputDataConfigHasBeenSet = true;
    m_outputDataConfig = std::forward<OutputDataConfigT>(value);
  }
  template <typename OutputDataConfigT = DocumentClassifierOutputDataConfig>
  DocumentClassifierProperties& WithOutputDataConfig(OutputDataConfigT&& value) {
    SetOutputDataConfig(std::forward<OutputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the document classifier, including the number of documents
   * used for training the classifier, the number of documents used for test the
   * classifier, and an accuracy rating.</p>
   */
  inline const ClassifierMetadata& GetClassifierMetadata() const { return m_classifierMetadata; }
  inline bool ClassifierMetadataHasBeenSet() const { return m_classifierMetadataHasBeenSet; }
  template <typename ClassifierMetadataT = ClassifierMetadata>
  void SetClassifierMetadata(ClassifierMetadataT&& value) {
    m_classifierMetadataHasBeenSet = true;
    m_classifierMetadata = std::forward<ClassifierMetadataT>(value);
  }
  template <typename ClassifierMetadataT = ClassifierMetadata>
  DocumentClassifierProperties& WithClassifierMetadata(ClassifierMetadataT&& value) {
    SetClassifierMetadata(std::forward<ClassifierMetadataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role that grants Amazon Comprehend
   * read access to your input data.</p>
   */
  inline const Aws::String& GetDataAccessRoleArn() const { return m_dataAccessRoleArn; }
  inline bool DataAccessRoleArnHasBeenSet() const { return m_dataAccessRoleArnHasBeenSet; }
  template <typename DataAccessRoleArnT = Aws::String>
  void SetDataAccessRoleArn(DataAccessRoleArnT&& value) {
    m_dataAccessRoleArnHasBeenSet = true;
    m_dataAccessRoleArn = std::forward<DataAccessRoleArnT>(value);
  }
  template <typename DataAccessRoleArnT = Aws::String>
  DocumentClassifierProperties& WithDataAccessRoleArn(DataAccessRoleArnT&& value) {
    SetDataAccessRoleArn(std::forward<DataAccessRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ID for the Amazon Web Services Key Management Service (KMS) key that Amazon
   * Comprehend uses to encrypt data on the storage volume attached to the ML compute
   * instance(s) that process the analysis job. The VolumeKmsKeyId can be either of
   * the following formats:</p> <ul> <li> <p>KMS Key ID:
   * <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code> </p> </li> <li> <p>Amazon
   * Resource Name (ARN) of a KMS Key:
   * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
   * </p> </li> </ul>
   */
  inline const Aws::String& GetVolumeKmsKeyId() const { return m_volumeKmsKeyId; }
  inline bool VolumeKmsKeyIdHasBeenSet() const { return m_volumeKmsKeyIdHasBeenSet; }
  template <typename VolumeKmsKeyIdT = Aws::String>
  void SetVolumeKmsKeyId(VolumeKmsKeyIdT&& value) {
    m_volumeKmsKeyIdHasBeenSet = true;
    m_volumeKmsKeyId = std::forward<VolumeKmsKeyIdT>(value);
  }
  template <typename VolumeKmsKeyIdT = Aws::String>
  DocumentClassifierProperties& WithVolumeKmsKeyId(VolumeKmsKeyIdT&& value) {
    SetVolumeKmsKeyId(std::forward<VolumeKmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Configuration parameters for a private Virtual Private Cloud (VPC)
   * containing the resources you are using for your custom classifier. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/vpc/latest/userguide/what-is-amazon-vpc.html">Amazon
   * VPC</a>. </p>
   */
  inline const VpcConfig& GetVpcConfig() const { return m_vpcConfig; }
  inline bool VpcConfigHasBeenSet() const { return m_vpcConfigHasBeenSet; }
  template <typename VpcConfigT = VpcConfig>
  void SetVpcConfig(VpcConfigT&& value) {
    m_vpcConfigHasBeenSet = true;
    m_vpcConfig = std::forward<VpcConfigT>(value);
  }
  template <typename VpcConfigT = VpcConfig>
  DocumentClassifierProperties& WithVpcConfig(VpcConfigT&& value) {
    SetVpcConfig(std::forward<VpcConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the mode in which the specific classifier was trained. This also
   * indicates the format of input documents and the format of the confusion matrix.
   * Each classifier can only be trained in one mode and this cannot be changed once
   * the classifier is trained.</p>
   */
  inline DocumentClassifierMode GetMode() const { return m_mode; }
  inline bool ModeHasBeenSet() const { return m_modeHasBeenSet; }
  inline void SetMode(DocumentClassifierMode value) {
    m_modeHasBeenSet = true;
    m_mode = value;
  }
  inline DocumentClassifierProperties& WithMode(DocumentClassifierMode value) {
    SetMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ID for the KMS key that Amazon Comprehend uses to encrypt trained custom
   * models. The ModelKmsKeyId can be either of the following formats:</p> <ul> <li>
   * <p>KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code> </p> </li>
   * <li> <p>Amazon Resource Name (ARN) of a KMS Key:
   * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
   * </p> </li> </ul>
   */
  inline const Aws::String& GetModelKmsKeyId() const { return m_modelKmsKeyId; }
  inline bool ModelKmsKeyIdHasBeenSet() const { return m_modelKmsKeyIdHasBeenSet; }
  template <typename ModelKmsKeyIdT = Aws::String>
  void SetModelKmsKeyId(ModelKmsKeyIdT&& value) {
    m_modelKmsKeyIdHasBeenSet = true;
    m_modelKmsKeyId = std::forward<ModelKmsKeyIdT>(value);
  }
  template <typename ModelKmsKeyIdT = Aws::String>
  DocumentClassifierProperties& WithModelKmsKeyId(ModelKmsKeyIdT&& value) {
    SetModelKmsKeyId(std::forward<ModelKmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version name that you assigned to the document classifier.</p>
   */
  inline const Aws::String& GetVersionName() const { return m_versionName; }
  inline bool VersionNameHasBeenSet() const { return m_versionNameHasBeenSet; }
  template <typename VersionNameT = Aws::String>
  void SetVersionName(VersionNameT&& value) {
    m_versionNameHasBeenSet = true;
    m_versionName = std::forward<VersionNameT>(value);
  }
  template <typename VersionNameT = Aws::String>
  DocumentClassifierProperties& WithVersionName(VersionNameT&& value) {
    SetVersionName(std::forward<VersionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the source model. This model was imported
   * from a different Amazon Web Services account to create the document classifier
   * model in your Amazon Web Services account.</p>
   */
  inline const Aws::String& GetSourceModelArn() const { return m_sourceModelArn; }
  inline bool SourceModelArnHasBeenSet() const { return m_sourceModelArnHasBeenSet; }
  template <typename SourceModelArnT = Aws::String>
  void SetSourceModelArn(SourceModelArnT&& value) {
    m_sourceModelArnHasBeenSet = true;
    m_sourceModelArn = std::forward<SourceModelArnT>(value);
  }
  template <typename SourceModelArnT = Aws::String>
  DocumentClassifierProperties& WithSourceModelArn(SourceModelArnT&& value) {
    SetSourceModelArn(std::forward<SourceModelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Number (ARN) of the flywheel</p>
   */
  inline const Aws::String& GetFlywheelArn() const { return m_flywheelArn; }
  inline bool FlywheelArnHasBeenSet() const { return m_flywheelArnHasBeenSet; }
  template <typename FlywheelArnT = Aws::String>
  void SetFlywheelArn(FlywheelArnT&& value) {
    m_flywheelArnHasBeenSet = true;
    m_flywheelArn = std::forward<FlywheelArnT>(value);
  }
  template <typename FlywheelArnT = Aws::String>
  DocumentClassifierProperties& WithFlywheelArn(FlywheelArnT&& value) {
    SetFlywheelArn(std::forward<FlywheelArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_documentClassifierArn;

  LanguageCode m_languageCode{LanguageCode::NOT_SET};

  ModelStatus m_status{ModelStatus::NOT_SET};

  Aws::String m_message;

  Aws::Utils::DateTime m_submitTime{};

  Aws::Utils::DateTime m_endTime{};

  Aws::Utils::DateTime m_trainingStartTime{};

  Aws::Utils::DateTime m_trainingEndTime{};

  DocumentClassifierInputDataConfig m_inputDataConfig;

  DocumentClassifierOutputDataConfig m_outputDataConfig;

  ClassifierMetadata m_classifierMetadata;

  Aws::String m_dataAccessRoleArn;

  Aws::String m_volumeKmsKeyId;

  VpcConfig m_vpcConfig;

  DocumentClassifierMode m_mode{DocumentClassifierMode::NOT_SET};

  Aws::String m_modelKmsKeyId;

  Aws::String m_versionName;

  Aws::String m_sourceModelArn;

  Aws::String m_flywheelArn;
  bool m_documentClassifierArnHasBeenSet = false;
  bool m_languageCodeHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_messageHasBeenSet = false;
  bool m_submitTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_trainingStartTimeHasBeenSet = false;
  bool m_trainingEndTimeHasBeenSet = false;
  bool m_inputDataConfigHasBeenSet = false;
  bool m_outputDataConfigHasBeenSet = false;
  bool m_classifierMetadataHasBeenSet = false;
  bool m_dataAccessRoleArnHasBeenSet = false;
  bool m_volumeKmsKeyIdHasBeenSet = false;
  bool m_vpcConfigHasBeenSet = false;
  bool m_modeHasBeenSet = false;
  bool m_modelKmsKeyIdHasBeenSet = false;
  bool m_versionNameHasBeenSet = false;
  bool m_sourceModelArnHasBeenSet = false;
  bool m_flywheelArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace Comprehend
}  // namespace Aws
