﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/apigateway/model/CacheClusterSize.h>
#include <aws/apigateway/model/DeploymentCanarySettings.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace APIGateway {
namespace Model {

/**
 * <p>Requests API Gateway to create a Deployment resource.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/CreateDeploymentRequest">AWS
 * API Reference</a></p>
 */
class CreateDeploymentRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API CreateDeploymentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDeployment"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The string identifier of the associated RestApi.</p>
   */
  inline const Aws::String& GetRestApiId() const { return m_restApiId; }
  inline bool RestApiIdHasBeenSet() const { return m_restApiIdHasBeenSet; }
  template <typename RestApiIdT = Aws::String>
  void SetRestApiId(RestApiIdT&& value) {
    m_restApiIdHasBeenSet = true;
    m_restApiId = std::forward<RestApiIdT>(value);
  }
  template <typename RestApiIdT = Aws::String>
  CreateDeploymentRequest& WithRestApiId(RestApiIdT&& value) {
    SetRestApiId(std::forward<RestApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the Stage resource for the Deployment resource to create.</p>
   */
  inline const Aws::String& GetStageName() const { return m_stageName; }
  inline bool StageNameHasBeenSet() const { return m_stageNameHasBeenSet; }
  template <typename StageNameT = Aws::String>
  void SetStageName(StageNameT&& value) {
    m_stageNameHasBeenSet = true;
    m_stageName = std::forward<StageNameT>(value);
  }
  template <typename StageNameT = Aws::String>
  CreateDeploymentRequest& WithStageName(StageNameT&& value) {
    SetStageName(std::forward<StageNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the Stage resource for the Deployment resource to
   * create.</p>
   */
  inline const Aws::String& GetStageDescription() const { return m_stageDescription; }
  inline bool StageDescriptionHasBeenSet() const { return m_stageDescriptionHasBeenSet; }
  template <typename StageDescriptionT = Aws::String>
  void SetStageDescription(StageDescriptionT&& value) {
    m_stageDescriptionHasBeenSet = true;
    m_stageDescription = std::forward<StageDescriptionT>(value);
  }
  template <typename StageDescriptionT = Aws::String>
  CreateDeploymentRequest& WithStageDescription(StageDescriptionT&& value) {
    SetStageDescription(std::forward<StageDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description for the Deployment resource to create.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateDeploymentRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Enables a cache cluster for the Stage resource specified in the input.</p>
   */
  inline bool GetCacheClusterEnabled() const { return m_cacheClusterEnabled; }
  inline bool CacheClusterEnabledHasBeenSet() const { return m_cacheClusterEnabledHasBeenSet; }
  inline void SetCacheClusterEnabled(bool value) {
    m_cacheClusterEnabledHasBeenSet = true;
    m_cacheClusterEnabled = value;
  }
  inline CreateDeploymentRequest& WithCacheClusterEnabled(bool value) {
    SetCacheClusterEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The stage's cache capacity in GB. For more information about choosing a cache
   * size, see <a
   * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html">Enabling
   * API caching to enhance responsiveness</a>.</p>
   */
  inline CacheClusterSize GetCacheClusterSize() const { return m_cacheClusterSize; }
  inline bool CacheClusterSizeHasBeenSet() const { return m_cacheClusterSizeHasBeenSet; }
  inline void SetCacheClusterSize(CacheClusterSize value) {
    m_cacheClusterSizeHasBeenSet = true;
    m_cacheClusterSize = value;
  }
  inline CreateDeploymentRequest& WithCacheClusterSize(CacheClusterSize value) {
    SetCacheClusterSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map that defines the stage variables for the Stage resource that is
   * associated with the new deployment. Variable names can have alphanumeric and
   * underscore characters, and the values must match
   * <code>[A-Za-z0-9-._~:/?#&amp;=,]+</code>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetVariables() const { return m_variables; }
  inline bool VariablesHasBeenSet() const { return m_variablesHasBeenSet; }
  template <typename VariablesT = Aws::Map<Aws::String, Aws::String>>
  void SetVariables(VariablesT&& value) {
    m_variablesHasBeenSet = true;
    m_variables = std::forward<VariablesT>(value);
  }
  template <typename VariablesT = Aws::Map<Aws::String, Aws::String>>
  CreateDeploymentRequest& WithVariables(VariablesT&& value) {
    SetVariables(std::forward<VariablesT>(value));
    return *this;
  }
  template <typename VariablesKeyT = Aws::String, typename VariablesValueT = Aws::String>
  CreateDeploymentRequest& AddVariables(VariablesKeyT&& key, VariablesValueT&& value) {
    m_variablesHasBeenSet = true;
    m_variables.emplace(std::forward<VariablesKeyT>(key), std::forward<VariablesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The input configuration for the canary deployment when the deployment is a
   * canary release deployment. </p>
   */
  inline const DeploymentCanarySettings& GetCanarySettings() const { return m_canarySettings; }
  inline bool CanarySettingsHasBeenSet() const { return m_canarySettingsHasBeenSet; }
  template <typename CanarySettingsT = DeploymentCanarySettings>
  void SetCanarySettings(CanarySettingsT&& value) {
    m_canarySettingsHasBeenSet = true;
    m_canarySettings = std::forward<CanarySettingsT>(value);
  }
  template <typename CanarySettingsT = DeploymentCanarySettings>
  CreateDeploymentRequest& WithCanarySettings(CanarySettingsT&& value) {
    SetCanarySettings(std::forward<CanarySettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether active tracing with X-ray is enabled for the Stage.</p>
   */
  inline bool GetTracingEnabled() const { return m_tracingEnabled; }
  inline bool TracingEnabledHasBeenSet() const { return m_tracingEnabledHasBeenSet; }
  inline void SetTracingEnabled(bool value) {
    m_tracingEnabledHasBeenSet = true;
    m_tracingEnabled = value;
  }
  inline CreateDeploymentRequest& WithTracingEnabled(bool value) {
    SetTracingEnabled(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_restApiId;

  Aws::String m_stageName;

  Aws::String m_stageDescription;

  Aws::String m_description;

  bool m_cacheClusterEnabled{false};

  CacheClusterSize m_cacheClusterSize{CacheClusterSize::NOT_SET};

  Aws::Map<Aws::String, Aws::String> m_variables;

  DeploymentCanarySettings m_canarySettings;

  bool m_tracingEnabled{false};
  bool m_restApiIdHasBeenSet = false;
  bool m_stageNameHasBeenSet = false;
  bool m_stageDescriptionHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_cacheClusterEnabledHasBeenSet = false;
  bool m_cacheClusterSizeHasBeenSet = false;
  bool m_variablesHasBeenSet = false;
  bool m_canarySettingsHasBeenSet = false;
  bool m_tracingEnabledHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
