/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.rest.basic.auth.extension;

import jakarta.ws.rs.container.ContainerRequestContext;
import jakarta.ws.rs.container.ContainerRequestFilter;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.SecurityContext;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import javax.annotation.Priority;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.rest.basic.auth.extension.BasicAuthSecurityRestExtension;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Priority(value=1000)
public class JaasBasicAuthFilter
implements ContainerRequestFilter {
    private static final Logger log = LoggerFactory.getLogger(JaasBasicAuthFilter.class);
    private static final Set<RequestMatcher> INTERNAL_REQUEST_MATCHERS = Set.of(new RequestMatcher("POST", "/?connectors/([^/]+)/tasks/?"), new RequestMatcher("PUT", "/?connectors/[^/]+/fence/?"));
    private static final String CONNECT_LOGIN_MODULE = "KafkaConnect";
    static final String AUTHORIZATION = "Authorization";
    final Configuration configuration;

    public JaasBasicAuthFilter(Configuration configuration) {
        this.configuration = configuration;
    }

    public void filter(ContainerRequestContext requestContext) {
        if (this.isInternalRequest(requestContext)) {
            log.trace("Skipping authentication for internal request");
            return;
        }
        try {
            log.debug("Authenticating request");
            BasicAuthCredentials credentials = new BasicAuthCredentials(requestContext.getHeaderString(AUTHORIZATION));
            LoginContext loginContext = new LoginContext(CONNECT_LOGIN_MODULE, null, new BasicAuthCallBackHandler(credentials), this.configuration);
            loginContext.login();
            this.setSecurityContextForRequest(requestContext, credentials);
        }
        catch (LoginException | ConfigException e) {
            log.debug("Request failed authentication", e);
            requestContext.abortWith(Response.status((Response.Status)Response.Status.UNAUTHORIZED).entity((Object)"User cannot access the resource.").build());
        }
    }

    private boolean isInternalRequest(ContainerRequestContext requestContext) {
        return INTERNAL_REQUEST_MATCHERS.stream().anyMatch(m -> m.test(requestContext));
    }

    private void setSecurityContextForRequest(final ContainerRequestContext requestContext, final BasicAuthCredentials credential) {
        requestContext.setSecurityContext(new SecurityContext(){

            public Principal getUserPrincipal() {
                return credential::username;
            }

            public boolean isUserInRole(String role) {
                return false;
            }

            public boolean isSecure() {
                return "https".equalsIgnoreCase(requestContext.getUriInfo().getRequestUri().getScheme());
            }

            public String getAuthenticationScheme() {
                return "BASIC";
            }
        });
    }

    static class BasicAuthCredentials {
        private String username;
        private String password;

        public BasicAuthCredentials(String authorizationHeader) {
            if (authorizationHeader == null) {
                log.trace("No credentials were provided with the request");
                return;
            }
            int spaceIndex = authorizationHeader.indexOf(32);
            if (spaceIndex <= 0) {
                log.trace("Request credentials were malformed; no space present in value for authorization header");
                return;
            }
            String method = authorizationHeader.substring(0, spaceIndex);
            if (!"BASIC".equalsIgnoreCase(method)) {
                log.trace("Request credentials used {} authentication, but only {} supported; ignoring", (Object)method, (Object)"BASIC");
                return;
            }
            authorizationHeader = authorizationHeader.substring(spaceIndex + 1);
            authorizationHeader = new String(Base64.getDecoder().decode(authorizationHeader), StandardCharsets.UTF_8);
            int i = authorizationHeader.indexOf(58);
            if (i <= 0) {
                log.trace("Request credentials were malformed; no colon present between username and password");
                return;
            }
            this.username = authorizationHeader.substring(0, i);
            this.password = authorizationHeader.substring(i + 1);
        }

        public String username() {
            return this.username;
        }

        public String password() {
            return this.password;
        }
    }

    public static class BasicAuthCallBackHandler
    implements CallbackHandler {
        private final String username;
        private final String password;

        public BasicAuthCallBackHandler(BasicAuthCredentials credentials) {
            this.username = credentials.username();
            this.password = credentials.password();
        }

        @Override
        public void handle(Callback[] callbacks) {
            ArrayList<Callback> unsupportedCallbacks = new ArrayList<Callback>();
            for (Callback callback : callbacks) {
                if (callback instanceof NameCallback) {
                    ((NameCallback)callback).setName(this.username);
                    continue;
                }
                if (callback instanceof PasswordCallback) {
                    ((PasswordCallback)callback).setPassword(this.password != null ? this.password.toCharArray() : null);
                    continue;
                }
                unsupportedCallbacks.add(callback);
            }
            if (!unsupportedCallbacks.isEmpty()) {
                throw new ConnectException(String.format("Unsupported callbacks %s; request authentication will fail. This indicates the Connect worker was configured with a JAAS LoginModule that is incompatible with the %s, and will need to be corrected and restarted.", unsupportedCallbacks, BasicAuthSecurityRestExtension.class.getSimpleName()));
            }
        }
    }

    private static class RequestMatcher
    implements Predicate<ContainerRequestContext> {
        private final String method;
        private final Pattern path;

        public RequestMatcher(String method, String path) {
            this.method = method;
            this.path = Pattern.compile(path);
        }

        @Override
        public boolean test(ContainerRequestContext requestContext) {
            return requestContext.getMethod().equals(this.method) && this.path.matcher(requestContext.getUriInfo().getPath()).matches();
        }
    }
}

