/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.server.kerberos.shared.crypto.encryption;


import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.security.auth.kerberos.KerberosKey;
import javax.security.auth.kerberos.KerberosPrincipal;

import org.apache.directory.shared.kerberos.codec.types.EncryptionType;
import org.apache.directory.shared.kerberos.components.EncryptionKey;
import org.junit.jupiter.api.Test;


/**
 * Test cases for string-to-key functions for DES-, DES3-, AES-, and RC4-based
 * encryption types.
 * 
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class KerberosKeyFactoryTest
{
    /**
     * Tests that key derivation can be performed for a DES key.
     */
    @Test
    public void testDesKerberosKey()
    {
        KerberosPrincipal principal = new KerberosPrincipal( "hnelson@EXAMPLE.COM" );
        KerberosKey key = new KerberosKey( principal, "secret".toCharArray(), "DES" );

        assertEquals( 8, key.getEncoded().length, "DES key length" );
    }


    /**
     * Tests that key derivation can be performed for a Triple-DES key.
     */
    @Test
    public void testTripleDesKerberosKey()
    {
        KerberosPrincipal principal = new KerberosPrincipal( "hnelson@EXAMPLE.COM" );
        String algorithm = VendorHelper.getTripleDesAlgorithm();
        KerberosKey key = new KerberosKey( principal, "secret".toCharArray(), algorithm );

        assertEquals( 24, key.getEncoded().length, "DESede key length" );
    }


    /**
     * Tests that key derivation can be performed for an RC4-HMAC key.
     */
    @Test
    public void testArcFourHmacKerberosKey()
    {
        if ( !VendorHelper.isArcFourHmacSupported() )
        {
            return;
        }

        KerberosPrincipal principal = new KerberosPrincipal( "hnelson@EXAMPLE.COM" );
        KerberosKey key = new KerberosKey( principal, "secret".toCharArray(), "ArcFourHmac" );

        assertEquals( 16, key.getEncoded().length, "ArcFourHmac key length" );
    }


    /**
     * Tests that key derivation can be performed for an AES-128 key.
     *
     * @throws Exception
     */
    @Test
    public void testAes128KerberosKey() throws Exception
    {
        KerberosPrincipal principal = new KerberosPrincipal( "hnelson@EXAMPLE.COM" );
        KerberosKey key = new KerberosKey( principal, "secret".toCharArray(), "AES128" );

        assertEquals( 16, key.getEncoded().length, "AES128 key length" );
    }


    /**
     * Tests that key derivation can be performed for an AES-256 key.
     */
    @Test
    public void testAes256KerberosKey()
    {
        try
        {
            KerberosPrincipal principal = new KerberosPrincipal( "hnelson@EXAMPLE.COM" );
            KerberosKey kerberosKey = new KerberosKey( principal, "secret".toCharArray(), "AES256" );
            assertEquals( 32, kerberosKey.getEncoded().length, "AES256 key length" );
        }
        catch ( IllegalArgumentException iae )
        {
            // Algorithm AES256 not enabled
        }
    }


    /**
     * Tests that key derivation can be performed by the factory for the des-cbc-md5 encryption type.
     */
    @Test
    public void testKerberosKeyFactoryOnlyDes()
    {
        String principalName = "hnelson@EXAMPLE.COM";
        String passPhrase = "secret";

        Set<EncryptionType> encryptionTypes = new HashSet<EncryptionType>();
        encryptionTypes.add( EncryptionType.DES_CBC_MD5 );

        Map<EncryptionType, EncryptionKey> map = KerberosKeyFactory.getKerberosKeys( principalName, passPhrase,
            encryptionTypes );

        assertEquals( 1, map.values().size(), "List length" );

        EncryptionKey kerberosKey = map.get( EncryptionType.DES_CBC_MD5 );

        EncryptionType keyType = kerberosKey.getKeyType();
        int keyLength = kerberosKey.getKeyValue().length;
        byte[] keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.DES_CBC_MD5, keyType );
        assertEquals( 8, keyLength );
        byte[] expectedBytes = new byte[]
            { ( byte ) 0xF4, ( byte ) 0xA7, ( byte ) 0x13, ( byte ) 0x64, ( byte ) 0x8A, ( byte ) 0x61, ( byte ) 0xCE,
                ( byte ) 0x5B };
        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );
    }


    /**
     * Tests that key derivation can be performed by the factory for the des3-cbc-sha1-kd encryption type.
     */
    @Test
    public void testKerberosKeyFactoryOnlyTripleDes()
    {
        if ( !VendorHelper.isTripleDesSupported() )
        {
            return;
        }

        String principalName = "hnelson@EXAMPLE.COM";
        String passPhrase = "secret";

        Set<EncryptionType> encryptionTypes = new HashSet<EncryptionType>();
        encryptionTypes.add( EncryptionType.DES3_CBC_SHA1_KD );

        Map<EncryptionType, EncryptionKey> map = KerberosKeyFactory.getKerberosKeys( principalName, passPhrase,
            encryptionTypes );

        assertEquals( 1, map.values().size(), "List length" );

        EncryptionKey kerberosKey = map.get( EncryptionType.DES3_CBC_SHA1_KD );

        EncryptionType keyType = kerberosKey.getKeyType();
        int keyLength = kerberosKey.getKeyValue().length;
        byte[] keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.DES3_CBC_SHA1_KD, keyType );
        assertEquals( 24, keyLength );
        byte[] expectedBytes = new byte[]
            { ( byte ) 0x57, ( byte ) 0x07, ( byte ) 0xCE, ( byte ) 0x29, ( byte ) 0x52, ( byte ) 0x92, ( byte ) 0x2C,
                ( byte ) 0x1C, ( byte ) 0x8C, ( byte ) 0xBF, ( byte ) 0x43, ( byte ) 0xC2, ( byte ) 0x3D,
                ( byte ) 0x8F, ( byte ) 0x8C, ( byte ) 0x5E, ( byte ) 0x9E, ( byte ) 0x8C, ( byte ) 0xF7,
                ( byte ) 0x5D, ( byte ) 0x3E, ( byte ) 0x4A, ( byte ) 0x5E, ( byte ) 0x25 };
        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );
    }


    /**
     * Tests that key derivation can be performed by the factory for the rc4-hmac encryption type.
     */
    @Test
    public void testKerberosKeyFactoryOnlyArcFourHmac()
    {
        if ( !VendorHelper.isArcFourHmacSupported() )
        {
            return;
        }

        String principalName = "hnelson@EXAMPLE.COM";
        String passPhrase = "secret";

        Set<EncryptionType> encryptionTypes = new HashSet<EncryptionType>();
        encryptionTypes.add( EncryptionType.RC4_HMAC );

        Map<EncryptionType, EncryptionKey> map = KerberosKeyFactory.getKerberosKeys( principalName, passPhrase,
            encryptionTypes );

        assertEquals( 1, map.values().size(), "List length" );

        EncryptionKey kerberosKey = map.get( EncryptionType.RC4_HMAC );

        EncryptionType keyType = kerberosKey.getKeyType();
        int keyLength = kerberosKey.getKeyValue().length;
        byte[] keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.RC4_HMAC, keyType );
        assertEquals( 16, keyLength );
        byte[] expectedBytes = new byte[]
            { ( byte ) 0x87, ( byte ) 0x8D, ( byte ) 0x80, ( byte ) 0x14, ( byte ) 0x60, ( byte ) 0x6C, ( byte ) 0xDA,
                ( byte ) 0x29, ( byte ) 0x67, ( byte ) 0x7A, ( byte ) 0x44, ( byte ) 0xEF, ( byte ) 0xA1,
                ( byte ) 0x35, ( byte ) 0x3F, ( byte ) 0xC7 };
        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );
    }


    /**
     * Tests that key derivation can be performed by the factory for the aes128-cts-hmac-sha1-96 encryption type.
     */
    @Test
    public void testKerberosKeyFactoryOnlyAes128()
    {
        if ( VendorHelper.isIbm() )
        {
            return;
        }

        String principalName = "hnelson@EXAMPLE.COM";
        String passPhrase = "secret";

        Set<EncryptionType> encryptionTypes = new HashSet<EncryptionType>();
        encryptionTypes.add( EncryptionType.AES128_CTS_HMAC_SHA1_96 );

        Map<EncryptionType, EncryptionKey> map = KerberosKeyFactory.getKerberosKeys( principalName, passPhrase,
            encryptionTypes );

        assertEquals( 1, map.values().size(), "List length" );

        EncryptionKey kerberosKey = map.get( EncryptionType.AES128_CTS_HMAC_SHA1_96 );

        EncryptionType keyType = kerberosKey.getKeyType();
        int keyLength = kerberosKey.getKeyValue().length;
        byte[] keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.AES128_CTS_HMAC_SHA1_96, keyType );
        assertEquals( 16, keyLength );
        byte[] expectedBytes = new byte[]
            { ( byte ) 0xAD, ( byte ) 0x21, ( byte ) 0x4B, ( byte ) 0x38, ( byte ) 0xB6, ( byte ) 0x9D, ( byte ) 0xFC,
                ( byte ) 0xCA, ( byte ) 0xAC, ( byte ) 0xF1, ( byte ) 0x5F, ( byte ) 0x34, ( byte ) 0x6D,
                ( byte ) 0x41, ( byte ) 0x7B, ( byte ) 0x90 };

        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );
    }


    /**
     * Tests that key derivation can be performed by the factory for the aes256-cts-hmac-sha1-96 encryption type.
     */
    @Test
    public void testKerberosKeyFactoryOnlyAes256()
    {
        if ( VendorHelper.isIbm() )
        {
            return;
        }

        String principalName = "hnelson@EXAMPLE.COM";
        String passPhrase = "secret";

        Set<EncryptionType> encryptionTypes = new HashSet<EncryptionType>();
        encryptionTypes.add( EncryptionType.AES256_CTS_HMAC_SHA1_96 );

        Map<EncryptionType, EncryptionKey> map = KerberosKeyFactory.getKerberosKeys( principalName, passPhrase,
            encryptionTypes );

        EncryptionKey kerberosKey = map.get( EncryptionType.AES256_CTS_HMAC_SHA1_96 );

        if ( kerberosKey != null )
        {
            assertEquals( 1, map.values().size(), "List length" );

            EncryptionType keyType = kerberosKey.getKeyType();
            int keyLength = kerberosKey.getKeyValue().length;
            byte[] keyBytes = kerberosKey.getKeyValue();

            assertEquals( EncryptionType.AES256_CTS_HMAC_SHA1_96, keyType );
            assertEquals( 32, keyLength );
            byte[] expectedBytes = new byte[]
                { ( byte ) 0x3D, ( byte ) 0x33, ( byte ) 0x31, ( byte ) 0x8F, ( byte ) 0xBE, ( byte ) 0x47,
                    ( byte ) 0xE5, ( byte ) 0x2A, ( byte ) 0x21, ( byte ) 0x50, ( byte ) 0x77, ( byte ) 0xA4,
                    ( byte ) 0x15, ( byte ) 0x58, ( byte ) 0xCA, ( byte ) 0xE7, ( byte ) 0x36, ( byte ) 0x50,
                    ( byte ) 0x1F, ( byte ) 0xA7, ( byte ) 0xA4, ( byte ) 0x85, ( byte ) 0x82, ( byte ) 0x05,
                    ( byte ) 0xF6, ( byte ) 0x8F, ( byte ) 0x67, ( byte ) 0xA2, ( byte ) 0xB5, ( byte ) 0xEA,
                    ( byte ) 0x0E, ( byte ) 0xBF };
            assertTrue( Arrays.equals( expectedBytes, keyBytes ) );
        }
    }


    /**
     * Tests that key derivation can be performed by the factory for multiple encryption types.
     */
    @Test
    public void testKerberosKeyFactory()
    {
        if ( VendorHelper.isIbm() )
        {
            return;
        }

        String principalName = "hnelson@EXAMPLE.COM";
        String passPhrase = "secret";

        Map<EncryptionType, EncryptionKey> map = KerberosKeyFactory.getKerberosKeys( principalName, passPhrase );

        EncryptionKey kerberosKey = map.get( EncryptionType.DES_CBC_MD5 );

        EncryptionType keyType = kerberosKey.getKeyType();
        int keyLength = kerberosKey.getKeyValue().length;
        byte[] keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.DES_CBC_MD5 , keyType);
        assertEquals( 8, keyLength );
        byte[] expectedBytes = new byte[]
            { ( byte ) 0xF4, ( byte ) 0xA7, ( byte ) 0x13, ( byte ) 0x64, ( byte ) 0x8A, ( byte ) 0x61, ( byte ) 0xCE,
                ( byte ) 0x5B };
        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );

        kerberosKey = map.get( EncryptionType.DES3_CBC_SHA1_KD );
        keyType = kerberosKey.getKeyType();
        keyLength = kerberosKey.getKeyValue().length;
        keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.DES3_CBC_SHA1_KD, keyType );
        assertEquals( 24, keyLength );
        expectedBytes = new byte[]
            { ( byte ) 0x57, ( byte ) 0x07, ( byte ) 0xCE, ( byte ) 0x29, ( byte ) 0x52, ( byte ) 0x92, ( byte ) 0x2C,
                ( byte ) 0x1C, ( byte ) 0x8C, ( byte ) 0xBF, ( byte ) 0x43, ( byte ) 0xC2, ( byte ) 0x3D,
                ( byte ) 0x8F, ( byte ) 0x8C, ( byte ) 0x5E, ( byte ) 0x9E, ( byte ) 0x8C, ( byte ) 0xF7,
                ( byte ) 0x5D, ( byte ) 0x3E, ( byte ) 0x4A, ( byte ) 0x5E, ( byte ) 0x25 };
        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );

        kerberosKey = map.get( EncryptionType.RC4_HMAC );
        keyType = kerberosKey.getKeyType();
        keyLength = kerberosKey.getKeyValue().length;
        keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.RC4_HMAC, keyType );
        assertEquals( 16, keyLength );
        expectedBytes = new byte[]
            { ( byte ) 0x87, ( byte ) 0x8D, ( byte ) 0x80, ( byte ) 0x14, ( byte ) 0x60, ( byte ) 0x6C, ( byte ) 0xDA,
                ( byte ) 0x29, ( byte ) 0x67, ( byte ) 0x7A, ( byte ) 0x44, ( byte ) 0xEF, ( byte ) 0xA1,
                ( byte ) 0x35, ( byte ) 0x3F, ( byte ) 0xC7 };
        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );

        kerberosKey = map.get( EncryptionType.AES128_CTS_HMAC_SHA1_96 );
        keyType = kerberosKey.getKeyType();
        keyLength = kerberosKey.getKeyValue().length;
        keyBytes = kerberosKey.getKeyValue();

        assertEquals( EncryptionType.AES128_CTS_HMAC_SHA1_96, keyType );
        assertEquals( 16, keyLength );
        expectedBytes = new byte[]
            { ( byte ) 0xAD, ( byte ) 0x21, ( byte ) 0x4B, ( byte ) 0x38, ( byte ) 0xB6, ( byte ) 0x9D, ( byte ) 0xFC,
                ( byte ) 0xCA, ( byte ) 0xAC, ( byte ) 0xF1, ( byte ) 0x5F, ( byte ) 0x34, ( byte ) 0x6D,
                ( byte ) 0x41, ( byte ) 0x7B, ( byte ) 0x90 };
        assertTrue( Arrays.equals( expectedBytes, keyBytes ) );

        kerberosKey = map.get( EncryptionType.AES256_CTS_HMAC_SHA1_96 );

        if ( kerberosKey != null )
        {
            keyType = kerberosKey.getKeyType();
            keyLength = kerberosKey.getKeyValue().length;
            keyBytes = kerberosKey.getKeyValue();

            assertEquals( EncryptionType.AES256_CTS_HMAC_SHA1_96, keyType );
            assertEquals( 32, keyLength );
            expectedBytes = new byte[]
                { ( byte ) 0x3D, ( byte ) 0x33, ( byte ) 0x31, ( byte ) 0x8F, ( byte ) 0xBE, ( byte ) 0x47,
                    ( byte ) 0xE5, ( byte ) 0x2A, ( byte ) 0x21, ( byte ) 0x50, ( byte ) 0x77, ( byte ) 0xA4,
                    ( byte ) 0x15, ( byte ) 0x58, ( byte ) 0xCA, ( byte ) 0xE7, ( byte ) 0x36, ( byte ) 0x50,
                    ( byte ) 0x1F, ( byte ) 0xA7, ( byte ) 0xA4, ( byte ) 0x85, ( byte ) 0x82, ( byte ) 0x05,
                    ( byte ) 0xF6, ( byte ) 0x8F, ( byte ) 0x67, ( byte ) 0xA2, ( byte ) 0xB5, ( byte ) 0xEA,
                    ( byte ) 0x0E, ( byte ) 0xBF };
            assertTrue( Arrays.equals( expectedBytes, keyBytes ) );
        }
    }
}
