package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ケンタッキーフライドチキンの住所を取得するクラスです。
 * @author Kumano Tatsuo
 * @since 3.16
 * 2005/12/04
 */
public class KFC {
	/**
	 * 最初のURL
	 */
	private static final String URL = "http://search.kfc.co.jp/store_search.cgi?gid=KFC&city=";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "EUC-JP";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "kfc_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * ケンタッキーフライドチキンの座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param city 市区町村コード
	 * @param prefecture 都道府県名
	 * @param isj 街区レベル位置参照情報
	 * @return 座標と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String city, final String prefecture,
			final Map<String, Point2D> isj) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(KFC.CACHE_DIR).exists()) {
			new File(KFC.CACHE_DIR).mkdirs();
		}
		final String cacheFile = KFC.CACHE_DIR + File.separator + KFC.PREFIX + city + KFC.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(KFC.getAddresses(city, prefecture), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
				cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "KFC");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * ケンタッキーフライドチキンの市区町村別ページを解析して住所の一覧を取得します。
	 * @param city 市区町村コード
	 * @param prefecture 都道府県名
	 * @return 住所と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String city, final String prefecture) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL + city);
			System.out.println("DEBUG: getting " + url);
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			final Pattern pattern = Pattern.compile("<td [^<>]+>([^<>]+)</td>");
			final Pattern pattern2 = Pattern.compile("<span class=\"sn\">([^<>]+)</span>");
			final Pattern pattern3 = Pattern.compile("<td class=\"TDL\">[^<>]+<br>([^<>]+)</td>");
			String caption = null;
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				if (line.replace("\t", "").equals("<tr>")) {
					if (scanner.hasNextLine()) {
						final String line2 = scanner.nextLine();
						if (line2.contains("住所")) {
							if (scanner.hasNextLine()) {
								final String line3 = scanner.nextLine();
								final Matcher matcher3 = pattern3.matcher(line3);
								if (matcher3.find()) {
									final String address = matcher3.group(1).trim();
									if (caption != null) {
										ret.put(address, caption);
										break;
									}
								}
							}
						} else {
							final Matcher matcher = pattern.matcher(line2);
							if (matcher.find()) {
								caption = matcher.group(1);
								if (scanner.hasNextLine()) {
									final String line3 = scanner.nextLine();
									final Matcher matcher2 = pattern.matcher(line3);
									if (matcher2.find()) {
										final String address = matcher2.group(1).trim();
										if (address.startsWith(prefecture)) {
											ret.put(address, caption);
										} else {
											ret.put(prefecture + address, caption);
										}
									}
								}
							}
						}
					}
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					caption = matcher2.group(1);
				}
			}
			scanner.close();
		} catch (FileNotFoundException e) {
		}
		return ret;
	}
}
