package bubble;
import java.awt.Point;
import java.util.Random;

/**
 * 動いている泡の情報をカプセル化するクラスです。
 * @author Kumano Tatsuo
 * 作成日：2004/10/16
 */
public class MovingBubble {
    /**
     * 泡の座標
     */
    private Point location;

    /**
     * 泡の色
     */
    private int color;

    /**
     * 泡のx方向の速度
     */
    private int speedX;

    /**
     * 泡のy方向の速度
     */
    private int speedY;

    /**
     * 目的地のグリッド位置
     */
    private Point destinationGrid;

    /**
     * @param location 座標
     * @param color 色
     */
    public MovingBubble(Point location, int color) {
        this.location = location;
        this.color = color;
        this.speedX = new Random().nextInt(5) - 2;
        this.speedY = Const.VERTICAL_DROP_INITIAL_SPEED + location.y * 2;
    }

    /**
     * 泡を下方向に移動させます。
     * 左右にも散ります。
     */
    public void moveDown() {
        this.location.x += this.speedX;
        this.location.y += (this.speedY += Const.VERTICAL_DROP_ACCEL) / 100;
    }

    /**
     * 目的地を設定します。
     * @param destination 目的地のグリッド座標
     */
    public void setDestination(Point destination) {
        this.destinationGrid = destination;
        this.speedY = new Random().nextInt(Const.CHAIN_REACTION_INITIAL_SPEED / 2)
                + Const.CHAIN_REACTION_INITIAL_SPEED / 2;
    }

    /**
     * 目的地を取得します。
     * @return 目的地
     */
    public Point getDestination() {
        return this.destinationGrid;
    }

    /**
     * 泡を目的地へ移動させます。
     * @param isJam おじゃまぷよかどうか
     */
    public void moveToDestination(boolean isJam) {
        Point p = Util.toLocation(this.destinationGrid.y, this.destinationGrid.x);
        int accel = isJam ? Const.JAM_REACTION_ACCEL : Const.CHAIN_REACTION_ACCEL * (this.destinationGrid.y + 1) * Const.CHAIN_REACTION_RATE;
        if (this.location.x > p.x) {
            this.speedX -= accel;
        } else {
            this.speedX += accel;
        }
        if (this.location.y > p.y) {
            this.speedY -= accel;
        } else {
            this.speedY += accel;
        }
        int decel = isJam ? Const.JAM_REACTION_DECEL : Const.CHAIN_REACTION_DECEL * (this.destinationGrid.y + 1) * Const.CHAIN_REACTION_RATE;
        this.speedX += (this.speedX > 0) ? -decel : decel;
        this.speedY += (this.speedY > 0) ? -decel : decel;
        this.location.x += this.speedX / 100;
        this.location.y += this.speedY / 100;
    }

    /**
     * 泡の座標を取得します。
     * @return 泡の座標
     */
    public Point getLocation() {
        return this.location;
    }

    /**
     * 泡の色を取得します。
     * @return 泡の色
     */
    public int getColor() {
        return this.color;
    }

    public String toString() {
        return "(" + this.location.x + ", " + this.location.y + ") " + this.color;
    }
}