package bubble;
import java.awt.Point;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Random;

/**
 * ゲームの情報を保持するクラスです。
 * @author Kumano Tatsuo
 * 作成日：2004/10/14
 */
public class Game {
    /** 動いている泡の色番号 */
    private int color;

    /** 次に出る泡の色 */
    private int nextColor;

    /** 次の次に出る泡の色 */
    private int nextNextColor;

    /** 固定した泡の色 */
    private int[][] data;

    /** 今左移動キーが押されているかどうか */
    private boolean isLeft;

    /** 今右移動キーが押されているかどうか */
    private boolean isRight;

    /** 今上移動キーが押されているかどうか */
    private boolean isUp;

    /** 最後に取得してから、左移動キーが押されたかどうか */
    private boolean isLeftKeyPressed;

    /** 最後に取得してから、右移動キーが押されたかどうか */
    private boolean isRightKeyPressed;

    /** 最後に取得してから、上移動キーが押されたかどうか */
    private boolean isUpKeyPressed;

    /** 動いている泡の座標 */
    private Point location;

    /** 動いている泡のグリッド位置 */
    private Point gridLocation;

    /** ゲームの状態 */
    private int status;

    /** ゲーム中の状態 */
    private int playingStatus;

    /** 落ちている泡 */
    private Collection<MovingBubble> fallingBubbles;

    /** 連鎖反応している泡 */
    private Collection<MovingBubble> chainBubbles;

    /** おじゃまぷよ */
    private Collection<MovingBubble> jumBubbles;

    /** 相手に送りたいおじゃまぷよの数 */
    private int putJam;

    /** 相手から受け取ったおじゃまぷよの数 */
    private int gotJam;

    /** 勝ち数 */
    private int winCount;

    /** 動いている泡が自動的に上に動く速度 */
    private int speed;

    /** ゲームが開始された時刻 */
    private long startTime;

    /** 連鎖の回数 */
    private int chainCount;

    /** ルール */
    private int rule;

    /** 泡の色を決める乱数 */
    private Random random;

    /** 勝ったかどうか */
    private boolean isWon;

    /** 使ってはいけない泡の色の一覧 */
    private boolean badColors[];

    /**
     * ハンデ
     */
    private int odds;
    
    /**
     * ゲームが初期化されるときに呼び出されます。 
     */
    public Game() {
        clearData();
        clearBadColors();
        this.status = Const.STATUS_BEFORE_START;
        this.fallingBubbles = new LinkedList<MovingBubble>();
        this.chainBubbles = new LinkedList<MovingBubble>();
        this.jumBubbles = new LinkedList<MovingBubble>();
        this.speed = Const.VERTICAL_SLOW_SPEED;
    }

    /**
     * 盤をクリアします。
     */
    public void clearData() {
        this.data = new int[Const.STAGE_ROWS][Const.STAGE_COLS];
        for (int i = 0; i < Const.STAGE_ROWS; ++i) {
            Arrays.fill(this.data[i], Const.NO_BUBBLE);
        }
        this.startTime = System.currentTimeMillis();
    }

    /**
     * 動いている泡の色を取得します。
     * @return color 色
     */
    public int getColor() {
        return this.color;
    }

    /**
     * 固定した泡の色を取得します。
     * @param row 行
     * @param col 列
     * @return 泡の色
     */
    public int getData(int row, int col) {
        if (row < 0 || col < 0 || row >= Const.STAGE_ROWS || col >= Const.STAGE_COLS) {
            return Const.NO_BUBBLE;
        }
        return this.data[row][col];
    }

    /**
     * 動いている泡の座標を取得します。
     * @return 動いている泡の座標
     */
    public Point getLocation() {
        return this.location;
    }

    /**
     * ゲームの状態を取得します。
     * @return ゲームの状態
     */
    public int getStatus() {
        return this.status;
    }

    /**
     * 動いている泡の色を設定します。
     * @param color 色
     */
    public void setColor(int color) {
        this.color = color;
    }

    /**
     * 固定した泡の色を設定します。
     * @param value 値
     * @param row 行
     * @param col 列
     */
    public void setData(int value, int row, int col) {
        this.data[row][col] = value;
    }

    /**
     * 動いている泡の座標を設定します。
     * @param x x座標
     * @param y y座標
     */
    public void setLocation(int x, int y) {
        if (this.location == null) {
            this.location = new Point(x, y);
        } else {
            this.location.setLocation(x, y);
        }
        this.gridLocation = Util.toGridLocation(x, y);
    }

    /**
     * 動いている泡を左に動かします。
     */
    public void moveLeft() {
        int limit = 0;
        if (this.location.x - Const.HORIZONTAL_SPEED > limit) {
            this.location.x -= Const.HORIZONTAL_SPEED;
        } else {
            this.location.x = limit;
        }
        Point newGridLocation = Util.toGridLocation(this.location.x, this.location.y);
        if (this.getData(newGridLocation.y, newGridLocation.x) == Const.NO_BUBBLE) {
            this.gridLocation = newGridLocation;
        }
    }

    /**
     * 動いている泡を右に動かします。
     */
    public void moveRight() {
        int limit = Const.GRID_WIDTH * (Const.STAGE_COLS - 1) - 1;
        if (this.location.x + Const.HORIZONTAL_SPEED < limit) {
            this.location.x += Const.HORIZONTAL_SPEED;
        } else {
            this.location.x = limit;
        }
        Point newGridLocation = Util.toGridLocation(this.location.x, this.location.y);
        if (this.getData(newGridLocation.y, newGridLocation.x) == Const.NO_BUBBLE) {
            this.gridLocation = newGridLocation;
        }
    }

    /**
     * ゆっくり上に動かします。
     */
    public void moveUpSlowly() {
        this.location.y -= this.speed;
        Point newGridLocation = Util.toGridLocation(this.location.x, this.location.y);
        if (this.getData(newGridLocation.y, newGridLocation.x) == Const.NO_BUBBLE) {
            this.gridLocation = newGridLocation;
        }
    }

    /**
     * 速く上に動かします。
     */
    public void moveUpFast() {
        this.location.y -= Const.VERTICAL_FAST_SPEED;
        Point newGridLocation = Util.toGridLocation(this.location.x, this.location.y);
        if (this.getData(newGridLocation.y, newGridLocation.x) == Const.NO_BUBBLE) {
            this.gridLocation = newGridLocation;
        }
    }

    /**
     * 動いている泡を加速します。
     */
    public void accel() {
        if (this.speed * 2 < Const.VERTICAL_FAST_SPEED) {
            this.speed *= 2;
        } else {
            this.speed = Const.VERTICAL_FAST_SPEED;
        }
    }

    /**
     * ゲームの状態を設定します。
     * @param status ゲームの状態
     */
    public void setStatus(int status) {
        this.status = status;
    }

    /**
     * 今左移動キーが押されているかどうかを取得します。
     * @return 今左移動キーが押されているかどうか
     */
    public boolean isLeft() {
        return this.isLeft;
    }

    /**
     * 左移動キーが押されているかどうかを設定します。
     * @param isLeft 左移動キーが押されているかどうか
     */
    public void setLeft(boolean isLeft) {
        this.isLeft = isLeft;
    }

    /**
     * 左移動キーが押されているかどうかを取得します。
     * @return 左移動キーが押されているかどうか
     */
    public boolean isRight() {
        return this.isRight;
    }

    /**
     * 右移動キーが押されているかどうかを設定します。
     * @param isRight 右移動キーが押されているかどうか
     */
    public void setRight(boolean isRight) {
        this.isRight = isRight;
    }

    /**
     * 上移動キーが押されているかどうかを取得します。
     * @return 上移動キーが押されているかどうか
     */
    public boolean isUp() {
        return this.isUp;
    }

    /**
     * 上移動キーが押されているかどうかを設定します。
     * @param isUp 上移動キーが押されているかどうか
     */
    public void setUp(boolean isUp) {
        this.isUp = isUp;
    }

    /**
     * ゲーム中の状態を取得します。
     * @return playingStatus
     */
    public int getPlayingStatus() {
        return this.playingStatus;
    }

    /**
     * ゲーム中の状態を設定します。
     * @param playingStatus ゲーム中の状態
     */
    public void setPlayingStatus(int playingStatus) {
        this.playingStatus = playingStatus;
    }

    /**
     * 落ちている泡の一覧を取得します。
     * @return droppingBubble 落ちている泡の一覧
     */
    public Collection<MovingBubble> getFallingBubbles() {
        return this.fallingBubbles;
    }

    /**
     * 次に出る泡の色を取得します。
     * @return nextColor 泡の色
     */
    public int getNextColor() {
        return this.nextColor;
    }

    /**
     * 次に出る泡の色を設定します。
     * @param nextColor 泡の色
     */
    public void setNextColor(int nextColor) {
        this.nextColor = nextColor;
    }

    /**
     * 連鎖反応中の泡の一覧を取得します。
     * @return chainBubble 連鎖反応中の泡の一覧
     */
    public Collection<MovingBubble> getChainBubbles() {
        return this.chainBubbles;
    }

    /**
     * 相手から受け取ったおじゃまぷよの数を取得します。
     * @return gotJam 受け取ったおじゃまぷよの数
     */
    public int getGotJam() {
        return this.gotJam;
    }

    /**
     * 相手から受け取るおじゃまぷよの数を設定します。
     * @param gotJam 受け取るおじゃまぷよの数
     */
    public void setGotJam(int gotJam) {
        this.gotJam = gotJam;
        if (gotJam > 0) {
        }
    }

    /**
     * 相手に送るおじゃまぷよの数を取得します。
     * @return putJam 送るおじゃまぷよの数
     */
    public int getPutJam() {
        return this.putJam;
    }

    /**
     * 相手に送りたいおじゃまぷよの数を設定します。
     * @param putJam  送りたいおじゃまぷよの数
     */
    public void setPutJam(int putJam) {
        this.putJam = putJam;
    }

    /**
     * おじゃまぷよを送ります。
     * @param number
     */
    public void sendJam(int number) {
        this.putJam += number;
        if (this.putJam > this.gotJam) {
            this.putJam -= this.gotJam;
            this.gotJam = 0;
        } else if (this.putJam < this.gotJam) {
            this.gotJam -= this.putJam;
            this.putJam = 0;
        } else {
            this.gotJam = 0;
            this.putJam = 0;
        }
    }

    /**
     * 移動中のおじゃまぷよを取得します。
     * @return jumBubbles 移動中のおじゃまぷよ
     */
    public Collection<MovingBubble> getJumBubbles() {
        return this.jumBubbles;
    }

    /**
     * 勝ち数を取得します。
     * @return winCount 勝ち数
     */
    public int getWinCount() {
        return this.winCount;
    }

    /**
     * 勝ち数を1増やします。
     */
    public void incrementWinCount() {
        ++this.winCount;
    }

    /**
     * 速度を元に戻します。 
     */
    public void initializeSpeed() {
        this.speed = Const.VERTICAL_SLOW_SPEED;
    }

    /**
     * 最後に加速してから経過した時間を取得します。
     * @return 経過時間
     */
    public long getElapsedTime() {
        return System.currentTimeMillis() - this.startTime;
    }

    /**
     * タイマーをリセットします。
     */
    public void resetTimer() {
        this.startTime = System.currentTimeMillis();
    }

    /**
     * 連鎖数を取得します。
     * @return 連鎖数
     */
    public int getChainCount() {
        return this.chainCount;
    }

    /**
     * 連鎖数をリセットします。
     */
    public void resetChainCount() {
        this.chainCount = 0;
    }

    /**
     * 連鎖数を1増やします。
     */
    public void incrementChainCount() {
        ++this.chainCount;
    }

    /**
     * 動いている泡のグリッド位置を取得します。
     * @return 動いている泡のグリッド位置
     */
    public Point getGridLocation() {
        return this.gridLocation;
    }

    /**
     * ルールを取得します。
     * @return ルール
     */
    public int getRule() {
        return this.rule;
    }

    /**
     * ルールを設定します。
     * @param rule ルール
     */
    public void setRule(int rule) {
        this.rule = rule;
    }

    /**
     * 最後にこのメソッドが呼び出されてから、左移動キーが押されたかどうかを取得します。
     * @return 左移動キーが押されたかどうか
     */
    public boolean isLeftKeyPressed() {
        boolean ret = this.isLeftKeyPressed;
        this.isLeftKeyPressed = false;
        return ret;
    }

    /**
     * 最後にこのメソッドが呼び出されてから、右移動キーが押されたかどうかを取得します。
     * @return 右移動キーが押されたかどうか
     */
    public boolean isRightKeyPressed() {
        boolean ret = this.isRightKeyPressed;
        this.isRightKeyPressed = false;
        return ret;
    }

    /**
     * 最後にこのメソッドが呼び出されてから、上移動キーが押されたかどうかを取得します。
     * @return 上移動キーが押されたかどうか
     */
    public boolean isUpKeyPressed() {
        boolean ret = this.isUpKeyPressed;
        this.isUpKeyPressed = false;
        return ret;
    }

    /**
     * 最後に取得されてから、左移動キーが押されたかどうかを設定します。
     * @param isLeftKeyPressed 左移動キーが押されたかどうか
     */
    public void setLeftKeyPressed(boolean isLeftKeyPressed) {
        this.isLeftKeyPressed = isLeftKeyPressed;
    }

    /**
     * 最後に取得されてから、右移動キーが押されたかどうかを設定します。
     * @param isRightKeyPressed 右移動キーが押されたかどうか
     */
    public void setRightKeyPressed(boolean isRightKeyPressed) {
        this.isRightKeyPressed = isRightKeyPressed;
    }

    /**
     * 最後に取得されてから、上移動キーが押されたかどうかを設定します。
     * @param isUpKeyPressed 上移動キーが押されたかどうか
     */
    public void setUpKeyPressed(boolean isUpKeyPressed) {
        this.isUpKeyPressed = isUpKeyPressed;
    }

    /**
     * 泡を発生するための乱数を設定します。
     * @param random 乱数
     */
    public void setRandom(Random random) {
        this.random = random;
    }

    /**
     * 泡の色をランダムに発生させます。
     * @return 泡の色
     */
    public int getRandomColor() {
        for (int i = 0; i < 10000; ++i) {
            int newColor = this.random.nextInt(Const.COLOR_NUMBER);
            if (!this.badColors[newColor]) {
                return newColor;
            }
        }
        return 0;
    }

    /**
     * 勝ったかどうかを取得します。
     * @return 勝ったかどうか
     */
    public boolean isWon() {
        return this.isWon;
    }

    /**
     * 勝ったかどうかを設定します。
     * @param isWon 勝ったかどうか
     */
    public void setWon(boolean isWon) {
        this.isWon = isWon;
    }

    /**
     * 使ってはいけない色の一覧をクリアします。 
     */
    public void clearBadColors() {
        this.badColors = new boolean[Const.COLOR_NUMBER];
    }

    /**
     * 使用禁止にする色の一覧を設定します。
     * @param badColors 使用禁止にする色の一覧
     */
    public void setBadColors(boolean[] badColors) {
        this.badColors = badColors;
    }

    /**
     * 指定した色が使用不可能かどうかを取得します。
     * @param newColor 色
     * @return 使用不可能かどうか
     */
    public boolean isBad(int newColor) {
        return this.badColors[newColor];
    }

    /**
     * 次の次に出る泡の色を取得します。
     * @return 次の次に出る泡の色
     */
    public int getNextNextColor() {
        return this.nextNextColor;
    }

    /**
     * 次の次に出る泡の色を設定します。
     * @param nextNextColor 次の次に出る泡の色
     */
    public void setNextNextColor(int nextNextColor) {
        this.nextNextColor = nextNextColor;
    }
    
    /**
     * ハンデを取得します。
     * @return ハンデ
     */
    public int getOdds() {
        return this.odds;
    }
    
    /**
     * ハンデを取得します。
     * @param odds
     */
    public void setOdds(int odds) {
        this.odds = odds;
    }
}