<?php
/*
Plugin Name: JSeries Notifier
Plugin URI: http://wppluginsj.sourceforge.jp/jseries-notifier/
Version: 0.9.0
Description: Notify updates of plugins which is supplied as WordPress Plugins/JSeries.
Author: IKEDA Yuriko
Author URI: http://www.yuriko.net/cat/wordpress/
Text Domain: jseries-notifier
*/

/*  Copyright (c) 2007-2009 yuriko

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

if (! is_admin() || defined('WP_INSTALLING') && WP_INSTALLING ) {
	return;
}
define ('JN_USE_LOCAL_FILE', false); // Debug use
define ('JSERIES_SITE_HOST', 'sourceforge.jp');
define ('JSERIES_REL_JS_PATH', '/projects/wppluginsj/releases/compact.js');
define ('JSERIES_REL_JSON', 'w.set_data({');
define ('JN_RETRIEVE_INTERNAL', 86400);
define ('JN_BAD_VERSION', '0.0.0');
define ('JN_SFJP_NO_RESPONSE', -1);
define ('JN_JSON_PARSE_ERROR', -2);

$JS_Notifier = new JSeries_Notifier;

/* ==================================================
 *   JSeries_Notifier class
   ================================================== */

class JSeries_Notifier {
	var $wp_vers;
	var $plugin_dir;
	var $plugin_url;
	var $plugin_name;
	var $plugin_dist_uri;
	var $versions;

function JSeries_Notifier() {
	$this->__construct();
}

function __construct() {
	$this->wp_vers = NULL;
	if ($this->check_wp_version('2.3', '<')) {
		return;
	}
	$this->set_plugin_dir();
	add_action('plugins_loaded', array($this, 'load_textdomain'));
	add_action('load-plugins.php', array($this, 'check_updates'));
	add_action('wp_update_plugins', array($this, 'check_updates'));
	add_action('admin_init', array($this, 'maybe_check_updates'));
	if ($this->check_wp_version('2.6')) {
		add_action('after_plugin_row', array($this, 'plugin_update_row'), 10, 2);
	} else {
		add_action('after_plugin_row', array($this, 'plugin_update_row23'));
	}
	register_activation_hook(__FILE__, array($this, 'delete_option'));
	register_deactivation_hook(__FILE__, array($this, 'delete_option'));
	return;
}

/* ==================================================
 * @param	string  $key
 * @return	boolean $charset
 */
// public 
function get($key) {
	return $this->$key;
}

/* ==================================================
 * @param	none
 * @return	none
 */
// private 
function set_plugin_dir() {
	$this->plugin_dir = basename(dirname(__FILE__));
	if (function_exists('plugins_url')) {
		$this->plugin_url = plugins_url($this->plugin_dir . '/');
	} else {
		$this->plugin_url = get_bloginfo('wpurl') . '/' 
		. (defined('PLUGINDIR') ? PLUGINDIR . '/': 'wp-content/plugins/') 
		. $this->plugin_dir . '/';
	}
}

/* ==================================================
 * @param	string   $domain
 * @param	string   $subdir
 * @return	none
 */
// public 
function load_textdomain() {
	$domain = 'jseries-notifier';
	$subdir = '';
	$lang_dir = $this->get('plugin_dir') . ($subdir ? '/' . $subdir : '');
	if ($this->check_wp_version('2.6')) {
		load_plugin_textdomain($domain, false, $lang_dir);
	} else {
		$plugin_path = defined('PLUGINDIR') ? PLUGINDIR . '/': 'wp-content/plugins/';
		load_plugin_textdomain($domain, $plugin_path . $lang_dir);
	}
}

/* ==================================================
 * @param	string   $version
 * @param	string   $operator
 * @return	boolean  $result
 */
// private 
function check_wp_version($version, $operator = '>=') {
	if (! $this->wp_vers) {
		$this->wp_vers = get_bloginfo('version');
		if (! is_numeric($this->wp_vers)) {
			$this->wp_vers = preg_replace('/[^.0-9]/', '', $this->wp_vers);  // strip 'ME'
		}
	}
	return version_compare($this->wp_vers, $version, $operator);
}

/* ==================================================
 * @param	none
 * @return	none
 */
// public 
function delete_option() {
	delete_option('jseries_plugins');
}

/* ==================================================
 * @param	none
 * @return	array  $versions
 * based on wp_update_plugins() at wp-admin/includes/update.php of WP 2.3.1
 */
// public 
function maybe_check_updates() {
	$current = get_option('jseries_plugins');
	if ( isset( $current->last_checked ) && (JN_RETRIEVE_INTERNAL / 2) > ( time() - $current->last_checked ) ) {
		return;
	}
	$this->check_updates();
}

/* ==================================================
 * @param	none
 * @return	array  $versions
 * based on wp_update_plugins() at wp-admin/includes/update.php of WP 2.3.1
 */
// public 
function check_updates() {
	if (! function_exists('get_plugins') && file_exists(ABSPATH . 'wp-admin/includes/plugin.php')) {
		require_once(ABSPATH . 'wp-admin/includes/plugin.php');
	}

	$plugins = get_plugins();
	$current = get_option('jseries_plugins');
	$versions = array();

	$this_plugin = plugin_basename(__FILE__);
	$this->plugin_name = $plugins[$this_plugin]['Name'];
	if (isset($plugins[$this_plugin]['PluginURI'])) {
		$this->plugin_dist_uri = $plugins[$this_plugin]['PluginURI'];
	} elseif (preg_match('/a href="([^"]*)"/', $plugins[$this_plugin]['Title'], $url)) {
		$this->plugin_dist_uri = $url[1];
	} else {
		$this->plugin_dist_uri = 'http://wppluginsj.sourceforge.jp/';
	}

	$new_option = '';
	$new_option->last_checked = time();

	$plugin_changed = false;
	foreach ( $plugins as $file => $p ) {
		$versions[$p['Name']] = $p['Version'];
		$new_option->checked[ $file ] = $p['Version'];

		if ( ! isset( $current->checked[ $file ] ) ) {
			$plugin_changed = true;
			continue;
		}

		if ( $current->checked[ $file ] != $p['Version'] )
			$plugin_changed = true;
	}
	$this->versions = $versions;

	if (
		isset( $current->last_checked ) &&
		JN_RETRIEVE_INTERNAL > ( time() - $current->last_checked ) &&
		! $plugin_changed
	) {
		return false;
	}
	
	$js = $this->retrive_latest_versions();
	if ($js) {
		$response = $this->pickup_versions($js);
		if ( $response ) {
			$new_option->response = $response;
		}
	}
	update_option('jseries_plugins', $new_option);

}

/* ==================================================
 * @param	none
 * @return	mix     $response
 * based on wp_update_plugins() at wp-admin/includes/update.php of WP 2.3.1
 */
// private 
function retrive_latest_versions() {
	$response = '';
	if (defined('JN_USE_LOCAL_FILE') && JN_USE_LOCAL_FILE) {
		$response = file_get_contents(dirname(__FILE__) . '/compact.js');
	} elseif (function_exists('wp_remote_get')) {
		$http_result = wp_remote_get('http://' . JSERIES_SITE_HOST . JSERIES_REL_JS_PATH);
		if (is_wp_error($http_result)) {
			$response = $http_result;
		} else {
			$response = $http_result['body'];
		}
	} elseif (function_exists('fsockopen')) {
		global $wp_version;
		$http_request = array();
		$http_request[] = 'GET ' . JSERIES_REL_JS_PATH . ' HTTP/1.0';
		$http_request[] = 'Host: ' . JSERIES_SITE_HOST;
		$http_request[] = 'User-Agent: WordPress/' . $wp_version . '; ' . get_bloginfo('url');
	
		if ( false != ( $fs = @fsockopen( JSERIES_SITE_HOST, 80, $errno, $errstr, 3) ) && is_resource($fs) ) {
			fwrite($fs, implode("\r\n", preg_replace('/[\r\n]/', '', $http_request)) . "\r\n\r\n");
	
			while (! feof($fs)) {
				$response .= fgets($fs, 1160); // One TCP-IP packet
			}
			fclose($fs);
		}
	}
	return $response;
}

/* ==================================================
 * @param	string  $js
 * @return	mix     $response
 */
// private 
function pickup_versions($js) {
	if (empty($js)) {
		return $this->set_error(JN_SFJP_NO_RESPONSE);
	}
	if (is_wp_error($js)) {
		return $this->set_error(JN_SFJP_NO_RESPONSE, $js->get_error_message());
	}
	$pos = strpos($js, JSERIES_REL_JSON, 0);
	if ($pos === false) {
		return $this->set_error(JN_JSON_PARSE_ERROR);
	}
	$response = array();
	if (preg_match_all(
		'!\'package\': "([^"]*)",\s*\'version\': "([^"]*)",\s*\'date\': \d+,\s*\'link\': "([^"]*)"!s', 
		$js, $plugins, PREG_SET_ORDER, $pos)) {
		foreach ($plugins as $p) {
			$name = $p[1];
			$vers = $p[2];
			$link = $p[3];
			if (isset($this->versions[$name]) && version_compare($this->versions[$name], $vers, '<')) {
				$response[$name]->url = $link;
				$response[$name]->new_version = $vers;
			}
		}
	} else {
		return $this->set_error(JN_JSON_PARSE_ERROR);
	}

	return $response;
}

/* ==================================================
 * @param	int      $code
 * @param	string   $http_error
 * @return	array    $response
 */
// private  
function set_error($code, $http_error = '') {
	$response[$this->plugin_name]->url = $this->plugin_dist_uri;
	$response[$this->plugin_name]->new_version = JN_BAD_VERSION;
	$response[$this->plugin_name]->error_code = $code;
	$response[$this->plugin_name]->http_error = $http_error;
	return $response;
}

/* ==================================================
 * @param	string  $file
 * @return	none
 */
// public  
function plugin_update_row23($file) {
	global $plugin_data;
	$this->plugin_update_row($file, $plugin_data);
}

/* ==================================================
 * @param	string  $file
 * @return	none
 * based on wp_plugin_update_row() at wp-admin/includes/update.php of WP 2.5
 */
// public  
function plugin_update_row($file, $plugin_data) {
	$current = get_option( 'jseries_plugins' );
	if (! isset( $current->response[$plugin_data['Name']])) {
		return false;
	}
	$r = $current->response[$plugin_data['Name']];

	if ($r->new_version == JN_BAD_VERSION) {
		$error = __('<strong>An error occured!</strong> ', 'jseries-notifier') . $this->get_error_message($r->error_code);
		if (isset($r->http_error)) {
			$error .= $r->http_error;
		}
	} else {
		$error = '';
	}
	if ($this->check_wp_version('2.8')) {
		$before = '<tr class="plugin-update-tr"><td colspan="3" class="plugin-update"><div class="update-message">';
		$format = str_replace(' class="thickbox"', '', __('There is a new version of %1$s available. <a href="%2$s" class="thickbox" title="%1$s">View version %3$s Details</a>.'));
		$after = '</div></td></tr>';
	} elseif ($this->check_wp_version('2.7')) {
		$before =  "<tr><td colspan='5' class='plugin-update'>";
		$format = str_replace(' class="thickbox"', '', __('There is a new version of %1$s available. <a href="%2$s" class="thickbox" title="%1$s">View version %3$s Details</a>.'));
		$after = '</td></tr>';
	} elseif ($this->check_wp_version('2.5')) { // for WP 2.5 to 2.6
		$before = "<tr><td colspan='5' class='plugin-update'>";
		$format = __('There is a new version of %1$s available. <a href="%2$s">Download version %3$s here</a>.');
		$after = '</td></tr>';
	} else { // for WP 2.3
		$before = "<tr><td colspan='5' class='plugin-update'>";
		$format = __('There is a new version of %s available. <a href="%s">Download version %s here</a>.');
		$after = '</td></tr>';
	}
	echo $before 
	. sprintf(($error ? $error : $format), $plugin_data['Name'], attribute_escape($r->url), attribute_escape($r->new_version))
	. $after;
}

/* ==================================================
 * @param	int      $code
 * @return	string   $message
 */
// private  
function get_error_message($code) {
	$message = '';
	switch ($code) {
	case JN_SFJP_NO_RESPONSE:
		$message = __('No response from SourceForge.jp.', 'jseries-notifier');
		break;
	case JN_JSON_PARSE_ERROR:
		$message = __('The format of release updates has been changed. Please <a href="%2$s">confirm</a> a new version of %1$s that can parse the new format.', 'jseries-notifier');
		break;
	}
	return $message;
}

// ===== End of class ====================
}
?>