/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <stdlib.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "timidity.h"
#include "common.h"
#include "mblock.h"

#include "mblock_prv.h"

#define ADDRALIGN 8
/* #define DEBUG */

void init_mblock(tmdy_struct_ex_t *tmdy_struct, MBlockList *mblock)
{
    mblock->first = NULL;
    mblock->allocated = 0;
}

static MBlockNode *new_mblock_node(tmdy_struct_ex_t *tmdy_struct, size_t n)
{
    MBlockNode *p;
	
	timidity_mutex_lock(MBLOCK->busy);
    if(n > MIN_MBLOCK_SIZE)
    {
	if((p = (MBlockNode *)tmdy_struct->common->safe_malloc(tmdy_struct, n + sizeof(MBlockNode))) == NULL)
	    return NULL;
	p->block_size = n;
    }
    else if(MBLOCK->free_mblock_list == NULL)
    {
	if((p = (MBlockNode *)tmdy_struct->common->safe_malloc(tmdy_struct, sizeof(MBlockNode)
				     + MIN_MBLOCK_SIZE)) == NULL)
	    return NULL;
	p->block_size = MIN_MBLOCK_SIZE;
    }
    else
    {
	p = MBLOCK->free_mblock_list;
	MBLOCK->free_mblock_list = MBLOCK->free_mblock_list->next;
    }

    p->offset = 0;
    p->next = NULL;

	timidity_mutex_unlock(MBLOCK->busy);
    return p;
}

static int enough_block_memory(tmdy_struct_ex_t *tmdy_struct, MBlockList *mblock, size_t n)
{
    size_t newoffset;

    if(mblock->first == NULL)
	return 0;

    newoffset = mblock->first->offset + n;

    if(newoffset < mblock->first->offset) /* exceed representable in size_t */
	return 0;

    if(newoffset > mblock->first->block_size)
	return 0;

    return 1;
}

void *new_segment(tmdy_struct_ex_t *tmdy_struct, MBlockList *mblock, size_t nbytes)
{
    MBlockNode *p;
    void *addr;

    /* round up to ADDRALIGN */
    nbytes = ((nbytes + ADDRALIGN - 1) & ~(ADDRALIGN - 1));
    if(!enough_block_memory(tmdy_struct, mblock, nbytes))
    {
	p = new_mblock_node(tmdy_struct, nbytes);
	p->next = mblock->first;
	mblock->first = p;
	mblock->allocated += p->block_size;
    }
    else
	p = mblock->first;

    addr = (void *)(p->buffer + p->offset);
    p->offset += nbytes;

#ifdef DEBUG
    if(((unsigned long)addr) & (ADDRALIGN-1))
    {
	fprintf(stderr, "Bad address: 0x%x\n", addr);
	exit(1);
    }
#endif /* DEBUG */

    return addr;
}

static void reuse_mblock1(tmdy_struct_ex_t *tmdy_struct, MBlockNode *p)
{
	timidity_mutex_lock(MBLOCK->busy);
    if(p->block_size > MIN_MBLOCK_SIZE)
	free(p);
    else /* p->block_size <= MIN_MBLOCK_SIZE */
    {
	p->next = MBLOCK->free_mblock_list;
	MBLOCK->free_mblock_list = p;
    timidity_mutex_unlock(MBLOCK->busy);
    }
}

void reuse_mblock(tmdy_struct_ex_t *tmdy_struct, MBlockList *mblock)
{
    MBlockNode *p;

    if((p = mblock->first) == NULL)
	return;			/* There is nothing to collect memory */

    while(p)
    {
	MBlockNode *tmp;

	tmp = p;
	p = p->next;
	reuse_mblock1(tmdy_struct, tmp);
    }
    init_mblock(tmdy_struct, mblock);
}

char *strdup_mblock(tmdy_struct_ex_t *tmdy_struct, MBlockList *mblock, const char *str)
{
    int len;
    char *p;

    len = strlen(str);
    p = (char *)new_segment(tmdy_struct, mblock, len + 1); /* for '\0' */
    memcpy(p, str, len + 1);
    return p;
}

int free_global_mblock(tmdy_struct_ex_t *tmdy_struct)
{
    int cnt;

	timidity_mutex_lock(MBLOCK->busy);
    cnt = 0;
    while(MBLOCK->free_mblock_list)
    {
	MBlockNode *tmp;

	tmp = MBLOCK->free_mblock_list;
	MBLOCK->free_mblock_list = MBLOCK->free_mblock_list->next;
	free(tmp);
	cnt++;
    }
	timidity_mutex_unlock(MBLOCK->busy);
    return cnt;
}




mblock_ex_t* new_mblock(tmdy_struct_ex_t *tmdy_struct){
	int i;
	mblock_ex_t* mblock_ex;

	mblock_ex=(mblock_ex_t *)malloc(sizeof(mblock_ex_t));
	
timidity_mutex_init(mblock_ex->busy);	
mblock_ex->init_mblock=init_mblock;
mblock_ex->new_segment=new_segment;
mblock_ex->reuse_mblock=reuse_mblock;
mblock_ex->strdup_mblock=strdup_mblock;
mblock_ex->free_global_mblock=free_global_mblock;

mblock_ex->free_mblock_list = NULL;
	
	return mblock_ex;
}
void destroy_mblock(mblock_ex_t* mblock){
	timidity_mutex_destroy(mblock->busy);
	free(mblock);
}
