/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#ifndef __W32__
#include <unistd.h>
#endif
#include <stdlib.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "tmdy_struct.h"
#include "timidity.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "output.h"
#include "controls.h"
#include "tables.h"
#include "mblock.h"
#include "recache.h"
#include "resample.h"

#include "recache_prv.h"


#define CACHE_DATA_LEN (TMDY_RECACHE->allocate_cache_size/sizeof(sample_t))
#define RESAMPLATION_CACHE dest[i] = (TMDY_RESAMPLE->do_resamplation)(tmdy_struct, src, ofs, &resrc)



void resamp_cache_reset(tmdy_struct_ex_t *tmdy_struct)
{
    if(TMDY_RECACHE->cache_data == NULL)
    {
	TMDY_RECACHE->cache_data =
	    (sample_t *)TMDY_COMMON->safe_large_malloc(tmdy_struct, (CACHE_DATA_LEN+1) * sizeof(sample_t));
	memset(TMDY_RECACHE->cache_data, 0, (CACHE_DATA_LEN+1) * sizeof(sample_t));
	TMDY_UTILS->mblock->init_mblock(tmdy_struct, &TMDY_RECACHE->hash_entry_pool);
    }
    TMDY_RECACHE->cache_data_len = 0;
    memset(TMDY_RECACHE->cache_hash_table, 0, sizeof(TMDY_RECACHE->cache_hash_table));
    memset(TMDY_RECACHE->channel_note_table, 0, sizeof(TMDY_RECACHE->channel_note_table));
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_RECACHE->hash_entry_pool);
}

#define sp_hash(sp, note) ((unsigned int)(sp) + (unsigned int)(note))

struct cache_hash *resamp_cache_fetch(tmdy_struct_ex_t *tmdy_struct, Sample *sp, int note)
{
    unsigned int addr;
    struct cache_hash *p;

    if(sp->vibrato_control_ratio ||
       (sp->modes & MODES_PINGPONG) ||
       (sp->sample_rate == (TMDY_OUTPUT->play_mode)->rate &&
	sp->root_freq == freq_table[sp->note_to_use]))
	    return NULL;

    addr = sp_hash(sp, note) % HASH_TABLE_SIZE;
    p = TMDY_RECACHE->cache_hash_table[addr];
    while(p && (p->note != note || p->sp != sp))
	p = p->next;
    if(p && p->resampled != NULL)
	return p;
    return NULL;
}

void resamp_cache_refer_on(tmdy_struct_ex_t *tmdy_struct, Voice *vp, int32 sample_start)
{
    unsigned int addr;
    struct cache_hash *p;
    int note, ch;

    ch = vp->channel;

    if(vp->vibrato_control_ratio ||
       TMDY_PLAYMIDI->channel[ch].portamento ||
       (vp->sample->modes & MODES_PINGPONG) ||
       vp->orig_frequency != vp->frequency ||
       (vp->sample->sample_rate == (TMDY_OUTPUT->play_mode)->rate &&
	vp->sample->root_freq == freq_table[vp->sample->note_to_use]))
	    return;

    note = vp->note;

    if(TMDY_RECACHE->channel_note_table[ch].cache[note])
	resamp_cache_refer_off(tmdy_struct, ch, note, sample_start);

    addr = sp_hash(vp->sample, note) % HASH_TABLE_SIZE;
    p = TMDY_RECACHE->cache_hash_table[addr];
    while(p && (p->note != note || p->sp != vp->sample))
	p = p->next;

    if(!p)
    {
	p = (struct cache_hash *)
	    TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_RECACHE->hash_entry_pool, sizeof(struct cache_hash));
	p->cnt = 0;
	p->note = vp->note;
	p->sp = vp->sample;
	p->resampled = NULL;
	p->next = TMDY_RECACHE->cache_hash_table[addr];
	TMDY_RECACHE->cache_hash_table[addr] = p;
    }
    TMDY_RECACHE->channel_note_table[ch].cache[note] = p;
    TMDY_RECACHE->channel_note_table[ch].on[note] = sample_start;
}

void resamp_cache_refer_off(tmdy_struct_ex_t *tmdy_struct, int ch, int note, int32 sample_end)
{
    int32 sample_start, len;
    struct cache_hash *p;
    Sample *sp;

    p = TMDY_RECACHE->channel_note_table[ch].cache[note];
    if(p == NULL)
	return;

    sp = p->sp;
    if(sp->sample_rate == (TMDY_OUTPUT->play_mode)->rate &&
       sp->root_freq == freq_table[sp->note_to_use])
	return;
    sample_start = TMDY_RECACHE->channel_note_table[ch].on[note];

    len = sample_end - sample_start;
    if(len < 0)
    {
	TMDY_RECACHE->channel_note_table[ch].cache[note] = NULL;
	return;
    }

    if(!(sp->modes & MODES_LOOPING))
    {
	double a;
	int32 slen;

	a = ((double)sp->root_freq * (TMDY_OUTPUT->play_mode)->rate) /
	    ((double)sp->sample_rate * freq_table[note]);
	slen = (int32)((sp->data_length >> FRACTION_BITS) * a);
	if(len > slen)
	    len = slen;
    }
    p->cnt += len;
    TMDY_RECACHE->channel_note_table[ch].cache[note] = NULL;
}

void resamp_cache_refer_alloff(tmdy_struct_ex_t *tmdy_struct, int ch, int32 sample_end)
{
    int i;
    for(i = 0; i < 128; i++)
	resamp_cache_refer_off(tmdy_struct, ch, i, sample_end);
}


static void loop_connect(tmdy_struct_ex_t *tmdy_struct, sample_t *data, int32 start, int32 end)
{
    int i, mixlen;
    int32 t0, t1;

    mixlen = MIXLEN;
    if(start < mixlen)
	mixlen = start;
    if(end - start < mixlen)
	mixlen = end - start;
    if(mixlen <= 0)
	return;

    t0 = start - mixlen;
    t1 = end   - mixlen;

    for(i = 0; i < mixlen; i++)
    {
	double x, b;

	b = i / (double)mixlen;	/* 0 <= b < 1 */
	x = b * data[t0 + i] + (1.0 - b) * data[t1 + i];
#ifdef LOOKUP_HACK
	if(x < -128)
	    data[t1 + i] = -128;
	else if(x > 127)
	    data[t1 + i] = 127;
#else
	if(x < -32768)
	    data[t1 + i] = -32768;
	else if(x > 32767)
	    data[t1 + i] = 32767;
#endif /* LOOKUP_HACK */
	else
	    data[t1 + i] = (sample_t)x;
    }
}

static double sample_resamp_info(tmdy_struct_ex_t *tmdy_struct, Sample *sp, int note,
				 splen_t *loop_start, splen_t *loop_end,
				 splen_t *data_length)
{
    splen_t xls, xle, ls, le, ll, newlen;
    double a, xxls, xxle, xn;

    a = ((double)sp->sample_rate * freq_table[note]) /
	((double)sp->root_freq * (TMDY_OUTPUT->play_mode)->rate);
    a = TIM_FSCALENEG((double)(int32)TIM_FSCALE(a, FRACTION_BITS),
		      FRACTION_BITS);

    xn = sp->data_length / a;
    if(xn >= SPLEN_T_MAX)
    {
	/* Ignore this sample */
	*data_length = 0;
	return 0.0;
    }
    newlen = (splen_t)(TIM_FSCALENEG(xn, FRACTION_BITS) + 0.5);

    ls = sp->loop_start;
    le = sp->loop_end;
    ll = (le - ls);

    xxls = ls / a + 0.5;
    if(xxls >= SPLEN_T_MAX)
    {
	/* Ignore this sample */
	*data_length = 0;
	return 0.0;
    }
    xls = (splen_t)xxls;

    xxle = le / a + 0.5;
    if(xxle >= SPLEN_T_MAX)
    {
	/* Ignore this sample */
	*data_length = 0;
	return 0.0;
    }
    xle = (splen_t)xxle;

    if((sp->modes & MODES_LOOPING) &&
       ((xle - xls) >> FRACTION_BITS) < MIN_LOOPLEN)
    {
	splen_t n;
	splen_t newxle;
	double xl; /* Resampled new loop length */
	double xnewxle;

	xl = ll / a;
	if(xl >= SPLEN_T_MAX)
	{
	    /* Ignore this sample */
	    *data_length = 0;
	    return 0.0;
	}

	n = (splen_t)(0.0001 + MIN_LOOPLEN /
		    TIM_FSCALENEG(xl, FRACTION_BITS)) + 1;
	xnewxle = le / a + n * xl + 0.5;
	if(xnewxle >= SPLEN_T_MAX)
	{
	    /* Ignore this sample */
	    *data_length = 0;
	    return 0.0;
	}

	newxle = (splen_t)xnewxle;
	newlen += (newxle - xle)>>FRACTION_BITS;
	xle = newxle;
    }

    if(loop_start)
	*loop_start = (splen_t)(xls & ~FRACTION_MASK);
    if(loop_end)
	*loop_end = (splen_t)(xle & ~FRACTION_MASK);
    *data_length = newlen << FRACTION_BITS;
    return a;
}

static int cache_resampling(tmdy_struct_ex_t *tmdy_struct, struct cache_hash *p)
{
    Sample *sp, *newsp;
    sample_t *src, *dest;
    splen_t newlen, ofs, le, ls, ll, xls, xle;
    int32 i, incr;
    resample_rec_t resrc;
    double a;

    sp = p->sp;
    a = sample_resamp_info(tmdy_struct, sp, p->note, &xls, &xle, &newlen);
    if(newlen == 0)
	return CACHE_RESAMPLING_NOTOK;

    newlen >>= FRACTION_BITS;

    if(TMDY_RECACHE->cache_data_len + newlen + 1 > CACHE_DATA_LEN)
	return CACHE_RESAMPLING_NOTOK;

    resrc.loop_start = ls = sp->loop_start;
    resrc.loop_end = le = sp->loop_end;
    resrc.data_length = sp->data_length;

    ll = (sp->loop_end - sp->loop_start);
    dest = TMDY_RECACHE->cache_data + TMDY_RECACHE->cache_data_len;
    src = sp->data;

    newsp = (Sample *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_RECACHE->hash_entry_pool, sizeof(Sample));
    memcpy(newsp, sp, sizeof(Sample));
    newsp->data = dest;

    ofs = 0;
    incr = (splen_t)(TIM_FSCALE(a, FRACTION_BITS) + 0.5);
    if(sp->modes & MODES_LOOPING)
    {
	for(i = 0; i < newlen; i++)
	{
	    if(ofs >= le)
		ofs -= ll;

	    RESAMPLATION_CACHE;
	    ofs += incr;
	}
    }
    else
    {
	for(i = 0; i < newlen; i++)
	{
	    RESAMPLATION_CACHE;
	    ofs += incr;
	}
    }
    newsp->loop_start = xls;
    newsp->loop_end = xle;
    newsp->data_length = (newlen << FRACTION_BITS);
    if(sp->modes & MODES_LOOPING)
	loop_connect(tmdy_struct, dest, (int32)(xls >> FRACTION_BITS), (int32)(xle >> FRACTION_BITS));
    dest[xle >> FRACTION_BITS] = dest[xls >> FRACTION_BITS];

    newsp->root_freq = freq_table[p->note];
    newsp->sample_rate = (TMDY_OUTPUT->play_mode)->rate;

    p->resampled = newsp;
    TMDY_RECACHE->cache_data_len += newlen + 1;

    return CACHE_RESAMPLING_OK;
}

#define SORT_THRESHOLD 20
static void insort_cache_array(tmdy_struct_ex_t *tmdy_struct, struct cache_hash **data, long n)
{
    long i, j;
    struct cache_hash *x;

    for(i = 1; i < n; i++)
    {
	x = data[i];
	for(j = i - 1; j >= 0 && x->r < data[j]->r; j--)
	    data[j + 1] = data[j];
	data[j + 1] = x;
    }
}

static void qsort_cache_array(tmdy_struct_ex_t *tmdy_struct, struct cache_hash **a, long first, long last)
{
    long i = first, j = last;
    struct cache_hash *x, *t;

    if(j - i < SORT_THRESHOLD)
    {
	insort_cache_array(tmdy_struct, a + i, j - i + 1);
	return;
    }

    x = a[(first + last) / 2];

    for(;;)
    {
	while(a[i]->r < x->r)
	    i++;
	while(x->r < a[j]->r)
	    j--;
	if(i >= j)
	    break;
	t = a[i];
	a[i] = a[j];
	a[j] = t;
	i++;
	j--;
    }
    if(first < i - 1)
	qsort_cache_array(tmdy_struct, a, first, i - 1);
    if(j + 1 < last)
	qsort_cache_array(tmdy_struct, a, j + 1, last);
}

void resamp_cache_create(tmdy_struct_ex_t *tmdy_struct)
{
    int i, skip;
    int32 n, t1, t2, total;
    struct cache_hash **array;

    /* It is NP completion that solve the best cache hit rate.
     * So I thought better algorism O(n log n), but not a best solution.
     * Follows implementation takes good hit rate, and it is fast.
     */

    n = t1 = t2 = 0;
    total = 0;

    /* set size per count */
    for(i = 0; i < HASH_TABLE_SIZE; i++)
    {
	struct cache_hash *p, *q;

	q = NULL;
	p = TMDY_RECACHE->cache_hash_table[i];
	while(p)
	{
	    struct cache_hash *tmp;

	    t1 += p->cnt;

	    tmp = p;
	    p = p->next;
	    if(tmp->cnt > 0)
	    {
		Sample *sp;
		splen_t newlen;

		sp = tmp->sp;
		sample_resamp_info(tmdy_struct, sp, tmp->note, NULL, NULL, &newlen);
		if(newlen > 0)
		{
		    total += tmp->cnt;
		    tmp->r = (double)newlen / tmp->cnt;
		    tmp->next = q;
		    q = tmp;
		    n++;
		}
	    }
	}
	TMDY_RECACHE->cache_hash_table[i] = q;
    }

    if(n == 0)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "No pre-resampling cache hit");
	return;
    }

    array = (struct cache_hash **)TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_RECACHE->hash_entry_pool,
					      n * sizeof(struct cache_hash *));
    n = 0;
    for(i = 0; i < HASH_TABLE_SIZE; i++)
    {
	struct cache_hash *p;
	for(p = TMDY_RECACHE->cache_hash_table[i]; p; p = p->next)
	    array[n++] = p;
    }

    if(total > CACHE_DATA_LEN)
	qsort_cache_array(tmdy_struct, array, 0, n - 1);

    skip = 0;
    for(i = 0; i < n; i++)
    {
	if(array[i]->r != 0 &&
	   cache_resampling(tmdy_struct, array[i]) == CACHE_RESAMPLING_OK)
	    t2 += array[i]->cnt;
	else
	    skip++;
    }

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
	      "Resample cache: Key %d/%d(%.1f%%) "
	      "Sample %.1f%c/%.1f%c(%.1f%%)",
	      n - skip, n, 100.0 * (n - skip) / n,
	      t2 / (t2 >= 1048576 ? 1048576.0 : 1024.0),
	      t2 >= 1048576 ? 'M' : 'K',
	      t1 / (t1 >= 1048576 ? 1048576.0 : 1024.0),
	      t1 >= 1048576 ? 'M' : 'K',
	      100.0 * t2 / t1);

    /* update TMDY_RECACHE->cache_hash_table */
    if(skip)
    {
	for(i = 0; i < HASH_TABLE_SIZE; i++)
	{
	    struct cache_hash *p, *q;

	    q = NULL;
	    p = TMDY_RECACHE->cache_hash_table[i];

	    while(p)
	    {
		struct cache_hash *tmp;
		tmp = p;
		p = p->next;

		if(tmp->resampled)
		{
		    tmp->next = q;
		    q = tmp;
		}
	    }
	    TMDY_RECACHE->cache_hash_table[i] = q;
	}
    }
}



recache_ex_t* new_recache(tmdy_struct_ex_t *tmdy_struct){
	int i;
	recache_ex_t* recache_ex;

	recache_ex=(recache_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(recache_ex_t));

	recache_ex->resamp_cache_reset=resamp_cache_reset;
	recache_ex->resamp_cache_refer_on=resamp_cache_refer_on;
	recache_ex->resamp_cache_refer_off=resamp_cache_refer_off;
	recache_ex->resamp_cache_refer_alloff=resamp_cache_refer_alloff;
	recache_ex->resamp_cache_create=resamp_cache_create;
	recache_ex->resamp_cache_fetch=resamp_cache_fetch;
	
	recache_ex->cache_data = NULL;
	recache_ex->allocate_cache_size = DEFAULT_CACHE_DATA_SIZE;
	
		return recache_ex;
}
void destroy_recache(recache_ex_t* recache){	
	free(recache);
}
