/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "timidity.h"
#include "common.h"
#include "url.h"
#include "url_prv.h"

/* #define DEBUG */

int url_errno;
static struct URL_module *url_mod_list = NULL;
char *user_mailaddr = NULL;
char *url_user_agent = NULL;
int url_newline_code = '\n';
char *url_lib_version = URL_LIB_VERSION;
int uudecode_unquote_html = 0;

void url_add_module(tmdy_struct_ex_t *tmdy_struct, struct URL_module *m)
{
    m->chain = url_mod_list;
    url_mod_list = m;
}

void url_add_modules(tmdy_struct_ex_t *tmdy_struct, struct URL_module *m, ...)
{
    va_list ap;
    struct URL_module *mod;

    if(m == NULL)
	return;
    url_add_module(tmdy_struct, m);
    va_start(ap, m);
    while((mod = va_arg(ap, struct URL_module *)) != NULL)
	url_add_module(tmdy_struct, mod);
}

static int url_init_nop(tmdy_struct_ex_t *tmdy_struct)
{
    /* Do nothing any more */
    return 1;
}

int url_check_type(tmdy_struct_ex_t *tmdy_struct, char *s)
{
    struct URL_module *m;

    for(m = url_mod_list; m; m = m->chain)
	if(m->type != URL_none_t && m->name_check && m->name_check(tmdy_struct, s))
	    return m->type;
    return -1;
}

URL url_open(tmdy_struct_ex_t *tmdy_struct, char *s)
{
    struct URL_module *m;

    for(m = url_mod_list; m; m = m->chain)
    {
#ifdef DEBUG
	printf("Check URL type=%d\n", m->type);
#endif /* DEBUG */
	if(m->type != URL_none_t && m->name_check && m->name_check(tmdy_struct, s))
	{
#ifdef DEBUG
	    printf("open url (type=%d, name=%s)\n", m->type, s);
#endif /* DEBUG */
	    if(m->url_init != url_init_nop)
	    {
		if(m->url_init && m->url_init(tmdy_struct) < 0)
		    return NULL;
		m->url_init = url_init_nop;
	    }

	    url_errno = URLERR_NONE;
	    errno = 0;
	    return m->url_open(tmdy_struct, s);
	}
    }

    url_errno = URLERR_NOURL;
    errno = ENOENT;
    return NULL;
}

long url_read(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n)
{
    if(n <= 0)
	return 0;
    url_errno = URLERR_NONE;
    errno = 0;
    if(url->nread >= url->readlimit) {
        url->eof = 1;
	return 0;
    }
    if(url->nread + n > url->readlimit)
	n = (long)(url->readlimit - url->nread);
    n = url->url_read(tmdy_struct, url, buff, n);
    if(n > 0)
	url->nread += n;
    return n;
}

long url_safe_read(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n)
{
    long i;
    if(n <= 0)
	return 0;

    do /* Ignore signal intruption */
    {
	errno = 0;
	i = url_read(tmdy_struct, url, buff, n);
    } while(i == -1 && errno == EINTR);
#if 0
    /* Already done in url_read!! */
    if(i > 0)
	url->nread += i;
#endif
    return i;
}

long url_nread(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n)
{
    long insize = 0;
    char *s = (char *)buff;

    do
    {
	long i;
	i = url_safe_read(tmdy_struct, url, s + insize, n - insize);
	if(i <= 0)
	{
	    if(insize == 0)
		return i;
	    break;
	}
	insize += i;
    } while(insize < n);

    return insize;
}

char *url_gets(tmdy_struct_ex_t *tmdy_struct, URL url, char *buff, int n)
{
    if(url->nread >= url->readlimit)
	return NULL;

    if(url->url_gets == NULL)
    {
	int maxlen, i, c;
	int newline = url_newline_code;

	maxlen = n - 1;
	if(maxlen == 0)
	    *buff = '\0';
	if(maxlen <= 0)
	    return buff;
	i = 0;

	do
	{
	    if((c = TMDY_ARC->url->url_getc(tmdy_struct, url)) == EOF)
		break;
	    buff[i++] = c;
	} while(c != newline && i < maxlen);

	if(i == 0)
	    return NULL; /* EOF */
	buff[i] = '\0';
	return buff;
    }

    url_errno = URLERR_NONE;
    errno = 0;

    if(url->nread + n > url->readlimit)
	n = (long)(url->readlimit - url->nread) + 1;

    buff = url->url_gets(tmdy_struct, url, buff, n);
    if(buff != NULL)
	url->nread += strlen(buff);
    return buff;
}

int url_readline(tmdy_struct_ex_t *tmdy_struct, URL url, char *buff, int n)
{
    int maxlen, i, c;

    maxlen = n - 1;
    if(maxlen == 0)
	*buff = '\0';
    if(maxlen <= 0)
	return 0;
    do
    {
	i = 0;
	do
	{
	    if((c = TMDY_ARC->url->url_getc(tmdy_struct, url)) == EOF)
		break;
	    buff[i++] = c;
	} while(c != '\r' && c != '\n' && i < maxlen);
	if(i == 0)
	    return 0; /* EOF */
    } while(i == 1 && (c == '\r' || c == '\n'));

    if(c == '\r' || c == '\n')
	i--;
    buff[i] = '\0';
    return i;
}

int url_fgetc(tmdy_struct_ex_t *tmdy_struct, URL url)
{
    if(url->nread >= url->readlimit)
	return EOF;

    url->nread++;
    if(url->url_fgetc == NULL)
    {
	unsigned char c;
	if(url_read(tmdy_struct, url, &c, 1) <= 0)
	    return EOF;
	return (int)c;
    }
    url_errno = URLERR_NONE;
    errno = 0;
    return url->url_fgetc(tmdy_struct, url);
}

long url_seek(tmdy_struct_ex_t *tmdy_struct, URL url, long offset, int whence)
{
    long pos, savelimit;

    if(url->url_seek == NULL)
    {
	if(whence == SEEK_CUR && offset >= 0)
	{
	    pos = url_tell(tmdy_struct, url);
	    if(offset == 0)
		return pos;
	    savelimit = (long)url->readlimit;
	    url->readlimit = URL_MAX_READLIMIT;
	    url_skip(tmdy_struct, url, offset);
	    url->readlimit = savelimit;
	    url->nread = 0;
	    return pos;
	}

	if(whence == SEEK_SET)
	{
	    pos = url_tell(tmdy_struct, url);
	    if(pos != -1 && pos <= offset)
	    {
		if(pos == offset)
		    return pos;
		savelimit = (long)url->readlimit;
		url->readlimit = URL_MAX_READLIMIT;
		url_skip(tmdy_struct, url, offset - pos);
		url->readlimit = savelimit;
		url->nread = 0;
		return pos;
	    }
	}

	url_errno = errno = EPERM;
	return -1;
    }
    url_errno = URLERR_NONE;
    errno = 0;
    url->nread = 0;
    return url->url_seek(tmdy_struct, url, offset, whence);
}

long url_tell(tmdy_struct_ex_t *tmdy_struct, URL url)
{
    url_errno = URLERR_NONE;
    errno = 0;
    if(url->url_tell == NULL)
	return (long)url->nread;
    return url->url_tell(tmdy_struct, url);
}

void url_skip(tmdy_struct_ex_t *tmdy_struct, URL url, long n)
{
    char tmp[BUFSIZ];

    if(url->url_seek != NULL)
    {
	long savenread;

	savenread = (long)url->nread;
	if(savenread >= url->readlimit)
	    return;
	if(savenread + n > url->readlimit)
	    n = (long)(url->readlimit - savenread);
	if(url->url_seek(tmdy_struct, url, n, SEEK_CUR) != -1)
	{
	    url->nread = savenread + n;
	    return;
	}
	url->nread = savenread;
    }

    while(n > 0)
    {
	long c;

	c = n;
	if(c > sizeof(tmp))
	    c = sizeof(tmp);
	c = url_read(tmdy_struct, url, tmp, c);
	if(c <= 0)
	    break;
	n -= c;
    }
}

void url_rewind(tmdy_struct_ex_t *tmdy_struct, URL url)
{
    if(url->url_seek != NULL)
	url->url_seek(tmdy_struct, url, 0, SEEK_SET);
    url->nread = 0;
}

void url_set_readlimit(tmdy_struct_ex_t *tmdy_struct, URL url, long readlimit)
{
    if(readlimit < 0)
	url->readlimit = URL_MAX_READLIMIT;
    else
	url->readlimit = (unsigned long)readlimit;
    url->nread = 0;
}

URL alloc_url(tmdy_struct_ex_t *tmdy_struct, int size)
{
    URL url;
#ifdef HAVE_SAFE_MALLOC
    url = (URL)tmdy_struct->common->safe_malloc(tmdy_struct, size);
    memset(url, 0, size);
#else
    url = (URL)malloc(size);
    if(url != NULL)
	memset(url, 0, size);
    else
	url_errno = errno;
#endif /* HAVE_SAFE_MALLOC */

    url->nread = 0;
    url->readlimit = URL_MAX_READLIMIT;
    url->eof = 0;
    return url;
}

void url_close(tmdy_struct_ex_t *tmdy_struct, URL url)
{
    int save_errno = errno;

    if(url == NULL)
    {
	fprintf(stderr, "URL stream structure is NULL?\n");
#ifdef ABORT_AT_FATAL
	abort();
#endif /* ABORT_AT_FATAL */
    }
    else if(url->url_close == NULL)
    {
	fprintf(stderr, "URL Error: Already URL is closed (type=%d)\n",
		url->type);
#ifdef ABORT_AT_FATAL
	abort();
#endif /* ABORT_AT_FATAL */
    }
    else
    {
	url->url_close(tmdy_struct, url);
#if 0
	url->url_close = NULL;
#endif /* unix */
    }
    errno = save_errno;
}

#if defined(TILD_SCHEME_ENABLE)
#include <pwd.h>
char *url_expand_home_dir(tmdy_struct_ex_t *tmdy_struct, char *fname)
{
    static char path[BUFSIZ];
    char *dir;
    int dirlen;

    if(fname[0] != '~')
	return fname;

    if(IS_PATH_SEP(fname[1])) /* ~/... */
    {
	fname++;
	if((dir = getenv("HOME")) == NULL)
	    if((dir = getenv("home")) == NULL)
		return fname;
    }
    else /* ~user/... */
    {
	struct passwd *pw;
	int i;

	fname++;
	for(i = 0; i < sizeof(path) - 1 && fname[i] && !IS_PATH_SEP(fname[i]); i++)
	    path[i] = fname[i];
	path[i] = '\0';
	if((pw = getpwnam(path)) == NULL)
	    return fname - 1;
	fname += i;
	dir = pw->pw_dir;
    }
    dirlen = strlen(dir);
    strncpy(path, dir, sizeof(path) - 1);
    if(sizeof(path) > dirlen)
	strncat(path, fname, sizeof(path) - dirlen - 1);
    path[sizeof(path) - 1] = '\0';
    return path;
}
char *url_unexpand_home_dir(tmdy_struct_ex_t *tmdy_struct, char *fname)
{
    static char path[BUFSIZ];
    char *dir, *p;
    int dirlen;

    if(!IS_PATH_SEP(fname[0]))
	return fname;

    if((dir = getenv("HOME")) == NULL)
	if((dir = getenv("home")) == NULL)
	    return fname;
    dirlen = strlen(dir);
    if(dirlen == 0 || dirlen >= sizeof(path) - 2)
	return fname;
    memcpy(path, dir, dirlen);
    if(!IS_PATH_SEP(path[dirlen - 1]))
	path[dirlen++] = PATH_SEP;

#ifndef __W32__
    if(strncmp(path, fname, dirlen) != 0)
#else
    if(strncasecmp(path, fname, dirlen) != 0)
#endif /* __W32__ */
	return fname;

    path[0] = '~';
    path[1] = '/';
    p = fname + dirlen;
    if(strlen(p) >= sizeof(path) - 3)
	return fname;
    path[2] = '\0';
    strcat(path, p);
    return path;
}
#else
char *url_expand_home_dir(tmdy_struct_ex_t *tmdy_struct, char *fname)
{
    return fname;
}
char *url_unexpand_home_dir(tmdy_struct_ex_t *tmdy_struct, char *fname)
{
    return fname;
}
#endif

static char *url_strerror_txt[] =
{
    "",				/* URLERR_NONE */
    "Unknown URL",		/* URLERR_NOURL */
    "Operation not permitted",	/* URLERR_OPERM */
    "Can't open a URL",		/* URLERR_CANTOPEN */
    "Invalid URL form",		/* URLERR_IURLF */
    "URL too long",		/* URLERR_URLTOOLONG */
    "No mail address",		/* URLERR_NOMAILADDR */
    ""
};

char *url_strerror(tmdy_struct_ex_t *tmdy_struct, int no)
{
    if(no <= URLERR_NONE)
	return strerror(no);
    if(no >= URLERR_MAXNO)
	return "Internal error";
    return url_strerror_txt[no - URLERR_NONE];
}

void *url_dump(tmdy_struct_ex_t *tmdy_struct, URL url, long nbytes, long *read_size)
{
    long allocated, offset, read_len;
    char *buff;

    if(read_size != NULL)
      *read_size = 0;
    if(nbytes == 0)
	return NULL;
    if(nbytes >= 0)
    {
	buff = (void *)tmdy_struct->common->safe_malloc(tmdy_struct, nbytes);
	if(nbytes == 0)
	    return buff;
	read_len = url_nread(tmdy_struct, url, buff, nbytes);
	if(read_size != NULL)
	  *read_size = read_len;
	if(read_len <= 0)
	{
	    free(buff);
	    return NULL;
	}
	return buff;
    }

    allocated = 1024;
    buff = (char *)tmdy_struct->common->safe_malloc(tmdy_struct, allocated);
    offset = 0;
    read_len = allocated;
    while((nbytes = url_read(tmdy_struct, url, buff + offset, read_len)) > 0)
    {
	offset += nbytes;
	read_len -= nbytes;
	if(offset == allocated)
	{
	    read_len = allocated;
	    allocated *= 2;
	    buff = (char *)tmdy_struct->common->safe_realloc(tmdy_struct, buff, allocated);
	}
    }
    if(offset == 0)
    {
	free(buff);
	return NULL;
    }
    if(read_size != NULL)
      *read_size = offset;
    return buff;
}



/* macros to functions */
#define url_eof_mac(url) URLm((url), eof)

#define url_getc_mac(tmdy_struct, url) \
    ((url)->nread >= (url)->readlimit ? ((url)->eof = 1, EOF) : \
     (url)->url_fgetc != NULL ? ((url)->nread++, (url)->url_fgetc(tmdy_struct, url)) : \
      url_fgetc(tmdy_struct, url))
      
int mc_url_eof(tmdy_struct_ex_t *tmdy_struct, URL url){
	int mc_buf;
	mc_buf=url_eof_mac(url);
	return mc_buf;
}
int mc_url_getc(tmdy_struct_ex_t *tmdy_struct,URL url){
	int mc_buf;
	mc_buf=url_getc_mac(tmdy_struct, url);
	return mc_buf;
}
/**** mutexed functions ***/
URL ts_url_open(tmdy_struct_ex_t *tmdy_struct, char *url_string){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_open(tmdy_struct, url_string);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
void ts_url_close(tmdy_struct_ex_t *tmdy_struct, URL url){
	timidity_mutex_lock(TMDY_ARC->url->busy);
	url_close(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
}
long ts_url_read(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n){
	long ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_read(tmdy_struct, url, buff, n);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
long ts_url_safe_read(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n){
	long ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_safe_read(tmdy_struct, url, buff, n);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
long ts_url_nread(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n){
	long ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_nread(tmdy_struct, url, buff, n);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
char *ts_url_gets(tmdy_struct_ex_t *tmdy_struct, URL url, char *buff, int n){
	char *ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_gets(tmdy_struct, url, buff, n);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
int ts_url_readline(tmdy_struct_ex_t *tmdy_struct, URL url, char *buff, int n){
	int ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_readline(tmdy_struct, url, buff, n);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
int ts_url_fgetc(tmdy_struct_ex_t *tmdy_struct, URL url){
	int ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_fgetc(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
long ts_url_seek(tmdy_struct_ex_t *tmdy_struct, URL url, long offset, int whence){
	long ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_seek(tmdy_struct, url, offset, whence);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
long ts_url_tell(tmdy_struct_ex_t *tmdy_struct, URL url){
	long ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_tell(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
void ts_url_skip(tmdy_struct_ex_t *tmdy_struct, URL url, long n){
	timidity_mutex_lock(TMDY_ARC->url->busy);
	url_skip(tmdy_struct, url, n);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
}
void ts_url_rewind(tmdy_struct_ex_t *tmdy_struct, URL url){
	timidity_mutex_lock(TMDY_ARC->url->busy);
	url_rewind(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
}
void *ts_url_dump(tmdy_struct_ex_t *tmdy_struct, URL url, long nbytes, long *real_read){
	void *ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_dump(tmdy_struct, url, nbytes, real_read);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
void ts_url_set_readlimit(tmdy_struct_ex_t *tmdy_struct, URL url, long readlimit){
	timidity_mutex_lock(TMDY_ARC->url->busy);
	url_set_readlimit(tmdy_struct, url, readlimit);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
}
char *ts_url_strerror(tmdy_struct_ex_t *tmdy_struct, int no){
	char *ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_strerror(tmdy_struct, no);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_alloc_url(tmdy_struct_ex_t *tmdy_struct, int size){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=alloc_url(tmdy_struct, size);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
int ts_url_check_type(tmdy_struct_ex_t *tmdy_struct, char *url_string){
	int ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=ts_url_check_type(tmdy_struct, url_string);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
char *ts_url_expand_home_dir(tmdy_struct_ex_t *tmdy_struct, char *filename){
	char *ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_expand_home_dir(tmdy_struct, filename);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
char *ts_url_unexpand_home_dir(tmdy_struct_ex_t *tmdy_struct, char *filename){
	char *ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_unexpand_home_dir(tmdy_struct, filename);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}

void ts_url_add_module(tmdy_struct_ex_t *tmdy_struct, struct URL_module *m){
	timidity_mutex_lock(TMDY_ARC->url->busy);
	url_add_module(tmdy_struct, m);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
}
void ts_url_add_modules(tmdy_struct_ex_t *tmdy_struct, struct URL_module *m, ...){
    va_list ap;
    struct URL_module *mod;

	timidity_mutex_lock(TMDY_ARC->url->busy);
	if(m == NULL){
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return;
	}
    url_add_module(tmdy_struct, m);
    va_start(ap, m);
    while((mod = va_arg(ap, struct URL_module *)) != NULL)
	url_add_module(tmdy_struct, mod);
	timidity_mutex_unlock(TMDY_ARC->url->busy);

}
URL ts_url_file_open(tmdy_struct_ex_t *tmdy_struct, char *filename){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_file_open(tmdy_struct, filename);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_dir_open(tmdy_struct_ex_t *tmdy_struct, char *directory_name){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_dir_open(tmdy_struct,directory_name);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_pipe_open(tmdy_struct_ex_t *tmdy_struct, char *command){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_pipe_open(tmdy_struct, command);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
#ifdef SUPPORT_SOCKET
URL ts_url_http_open(tmdy_struct_ex_t *tmdy_struct, char *url_string){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_http_open(tmdy_struct, url_string);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_ftp_open(tmdy_struct_ex_t *tmdy_struct, char *url_string){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_ftp_open(tmdy_struct, url_string);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_newsgroup_open(tmdy_struct_ex_t *tmdy_struct, char *url_string){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_newsgroup_open(tmdy_struct, url_string);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_news_open(tmdy_struct_ex_t *tmdy_struct, char *url_string){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_news_open(tmdy_struct, url_string);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
#endif

URL ts_url_mem_open(tmdy_struct_ex_t *tmdy_struct, char *memory, long memsiz, int autofree){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_mem_open(tmdy_struct, memory, memsiz, autofree);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_inflate_open(tmdy_struct_ex_t *tmdy_struct, URL instream, long compsize, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_inflate_open(tmdy_struct, instream, compsize, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_buff_open(tmdy_struct_ex_t *tmdy_struct, URL url, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_buff_open(tmdy_struct, url, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_cache_open(tmdy_struct_ex_t *tmdy_struct, URL url, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_cache_open(tmdy_struct, url, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
void ts_url_cache_detach(tmdy_struct_ex_t *tmdy_struct, URL url){
	timidity_mutex_lock(TMDY_ARC->url->busy);
	url_cache_detach(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
}
void ts_url_cache_disable(tmdy_struct_ex_t *tmdy_struct, URL url){
	timidity_mutex_lock(TMDY_ARC->url->busy);
	url_cache_disable(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
}
URL ts_url_uudecode_open(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_uudecode_open(tmdy_struct, reader, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_b64decode_open(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_b64decode_open(tmdy_struct, reader, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_hqxdecode_open(tmdy_struct_ex_t *tmdy_struct, URL reader, int dataonly, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_hqxdecode_open(tmdy_struct,reader, dataonly, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_qsdecode_open(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_qsdecode_open(tmdy_struct, reader, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
char *ts_url_dir_name(tmdy_struct_ex_t *tmdy_struct, URL url){
	char *ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_dir_name(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
#ifdef SUPPORT_SOCKET
URL ts_url_cgi_escape_open(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_cgi_escape_open(tmdy_struct, reader, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
URL ts_url_cgi_unescape_open(tmdy_struct_ex_t *tmdy_struct, URL reader, int autoclose){
	URL ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_cgi_unescape_open(tmdy_struct, reader, autoclose);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
char *ts_url_newsgroup_name(tmdy_struct_ex_t *tmdy_struct, URL url){
	char *ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_newsgroup_name(tmdy_struct, url);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
int ts_url_news_connection_cache(tmdy_struct_ex_t *tmdy_struct, int flag){
	int ts_buf;
	timidity_mutex_lock(TMDY_ARC->url->busy);
	ts_buf=url_news_connection_cache(tmdy_struct, flag);
	timidity_mutex_unlock(TMDY_ARC->url->busy);
	return ts_buf;
}
#endif

url_ex_t* new_url(tmdy_struct_ex_t *tmdy_struct){
	int i;
	url_ex_t* url_ex;

	url_ex=(url_ex_t *)malloc(sizeof(url_ex_t));
	
	timidity_mutex_init(url_ex->busy);
	
	/* macos to functions */
	url_ex->url_eof=mc_url_eof;
	url_ex->url_getc=mc_url_getc;
	
	url_ex->url_open=ts_url_open;
	url_ex->url_close=ts_url_close;
	url_ex->url_read=ts_url_read;
	url_ex->url_safe_read=ts_url_safe_read;
	url_ex->url_nread=ts_url_nread;
	url_ex->url_gets=ts_url_gets;
	url_ex->url_readline=ts_url_readline;
	url_ex->url_fgetc=ts_url_fgetc;
	url_ex->url_seek=ts_url_seek;
	url_ex->url_tell=ts_url_tell;
	url_ex->url_skip=ts_url_skip;
	url_ex->url_rewind=ts_url_rewind;
	url_ex->url_dump=ts_url_dump;
	url_ex->url_set_readlimit=ts_url_set_readlimit;
	url_ex->url_strerror=ts_url_strerror;
	url_ex->alloc_url=ts_alloc_url;
	url_ex->url_check_type=ts_url_check_type;
	url_ex->url_expand_home_dir=ts_url_expand_home_dir;
	url_ex->url_unexpand_home_dir=ts_url_unexpand_home_dir;

/* one variable */
	url_ex->url_errno_p=&url_errno;

	url_ex->url_add_module=ts_url_add_module;
	url_ex->url_add_modules=ts_url_add_modules;
	url_ex->url_file_open=ts_url_file_open;
#ifndef __MACOS__
	url_ex->url_dir_open=ts_url_dir_open;
#endif
#ifdef HAVE_POPEN
	url_ex->url_pipe_open=ts_url_pipe_open;
#endif
#ifdef SUPPORT_SOCKET
	url_ex->url_http_open=ts_url_http_open;
	url_ex->url_ftp_open=ts_url_ftp_open;
	url_ex->url_newsgroup_open=ts_url_newsgroup_open;
	url_ex->url_news_open=ts_url_news_open;
#endif


/* No URL_module */
	url_ex->url_mem_open=ts_url_mem_open;
	url_ex->url_inflate_open=ts_url_inflate_open;
	url_ex->url_buff_open=ts_url_buff_open;
	url_ex->url_cache_open=ts_url_cache_open;
	url_ex->url_cache_detach=ts_url_cache_detach;
	url_ex->url_cache_disable=ts_url_cache_disable;
	url_ex->url_uudecode_open=ts_url_uudecode_open;
	url_ex->url_b64decode_open=ts_url_b64decode_open;
	url_ex->url_hqxdecode_open=ts_url_hqxdecode_open;
	url_ex->url_qsdecode_open=ts_url_qsdecode_open;
	url_ex->url_dir_name=ts_url_dir_name;
#ifdef SUPPORT_SOCKET	
	url_ex->url_cgi_escape_open=ts_url_cgi_escape_open;
	url_ex->url_cgi_unescape_open=ts_url_cgi_unescape_open;
	url_ex->url_newsgroup_name=ts_url_newsgroup_name;
	url_ex->url_news_connection_cache=ts_url_news_connection_cache;
#endif
/* variables */
	url_ex->url_lib_version=url_lib_version;
	url_ex->user_mailaddr=user_mailaddr;
	url_ex->url_user_agent=url_user_agent;
#ifdef SUPPORT_SOCKET	
	url_ex->url_http_proxy_host=url_http_proxy_host;	
	url_ex->url_http_proxy_port_p=&url_http_proxy_port;
	url_ex->url_ftp_proxy_host=url_ftp_proxy_host;
	url_ex->url_ftp_proxy_port_p=&url_ftp_proxy_port;
#endif
	url_ex->url_newline_code_p=&url_newline_code;
	url_ex->uudecode_unquote_html_p=&uudecode_unquote_html;

	url_ex->URL_module_file_p=&URL_module_file;
#ifndef __MACOS__
	url_ex->URL_module_dir_p=&URL_module_dir;
#endif /* __MACOS__ */
#ifdef SUPPORT_SOCKET
	url_ex->URL_module_http_p=&URL_module_http;
	url_ex->URL_module_ftp_p=&URL_module_ftp;
	url_ex->URL_module_news_p=&URL_module_news;
	url_ex->URL_module_newsgroup_p=&URL_module_newsgroup;
#endif /* SUPPORT_SOCKET */
#if !defined(__MACOS__) && defined(HAVE_POPEN)
	url_ex->URL_module_pipe_p=&URL_module_pipe;
#endif /* HAVE_POPEN */	

	return url_ex;
}
void destroy_url(url_ex_t* url){
	timidity_mutex_destroy(url->busy);
	free(url);
}
