#include "HLTransfer.h"
#include "HLServer.h"
#include "HLClient.h"
#include "TCPSocket.h"
#include "BMutex.h"
#include <exception>
#include <string.h>
#include <algorithm>
#include "FileUtils.h"
#include "ServerLog.h"

#include <fcntl.h>

#if !defined(WIN32)
#include <signal.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <errno.h>
#endif // !WIN32

#if defined(_SENDFILE_) && !defined(_FREEBSD_)
#include <sys/sendfile.h>
#elif defined(_SENDFILE_) && defined(_FREEBSD_)
#include <sys/uio.h>
#endif

#if defined(_CARBON_)
#include <CoreServices/CoreServices.h>
#endif

using namespace FileUtils;

static u_int32_t fnv (u_int8_t *buf, u_int32_t len, u_int32_t h);
//static int sendfile_wrapper(int inFD, int inSock, unsigned int inFilePos, char *inBuf, int inSize);
static int sendfile_wrapper(FILE *send_file, int inSock, unsigned int inFilePos, char *inBuf, int inSize);



HLTransfer::HLTransfer(HLTransferSocket *inSocket)
	: mSocket(inSocket), mFD(0)
{
    mSocket->mTransfer = this;
}

HLTransfer::~HLTransfer()
{
	DEBUG_CALL(printf("entering hltransfer destructor\n"));
	// i lock around this instead of inside of it
	// because other threads need to call delete
	// inside sections of code that have already locked gServer->mLock
	// mainly the client destructor
	if (mInfo.valid())
	{
		DEBUG_CALL(printf("ending transfer\n"));
		if (mInfo.type == kDownloadTransfer || mInfo.type == kPreviewTransfer)
			gServer->mTransAM->EndDownload(this);
		else if (mInfo.type == kUploadTransfer)
			gServer->mTransAM->EndUpload(this);
	}
	
	if (mFD > 0)
	{
		DEBUG_CALL(printf("closing transfer file\n"));
		close(mFD);
		mFD = 0;
	}

	DEBUG_CALL(printf("hltransfer freeing mem...\n"); fflush(stdout));
	delete mSocket;
}

void HLTransfer::Cancel()
{
	DEBUG_CALL(printf("canceling transfer...\n"));
	if (mSocket->IsConnected())
	{
        mSocket->Shutdown();
	}
}

void HLTransfer::Run()
{
	try
	{
		DEBUG_CALL(printf("entering transfer thread\n"); fflush(stdout));
        if (mSocket->IsConnected())
        {
			{
				StMutexLock serverLock(gServer->mLock);
				gServer->mDataBase.AddTransfer(*this);
				
				if (mInfo.type == kDownloadTransfer || mInfo.type == kPreviewTransfer)
				{
					ServerLog::DownloadLog(*(gServer->GetClientForID(mInfo.user.ID())),
						mInfo.fileName.c_str());
				}
				else if (mInfo.type == kUploadTransfer)
				{
					ServerLog::UploadLog(*(gServer->GetClientForID(mInfo.user.ID())),
						mInfo.fileName.c_str());
				}
			} // unlock gServer lock before transfering data
            
            if (mInfo.type == kDownloadTransfer || mInfo.type == kPreviewTransfer)
                DoDownload();
            else if (mInfo.type == kUploadTransfer)
                DoUpload();
        }
	}
	catch (exception &exc)
	{
        DEBUG_CALL(printf("exception in transfer thread: %s\n", exc.what()); fflush(stdout));
		//BEGIN_ERROR();
		//LOG_ERROR("exception: %s\n", exc.what());
		//END_ERROR();
	}
    
    // several clients have broken download code that requires this hack
    // pc hotline 1.8.5, frogblast, and pitbull
    // UPDATE: this code is needed to fix frogblast
    if (mInfo.valid())
    {
        if ((mInfo.type == kDownloadTransfer || mInfo.type == kPreviewTransfer) && mSocket->IsConnected())
        {
            mSocket->SetLinger(true);
#if 1 // BROKEN CLIENT FIX
            { // locking scope
                StMutexLock lock(gServer->mLock);
                if (mInfo.valid())
                {
                    gServer->mTransAM->EndDownload(this);
					// invalidate the transfer info
                    mInfo.type = kInvalidTransfer;
                }
    
            }
            
            sleep(2);
#endif // BROKEN CLIENT FIX
        }
    }
    mSocket->Close();
    
	StMutexLock lock(gServer->mLock);
	DEBUG_CALL(printf("exiting hltransfer\n"); fflush(stdout));
	gServer->mDataBase.EndTransfer(*this);
    delete this;
}

void HLTransfer::DoDownload()
{
	// start the network speed calculator
	mSpeed.Start(mInfo.localDataForkSize - mInfo.remoteDataForkSize);
	// size the dynamic buffer for this transfer
	mBuf.Resize(CalculateBufferSize());
	
	u_int32_t bufPos = 0;
	// zero out the first 8k
	memset(mBuf.Ptr(), 0, kMinBufferSize);
    
    if (mInfo.type == kDownloadTransfer)
    {
        // this doesn't need to be send for a preview
        struct htxf_file_hdr *xfr_hdr = (struct htxf_file_hdr *)mBuf.Ptr(bufPos);
        xfr_hdr->magic = htonl(0x46494C50); // 'FILP'
        xfr_hdr->version = htonl(0x00010000);
        xfr_hdr->reserved[3] = htonl(0x00000002); // fork count, INFO counts as a fork
        xfr_hdr->info_magic = htonl(0x494E464F); // 'INFO'
        // if i was supporting file comments, i'd need to add the length of the file comment here also
        xfr_hdr->info_len = htonl(74 /* size of static stuff */ + mInfo.fileName.length());
        xfr_hdr->file_magic = htonl(0x414D4143); // 'AMAC'
#if defined(_CARBON_)
        xfr_hdr->ftype = htonl(UNKNOWN_TYPE);
        xfr_hdr->fcreator = htonl(UNKNOWN_TYPE);
        
        FSRef ref;
        if (PathToFSRef(mInfo.filePath, &ref))
        {
            FSCatalogInfo catInfo;
            if (FSGetCatalogInfo(&ref, kFSCatInfoFinderInfo, &catInfo, NULL, NULL, NULL) == noErr)
            {
                FInfo *finderInfo = (FInfo *)catInfo.finderInfo;
                xfr_hdr->ftype = htonl(finderInfo->fdType);
                xfr_hdr->fcreator = htonl(finderInfo->fdCreator);
            }
        }
#else
        xfr_hdr->ftype = fileTypeFromName(mInfo.fileName);
        xfr_hdr->fcreator = htonl(creatorForType(xfr_hdr->ftype));
        xfr_hdr->ftype = htonl(xfr_hdr->ftype);
#endif
        // dunno what this is either
        xfr_hdr->unknown2[1] = htonl(0x00000100);
        // i'm obviously not supporting this stuff
        xfr_hdr->creation_date_base = xfr_hdr->modify_date_base = htonl(0x0770);
        xfr_hdr->creation_date = xfr_hdr->modify_date = 0;
        bufPos += SIZEOF_HTXF_FILE_HDR;
        // file name
        {
            // do some funky conversion
            string tempFileName = mInfo.fileName;
            FileUtils::HomeToNetworkString(tempFileName);
            *((u_int16_t *)mBuf.Ptr(bufPos)) = htons((u_int16_t)tempFileName.length());
            bufPos += 2;
            strcpy(mBuf.Ptr(bufPos), tempFileName.c_str());
            bufPos += tempFileName.length();
        }
        // file comment
        *((u_int16_t *)mBuf.Ptr(bufPos)) = (u_int16_t)0;
        bufPos += 2;
    }
	u_int8_t theFork = 0;
	bool done = false;
	while (!done)
	{
        if (mInfo.type == kDownloadTransfer)
        {
            // preview doesn't need this
            struct htxf_fork_info *fork_info = (struct htxf_fork_info *)mBuf.Ptr(bufPos);
            
            if (theFork == 0) // data fork
            {
                fork_info->type = htonl(0x44415441); // 'DATA'
                fork_info->unknown[0] = 0;
                fork_info->unknown[1] = 0;
                fork_info->len = htonl(mInfo.localDataForkSize - mInfo.remoteDataForkSize);
            }
            else if (theFork == 1) // resource fork
            {
                // this isn't used
                fork_info->type = htonl(0x4D414352); // 'MACR'
                fork_info->unknown[0] = 0;
                fork_info->unknown[1] = 0;
                fork_info->len = 0;
            }
            
            bufPos += SIZEOF_HTXF_FORK_INFO;
            
            mSocket->SendBytes(mBuf.Ptr(), bufPos);
        }
		bufPos = 0;
		

			// open the file for reading and seek to starting point
			FILE * send_file;

		if (theFork == 0) // data fork
		{

			//2003/07/21 added by ortana.
#if defined(WIN32)

			
			if((send_file = fopen( mInfo.filePath.c_str() , "rb")) == NULL )
			{
				ServerLog::ErrorLog(__FILE__, __LINE__, "error opening download file: %s", strerror(errno));
				return ;
			}

#else

			// open the file for reading and seek to starting point
			mFD = open(mInfo.filePath.c_str(), O_RDONLY, O_BINARY);
			if (mFD < 0)
			{
				ServerLog::ErrorLog(__FILE__, __LINE__, "error opening download file: %s", strerror(errno));
				return;
			}

#endif//WIN32

		}
		else if (theFork == 1)
		{
			// resource fork unsupported
			done = true;
		}

		size_t readCount = 1;
        unsigned int filePos = mInfo.remoteDataForkSize;

				while (!done && readCount > 0)
				{
					mSpeed.StartSendTimer();


					//2003/07/21 added by ortana.
#if defined(WIN32)

					int sentSize = sendfile_wrapper( send_file ,mSocket->Descriptor(), 
																					 filePos, mBuf.Ptr(), mBuf.Size());
#else

					int sentSize = sendfile_wrapper(mFD, mSocket->Descriptor(), filePos, mBuf.Ptr(), mBuf.Size());

#endif//WIN32

					
					if (sentSize > 0)
					{
						filePos += sentSize;
						mSpeed.Update(sentSize);
						if (filePos >= mInfo.localDataForkSize)
							done = true;
						else
							mSpeed.LimitSpeed(sentSize, User().MaxBps());
					}
					else
					{
						done = true;
						DEBUG_CALL(printf("sendfile_wrapper error(%d): %s\n",
							sentSize, strerror(errno)); fflush(stdout));
					}
				} // end while()

		
				//2003/07/21 added by ortana.
#if defined(WIN32)

				if( send_file != NULL )
					fclose( send_file );

#else

		if (mFD > 0)
		{
			close(mFD);
			mFD = 0;
		}

#endif//WIN32
		
				//2003/07/30 added by ortana.
#ifdef WIN32
		try
		{
	char log_buff[MAX_PATH + 50];
	sprintf( log_buff , "%s\t%s\t%s\t\%s" , (gServer->GetClientForID(mInfo.user.ID()))->Address().c_str() ,
		(gServer->GetClientForID(mInfo.user.ID()))->User().Login().c_str(),
		(gServer->GetClientForID(mInfo.user.ID()))->User().Name().c_str() ,
		mInfo.fileName.c_str());

	ServerLog::OtherLog("end_D/L" , log_buff );
		}
		catch( ... ){}
#endif

		if (theFork == 0) // data fork
		{
			theFork = 1;
            // remove this return to have the server send the fork header for resource fork
            // don't forget to increment the fork count in the transfer header
            return; 
		}
		else if (theFork == 1) // resource fork
		{
            // resource fork unsupported
            done = true;
			return;
		}
	}
}

void HLTransfer::DoUpload()
{
	// start the network speed calculator
	mSpeed.Start(mInfo.transferSize);
	
	// size the dynamic buffer for this transfer
	mBuf.Resize(CalculateBufferSize());
	
	struct htxf_file_hdr xfer_hdr;
	struct htxf_fork_info fork_info;
	
	mSocket->RecvBytes(&xfer_hdr, 40);
	
	u_int32_t info_type;
	mSocket->RecvBytes(&info_type, 4);
	info_type = ntohl(info_type);
	
	u_int32_t file_type;
	mSocket->RecvBytes(&file_type, 4);
	file_type = ntohl(file_type);
	
	u_int32_t file_creator;
	mSocket->RecvBytes(&file_creator, 4);
	file_creator = ntohl(file_creator);
	
	// uhhh...whatever
	mSocket->RecvBytes(mBuf.Ptr(), 40);
	
	// creation, modification date, and 2 extra bytes
	mSocket->RecvBytes(mBuf.Ptr(), 18);
	
	u_int16_t fname_len;
	// file name length
	mSocket->RecvBytes(&fname_len, 2);
	fname_len = ntohs(fname_len);
	if (fname_len && fname_len < 256)
	{
		char fname[256];
		mSocket->RecvBytes(fname, fname_len);
		fname[fname_len] = '\0';
	}
	
	u_int16_t fcomment_len;
	mSocket->RecvBytes(&fcomment_len, 2);
	fcomment_len = ntohs(fcomment_len);
	if (fcomment_len && fcomment_len < 256)
	{
		char fcomment[256];
		mSocket->RecvBytes(fcomment, fcomment_len);
		fcomment[fcomment_len] = '\0';
	}
	
	u_int32_t forkCount(0); // safety messure for right now
	u_int32_t totalGotSize(130 + fcomment_len + fname_len);
	
	try
	{
		// update network speed calculator
		mSpeed.Update(totalGotSize);
		while (forkCount < 2)
		{
			u_int32_t forkGotSize = 0;
			
			memset(&fork_info, 0, sizeof(fork_info));
			mSocket->RecvBytes(&fork_info, SIZEOF_HTXF_FORK_INFO);
			
			// only update the totalGotSize for fork_info if it is a rsrc fork
			//totalGotSize += SIZEOF_HTXF_FORK_INFO;
			
			/*
			 * Clients that don't include the fork_info size in transfer size
			 * --------------------------------------------------------------
			 * Hotline 1.8.5 PC
			 * Hotline 1.9 PC
			 * Hotline Afterbirth PC
			 *
			 * Clients that do include fork_info size in transfer size
			 * --------------------------------------------------------------
			 * Heidrun 0.5 (sends hdr for rsrc fork even when size == 0)
			 * Hotline 1.9 Mac ""
			 * Hotline 1.8.5J Mac ""
			 */
			// Don't update the for fork_info size because it isn't
			// included in the transfer size for uploads
			// TODO: this needs extensive testing with all clients
			//mSpeed.Update(SIZEOF_HTXF_FORK_INFO);
			
			fork_info.type = ntohl(fork_info.type);
			fork_info.len = ntohl(fork_info.len);
			DEBUG_CALL(printf("got fork header. fork size: %u\n", fork_info.len));
			if (fork_info.type == 0x44415441) // 'DATA'
			{
				// open and create the file
				mFD = open(mInfo.filePath.c_str(), O_WRONLY | O_CREAT | O_BINARY,
					gServer->Config().filePermissions);
				if (mFD < 0)
				{
                    ServerLog::ErrorLog(__FILE__, __LINE__, "could not open upload file: %s", strerror(errno));
					return;
				}
				else
				{
#ifdef WIN32
#pragma message("File locking is not implemented on win32")
#else
					struct flock lk;
					
					lk.l_type = F_WRLCK;
					lk.l_start = 0;
					lk.l_whence = SEEK_SET;
					lk.l_len = 0;
					lk.l_pid = getpid();
				
					if (fcntl(mFD, F_SETLK, &lk) < 0)
					{
						ServerLog::ErrorLog(__FILE__, __LINE__,
							"could not lock upload file: %s", strerror(errno));
						return;
					}
#endif // WIN32
				}
				
				if (mInfo.localDataForkSize)
				{
					off_t offSet = lseek(mFD, mInfo.localDataForkSize, SEEK_SET);
					if (offSet < 0)
					{
                        ServerLog::ErrorLog(__FILE__, __LINE__, "error seeking upload file: %s", strerror(errno));
						return;
					}
				}
			}
			else if (fork_info.type == 0x4D414352) // 'MACR'
			{
				// see giant comment above. apparently some clients include
				// the fork info size in the transfer size ONLY for the resource fork's fork info
				totalGotSize += SIZEOF_HTXF_FORK_INFO;
				
				// NOT going to support resource forks
			}
			else
			{
				DEBUG_CALL(printf("unknown fork type: 0x%X\n", fork_info.type));
				return;
			}
			
			unsigned long writeCount;
			bool forkDone = false;
			// if the fork len is == 0 then we we're already done with it
			while (!forkDone && fork_info.len > 0)
			{
				int writeReturn;
				if ((fork_info.len - forkGotSize) >= mBuf.Size())
					writeCount = mBuf.Size();
				else	
					writeCount = (fork_info.len - forkGotSize);
				
				mSocket->RecvBytes(mBuf.Ptr(), writeCount);
				totalGotSize += writeCount;
				forkGotSize += writeCount;
				
				// might want to add some error checking here
				// write() returns error values
				if (mFD > 0)
				{
					writeReturn = write(mFD, mBuf.Ptr(), writeCount);
					if (writeReturn < 0 || ((unsigned long)writeReturn) != writeCount)
					{
						ServerLog::ErrorLog(__FILE__, __LINE__,
							"error writing upload file: %s", strerror(errno));
						return;
					}
				}
				
				// update network speed calculator
				mSpeed.Update(writeCount);
				
				if (forkGotSize == fork_info.len)
					forkDone = true;
				
				/*
				if (totalGotSize >= mInfo.transferSize)
				{
					// file transfer is complete
					// rename the file removing the .hpf and
					// possibly adding an extension based on file type
					postProcessUpload(mInfo.filePath, file_type);
					return;
				}
				*/
			} // while !forkDone
			
			forkCount++;
			if (mFD > 0)
			{
				DEBUG_CALL(printf("closing upload file\n"));
#ifndef WIN32
				fsync(mFD);
#endif // WIN32
				close(mFD);
				mFD = 0;
			}
			
			if (totalGotSize >= mInfo.transferSize)
			{
				int availableBytes = mSocket->GetAvailBytes();
				if (availableBytes == 0)
				{
					DEBUG_CALL(printf("totalGotSize: %u transferSize: %u no availableBytes, breaking....\n",
						totalGotSize, mInfo.transferSize));
					break;
				}
				else if (availableBytes > 0)
				{
					DEBUG_CALL(printf("totalGotSize: %u transferSize: %u availableBytes: %u\n",
						totalGotSize, mInfo.transferSize, availableBytes));
				}
			}
		}
	}
	catch (exception &exc)
	{
		DEBUG_CALL(printf("exception in upload thread: %s\n", exc.what()));
	}
	catch (...)
	{
		DEBUG_CALL(printf("unknown exception in upload thread\n"));
	}
	

		//2003/07/30 added by ortana.
#ifdef WIN32
	try
	{
	char log_buff[MAX_PATH + 50];
	sprintf( log_buff , "%s\t%s\t%s\t\%s" , (gServer->GetClientForID(mInfo.user.ID()))->Address().c_str() ,
		(gServer->GetClientForID(mInfo.user.ID()))->User().Login().c_str(),
		(gServer->GetClientForID(mInfo.user.ID()))->User().Name().c_str() ,
		mInfo.fileName.c_str());

	ServerLog::OtherLog("end_U/L" , log_buff );
	}
	catch( ... ){}
#endif
	
	DEBUG_CALL(printf("end upload totalGotSize: %u transferSize: %u\n",
		totalGotSize, mInfo.transferSize));
	
	if (totalGotSize >= mInfo.transferSize)
	{
		postProcessUpload(mInfo.filePath, file_type);
	}
}

void HLTransfer::GetStats(string &outStats, u_int16_t inQueuePosition)
{
	char statsBuf[1024];
	string timeString;
	string sizeString;
    unsigned int timeRemaining;
	
	if (inQueuePosition > 0)
	{
		formatTime(gServer->mTransAM->SecondsUntilDownloadSpotOpens(), timeString);
		formatSize((mInfo.localDataForkSize - mInfo.remoteDataForkSize), sizeString);
		snprintf(statsBuf, 1023, "%u) %s %s\rnext spot in %s\r\r",
			inQueuePosition, mInfo.fileName.c_str(), sizeString.c_str(), timeString.c_str());
	}
	else
	{
		string speedString;
		formatSize(mSpeed.TotalBytes(), sizeString);
		formatSize(mSpeed.BytesPerSecond(), speedString);
        
        timeRemaining = mSpeed.EstimatedTimeRemaining();
        if (timeRemaining == (unsigned int)-1)
        {
            if (TransferedBytes() == 0)
            {
                // no data has been transfered yet, so just say connecting...
                snprintf(statsBuf, 1023, "%s\r%s Connecting...\r\r",
                    mInfo.fileName.c_str(), sizeString.c_str());
            }
            else
            {
                // the transfer has stalled
                snprintf(statsBuf, 1023, "%s\rStalled at %.1f%% of %s\r\r",
                    mInfo.fileName.c_str(), mSpeed.PercentageComplete(),
                    sizeString.c_str());
            }
        }
        else
        {
            formatTime(timeRemaining, timeString);
            
            // this is for the get user info window transfer stats
            snprintf(statsBuf, 1023, "%s\r%.1f%% of %s at %s/sec\r%s remaining\r\r",
                mInfo.fileName.c_str(), mSpeed.PercentageComplete(),
                sizeString.c_str(), speedString.c_str(), timeString.c_str());
        }
	}
	outStats.assign(statsBuf);
}

void HLTransfer::SendKeepAlive()
{
    // sends an empty packet to avoid timeout in
    // masqueraded connections while download is queued
    char buf[2];
    int status = write(mSocket->Descriptor(), buf, 0);
    if (status == -1)
    {
        DEBUG_CALL(printf("error sending keep alive: %s\n", strerror(errno)); fflush(stdout));
    }
}

unsigned int HLTransfer::CalculateBufferSize()
{
	unsigned int transferSize = 0;
	unsigned int socketBufSize = 0;
	unsigned int bufferSize = kMinBufferSize;
	if (mInfo.type == kUploadTransfer)
	{
		socketBufSize = mSocket->GetRcvBufSize();
		transferSize = mInfo.transferSize;
		
		if ((socketBufSize / 4) > kMinBufferSize)
			bufferSize = (socketBufSize / 4);
	}
	else if (mInfo.type == kDownloadTransfer || mInfo.type == kPreviewTransfer)
	{
		socketBufSize = mSocket->GetSndBufSize();
		transferSize = (mInfo.localDataForkSize - mInfo.remoteDataForkSize);
		if (transferSize <= (128 * 1024))
			bufferSize = kMinBufferSize;
		else if ((socketBufSize / 2) > kMinBufferSize)
			bufferSize = (socketBufSize / 2);
	}
	
	DEBUG_CALL(printf("calculate buffer transfer size: %u buffer size: %u\n",
		transferSize, bufferSize));
	return bufferSize;
}

HLTransferManager::HLTransferManager()
 : mTotalUploads(0), mTotalDownloads(0),
 mTotalUploadBytes(0), mTotalDownloadBytes(0)
{
}

HLTransferManager::~HLTransferManager()
{
}

bool HLTransferManager::QueueTransferInfo(HLTransferInfo &inTransferInfo, bool &isLeech)
{
    // this new QueueTransferInfo never returns isLeech = true
	isLeech = false;
	// i check to make sure maxDownloads is != 0 here
	// there is no point in queuing if downloads are off
	// UPDATE: actually, there is a point (for testing)
    if ((inTransferInfo.type == kDownloadTransfer || inTransferInfo.type == kPreviewTransfer)
        /*&& gServer->mConf.maxDownloads*/)
	{
		// this is a download
		
		// check if the user has a download already in the queue
        // if so, we'll replace it with this new one. this is the smart
        // way of doing a server side queue that only allows a user to
        // have one download in the queue
		TransferList::iterator iter = mDownloadQueue.begin();
		while (iter != mDownloadQueue.end())
		{
			// make sure User() will be valid
			if ((*iter)->Info().valid())
			{
				// user already has a download in the server side queue
				if ((*iter)->User().ID() == inTransferInfo.user.ID())
				{
                    // replace the old download with a new one, right now they
                    // will lose their spot in the queue, but I'm going to
                    // work on a better way that would let them keep the same queue spot
                    (*iter)->Cancel();
                    mTransferInfoList.push_back(inTransferInfo);
					return true;
				}
			}
			iter++;
		}
		
		// don't allow them to queue a d/l if they have one running
		iter = mDownloadList.begin();
		while (iter != mDownloadList.end())
		{
			// make sure User() will be valid
			if ((*iter)->Info().valid())
			{
				if ((*iter)->User().ID() == inTransferInfo.user.ID())
					return false;
			}
			iter++;
		}
		
		mTransferInfoList.push_back(inTransferInfo);
		return true;
	}
	else if (inTransferInfo.type == kUploadTransfer)
	{
		// this is an upload, uploads aren't queued server side
		// so this is counting uploads in the transfer info list
		if (mUploadList.size() < gServer->mConf.maxUploads)
		{
			// take into account currently running uploads
			// and ones that are scheduled to start, but haven't connected yet
			u_int32_t counter(mUploadList.size());
			TransferInfoList::iterator iter = mTransferInfoList.begin();
			while (iter != mTransferInfoList.end())
			{
				if ((*iter).type == kUploadTransfer)
					counter++;
				iter++;
			}
			
			if (counter < gServer->mConf.maxUploads)
			{
				// there aren't max active uploads
				mTransferInfoList.push_back(inTransferInfo);
				return true;
			}
		}
	}
	
	// unknown inTransferInfo->type ?
	return false;
}

bool HLTransferManager::GetTransferInfo(u_int32_t inRef, HLTransferInfo &outInfo)
{
	TransferInfoList::iterator iter	= mTransferInfoList.begin();
	while (iter != mTransferInfoList.end())
	{
		if ((*iter).valid() && (*iter).ref == inRef)
		{
			outInfo = (*iter);
			mTransferInfoList.erase(iter);
			return true;
		}
		iter++;
	}
	return false;
}

unsigned int HLTransferManager::SecondsUntilDownloadSpotOpens()
{
	unsigned int shortTime = 0;
	if (!mDownloadList.empty())
	{
		TransferList::iterator iter = mDownloadList.begin();
		shortTime = (*iter)->mSpeed.EstimatedTimeRemaining();
		while (iter != mDownloadList.end())
		{
			unsigned int newTime = (*iter)->mSpeed.EstimatedTimeRemaining();
			if (newTime != 0 && newTime < shortTime)
			{
				shortTime = newTime;
			}
			iter++;
		}
	}
	return shortTime;
}

void HLTransferManager::CancelStalledTransfers()
{
	TransferList::iterator iter = mDownloadList.begin();
	while (iter != mDownloadList.end())
	{
		if ((*iter)->mSpeed.IsStalled())
		{
			DEBUG_CALL(printf("found and canceling stalled download\n"));
			(*iter)->Cancel();
		}
		iter++;
	}
	
	iter = mUploadList.begin();
	while (iter != mUploadList.end())
	{
		if ((*iter)->mSpeed.IsStalled())
		{
			DEBUG_CALL(printf("found and canceling stalled upload\n"));
			(*iter)->Cancel();
		}
		iter++;
	}
}

void HLTransferManager::KeepAliveQueuedTransfers()
{
    // called every 4 minutes by HLServer::Start
    TransferList::iterator iter = mDownloadQueue.begin();
    while (iter != mDownloadQueue.end())
	{
        (*iter)->SendKeepAlive();
        iter++;
    }	
}

void HLTransferManager::CancelTransfersForUser(HLUser &inUser)
{
	// clean up transfer info from unconnected transfers
	TransferInfoList::iterator infoIter = mTransferInfoList.begin();
	while (infoIter != mTransferInfoList.end())
	{
		if ((*infoIter).user.ID() == inUser.ID())
		{
			infoIter = mTransferInfoList.erase(infoIter);
		}
		else
			infoIter++;
	}
	
	// active downloads
	TransferList::iterator transIter = mDownloadList.begin();
	while (transIter != mDownloadList.end())
	{
		if ((*transIter)->User().ID() == inUser.ID())
			(*transIter)->Cancel();
		transIter++;
	}
	
	// server side queued downloads
	transIter = mDownloadQueue.begin();
	while (transIter != mDownloadQueue.end())
	{
		if ((*transIter)->User().ID() == inUser.ID())
			(*transIter)->Cancel();
		transIter++;
	}
	
	// active uploads
	transIter = mUploadList.begin();
	while (transIter != mUploadList.end())
	{
		if ((*transIter)->User().ID() == inUser.ID())
			(*transIter)->Cancel();
		transIter++;
	}
}

void HLTransferManager::BeginDownload(HLTransfer *inTransfer, bool &outQueued)
{
	if (mDownloadList.size() < gServer->mConf.maxDownloads)
	{
		// i'm not going to allow more than 1 download
		// by the same user at the same time
		TransferList::iterator iter = mDownloadList.begin();
		bool alreadyTransfering = false;
		while (iter != mDownloadList.end() && !alreadyTransfering)
		{
			// make sure User() will be valid
			if ((*iter)->Info().valid())
			{
				if ((*iter)->User().ID() == inTransfer->User().ID())
					alreadyTransfering = true;
			}
			iter++;
		}
		
		if (!alreadyTransfering)
		{
			// no transfers by user, go ahead
			outQueued = false;
			mDownloadList.push_back(inTransfer);
			return;
		}
	}
	
	// transfer needs to be queued server side
	outQueued = true;
	// put the download in the queue
	mDownloadQueue.push_back(inTransfer);
	
	// tell the client what number they are in the queue
	HLPacket queueUpdatePacket(HTLS_HDR_QUEUE_UPDATE, 0, 0);
	queueUpdatePacket.AddUInt32Object(HTLS_DATA_HTXF_REF, inTransfer->Ref());
	queueUpdatePacket.AddUInt32Object(HTLS_DATA_QUEUE_POSITION, mDownloadQueue.size());
	gServer->SendTo(queueUpdatePacket, inTransfer->User().ID());
}

void HLTransferManager::BeginUpload(HLTransfer *inTransfer)
{
	mUploadList.push_back(inTransfer);
}

void HLTransferManager::EndDownload(HLTransfer *inTransfer)
{
    if (inTransfer->TransferedBytes())
    {
        mTotalDownloads++;
        mTotalDownloadBytes += inTransfer->TransferedBytes();
    }
    
	TransferList::iterator iter = 
		find(mDownloadList.begin(), mDownloadList.end(), inTransfer);
	if (iter != mDownloadList.end())
	{
		mDownloadList.remove(inTransfer);
		if (mDownloadList.size() < gServer->mConf.maxDownloads &&
			mDownloadQueue.size())
		{
			// need to unqueue the next download
			HLTransfer *qTransfer = mDownloadQueue.front();
			mDownloadQueue.pop_front();
			mDownloadList.push_back(qTransfer);
			qTransfer->mSocket->StartTransferThread();
		}
	}
	else
	{
		// the download is probably still in the queue
		mDownloadQueue.remove(inTransfer);
	}
		
	u_int32_t counter = 1;
	iter = mDownloadQueue.begin();
	while (iter != mDownloadQueue.end())
	{
		if ((*iter)->mSocket->IsConnected())
		{
			HLPacket queueUpdatePacket(HTLS_HDR_QUEUE_UPDATE, 0, 0);
			queueUpdatePacket.AddUInt32Object(HTLS_DATA_HTXF_REF, (*iter)->Ref());
			queueUpdatePacket.AddUInt32Object(HTLS_DATA_QUEUE_POSITION, counter);
			gServer->SendTo(queueUpdatePacket, (*iter)->User().ID());
		}
		counter++;
		iter++;
	}
}

void HLTransferManager::EndUpload(HLTransfer *inTransfer)
{
    if (inTransfer->TransferedBytes())
    {
        mTotalUploads++;
        mTotalUploadBytes += inTransfer->TransferedBytes();
	}
    
	TransferList::iterator iter = 
		find(mUploadList.begin(), mUploadList.end(), inTransfer);
	if (iter != mUploadList.end())
	{
		mUploadList.remove(inTransfer);
	}
}

u_int32_t HLTransferManager::MakeRef(const HLTransferInfo &inInfo)
{
	return fnv((u_int8_t *)&inInfo, sizeof(inInfo), inInfo.user.LastRecvTime());
}

HLTransferSocket::HLTransferSocket(int inDescriptor, struct sockaddr_in* inRemoteAddress)
    : AsyncTCPSocket(inDescriptor, inRemoteAddress),
    mTransfer(NULL), mBufPos(0), mGotMagic(false), mThreadStarted(false)
{
    DEBUG_CALL(printf("transfer socket connected!\n"); fflush(stdout));
}

HLTransferSocket::~HLTransferSocket()
{
    Shutdown();
    Close();
}

void HLTransferSocket::OnRecv()
{
    // don't do anything after we've recved the magic
    if (mGotMagic)
        return;
    
    int recvSize = Recv(&mBuf[mBufPos], SIZEOF_HTXF_HDR - mBufPos);
    if (recvSize > 0)
        mBufPos += recvSize;
    
    if (mBufPos == SIZEOF_HTXF_HDR)
    {
        mGotMagic = true;
        struct htxf_hdr *hdr = (struct htxf_hdr *)mBuf;
        hdr->magic = ntohl(hdr->magic);
        hdr->ref = ntohl(hdr->ref);
        hdr->len = ntohl(hdr->len);
        if (hdr->magic == HTXF_MAGIC_INT && hdr->ref)
        {
            StMutexLock serverLock(gServer->mLock);
			// set the HLTransferInfo for the HLTransfer
            if (gServer->mTransAM->GetTransferInfo(hdr->ref, mTransfer->mInfo))
            {
                bool isQueued = false;
                if (mTransfer->mInfo.type == kDownloadTransfer ||
					mTransfer->mInfo.type == kPreviewTransfer)
                {
                    DEBUG_CALL(printf("found download transfer info\n"));
                    gServer->mTransAM->BeginDownload(mTransfer, isQueued);
                }
                else if (mTransfer->mInfo.type == kUploadTransfer)
                {
                    DEBUG_CALL(printf("found upload transfer info\n"));
                    
                    // this is for upload resumes...stupid
                    if (hdr->len != 0 && mTransfer->mInfo.transferSize != hdr->len)
                    {
                        mTransfer->mInfo.transferSize = hdr->len;
                        DEBUG_CALL(printf("new transfer size: %u\n",
							mTransfer->mInfo.transferSize));
                    }
                    
                    gServer->mTransAM->BeginUpload(mTransfer);
                }
                
                // if the download is queued on the server then i just
                // don't start the transfer thread and wait for any
                // OnClose events
                if (!isQueued)
                {
                    StartTransferThread();
                }
            }
            else
            {
                DEBUG_CALL(printf("couldn't find transfer in queue\n"));
                // couldn't find transfer info, just kill the transfer
                mTransfer->Cancel();
            }
        }
        else
        {
            DEBUG_CALL(printf("bad transfer magic\n"));
            // bad magic, kill the transfer
            mTransfer->Cancel();
        }
    }
}

void HLTransferSocket::OnClose()
{
    DEBUG_CALL(printf("HLTransferSocket::OnClose\n"));
    // run the transfer thread, which will just clean up the transfer
    // since it is already disconnected
    // not sure if i need to lock here, but just to be safe
	// UPDATE: i think this is definitely needed because
	// this needs to be reentrant safe
    StMutexLock serverLock(gServer->mLock);
    StartTransferThread();
}

void HLTransferSocket::StartTransferThread()
{
    if (!mThreadStarted && mTransfer != NULL)
    {
        mThreadStarted = true;
        // this will make socket blocking
        // and remove socket from selector group
        SetNonBlock(false);
        // start the transfer thread running
        mTransfer->Create();
    }
}

/*
 * Fowler/Noll/Vo (FNV) Hash
 */
static u_int32_t fnv (u_int8_t *buf, u_int32_t len, u_int32_t h)
{
	u_int8_t *p, *end;

	end = buf+len;
	for (p = buf; p < end; p++) {
		h *= 16777619;
		h ^= *p;
	}

	return h;
}


#if defined(WIN32)

//-----------------------------------------------------------------------//
////2003/07/21 added by ortana
//-----------------------------------------------------------------------//
static int sendfile_wrapper(FILE *send_file , int inSock ,unsigned int inFilePos ,
														char *inBuf,int inSize)
{
	if( fseek( send_file , inFilePos , SEEK_SET ) != 0 )
	{
		fclose( send_file );
		return (-1);
	}

	size_t read_value = -1;

	if( read_value = fread( inBuf , inSize , 1 , send_file ) < 0 )
	{
		fclose( send_file );
		return (-1);
	}


	size_t readCount = inSize;

	if( read_value == 0 )
	{
		readCount = ftell( send_file ) - inFilePos;
	}


	if (readCount > 0)
	{
		const char *bytes = (char *)inBuf;    
		int len = readCount;
		int sent = 0;

		while (len > 0)
		{
			sent = send(inSock, (const netdata_t*)bytes, len, MSG_NOSIGNAL);
			if (sent == -1)
				return sent;
			else
			{
				bytes += sent;
				len -= sent;
			}
		}
	}

	return readCount;

}

#endif

static int sendfile_wrapper(int inFD, int inSock, unsigned int inFilePos, char *inBuf, int inSize)
{
#if defined(_SENDFILE_) && !defined(_FREEBSD_)
// this is for Linux sendfile()
	(void)inBuf;
	unsigned int filePos = inFilePos;
	ssize_t sentSize = sendfile(inSock, inFD, (off_t *)&filePos, inSize);
	return (int)sentSize;
#elif defined(_SENDFILE_) && defined(_FREEBSD_)
// this is FreeBSD sendfile()
	(void)inBuf;
	off_t sentSize = 0;
	if (sendfile(inFD, inSock, inFilePos, inSize, NULL, &sentSize, 0) == 0)
		return sentSize;
	else
		return (-1);
#else
// platforms that don't have sendfile()
	if (lseek(inFD, inFilePos, SEEK_SET) == -1)
		return (-1);
	
#ifdef WIN32
// apparently _read is broken on Windows and returns 0
// even though it has successfully read data into the buffer
// so i have to generate the correct readCount value myself
	int fpos = _tell(inFD);
	size_t readCount = _read(inFD, inBuf, inSize);


	
	if (readCount == 0)
		readCount = (_tell(inFD) - fpos);

	//		DEBUG_CALL(printf("readCount (%d)\n",
	//				readCount); fflush(stdout));
#else
	size_t readCount = read(inFD, inBuf, inSize);
#endif

	if (readCount > 0)
	{
		const char *bytes = (char *)inBuf;    
		int len = readCount;
		int sent = 0;
		
		while (len > 0)
		{
			sent = send(inSock, (const netdata_t*)bytes, len, MSG_NOSIGNAL);
			if (sent == -1)
				return sent;
			else
			{
				bytes += sent;
				len -= sent;
			}
		}
	}
	
	return readCount;
#endif // SENDFILE
}
