#include "HLDataBase.h"
#include "HLServer.h"
#include "ServerLog.h"
#include "ServerConf.h"

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#if !defined(WIN32)
#include <dirent.h>
#include <unistd.h>
#include <sys/errno.h>
#include <arpa/inet.h>
#include <sys/param.h>
#endif // !WIN32

#include "HLProtocol.h"
#include "HLObject.h" // for HLEncrypt
#include "FileUtils.h"
#include "dirchar.h"
#include "HLUserData.h"

using namespace FileUtils;

static void createDefaultAccounts();

HLDataBase::HLDataBase()
{
	mNews.reserve(32000);
}

HLDataBase::~HLDataBase()
{
}

void HLDataBase::Connect(const string &/*inHost*/, const string &/*inUser*/,
												 const string &/*inPassword*/, const string &/*inDataBase*/)
{
}

void HLDataBase::Init()
{
	DEBUG_CALL(printf("reading news file\n"));

	off_t newsSize;

	//2003/07/27 added by ortana.
#ifdef WIN32

	FILE* fp1 = fopen( gServer->Config().newsPath.c_str() , "rb" );

	if( fp1 == NULL ){ 
		DEBUG_CALL(printf("error opening news file\n"));
		return ;

	}
	//get file size
	fseek( fp1, 0L, SEEK_END );
	newsSize = ftell( fp1 );
	fseek( fp1, 0L, SEEK_SET );
	char *newsBuf = new char[newsSize + 1];
	if( fread( newsBuf , newsSize , 1 , fp1) < 1 )
	{
		DEBUG_CALL(printf("error reading news file\n"));
		return ;
	}

	fclose( fp1 );
#else

	int fd;
	fd = open(gServer->Config().newsPath.c_str(), O_RDONLY);
	if (fd < 0)
	{
		DEBUG_CALL(printf("error opening news file\n"));
		return;
	}

	int err;
	newsSize = lseek(fd, 0, SEEK_END);
	lseek(fd, 0, SEEK_SET);
	char *newsBuf = new char[newsSize + 1];
	if ((err = read(fd, newsBuf, newsSize)) != newsSize)
	{
		DEBUG_CALL(printf("error reading news file: %d\n", err));
		return;
	}

	close(fd);

#endif//WIN32

	newsBuf[newsSize] = '\0';
	mNews.assign(newsBuf);
	fixLineEndings(mNews);


	delete[] newsBuf;




	DEBUG_CALL(printf("setting up accounts\n"));
	// here it should do a lstat on the directory
	// and create accounts if the accounts directory is empty maybe?
	if (mkdir(gServer->Config().accountsPath.c_str(),
		gServer->Config().dirPermissions) == 0)
	{
		createDefaultAccounts();
	}
	else if (errno != EEXIST)
	{
		ServerLog::ErrorLog(__FILE__, __LINE__,
			"Error creating account directory at %s : %s",
			gServer->Config().accountsPath.c_str(), strerror(errno));
	}
}

void HLDataBase::AddUser(HLUser &/*inUser*/, const string &/*inAddress*/)
{
}

void HLDataBase::UpdateUser(HLUser &/*inUser*/)
{
}

void HLDataBase::RemoveUser(HLUser &/*inUser*/)
{
}

void HLDataBase::AddHostBan(const string &inHost, const HLUser &/*inUser*/,
														const string &inReason, const u_int32_t inTimeout)
{
	HostBan hostBan;
	hostBan.host = inHost;
	hostBan.reason = inReason;
	hostBan.expire =  (inTimeout == 0 ? 0 : inTimeout + time(0));

	HostBanList::iterator iter = mHostBanList.begin();
	while (iter != mHostBanList.end())
	{
		if ((*iter).host == inHost)
		{
			if (hostBan.expire > (*iter).expire || hostBan.expire == 0)
			{
				(*iter).expire = hostBan.expire;
				(*iter).reason = hostBan.reason;
			}
			return;
		}
		iter++;
	}
	mHostBanList.push_back(hostBan);
}

void HLDataBase::AddNewsPost(HLUser &inUser, const string &inPost)
{
	string formattedPost;
	HLServer::FormatNewsPost(inUser.Name(), time(0), inPost, formattedPost);
	mNews.insert(0, formattedPost);

	int fd;
	fd = open(gServer->Config().newsPath.c_str(), O_RDWR | O_CREAT, gServer->Config().filePermissions);
	if (fd >= 0)
	{
		int err;
		if ((err = write(fd, mNews.c_str(), mNews.length())) != (int)mNews.length())
		{
			DEBUG_CALL(printf("error writing news file: %d\n", err));
		}
		close(fd);
	}
}

void HLDataBase::GetNewsFile(string &outNews)
{
	// need to truncate news to < ~32k
	outNews = mNews;
	if (outNews.length() > 32000)
	{
		DEBUG_CALL(printf("length: %u size: %u\n",
			(unsigned int)outNews.length(), (unsigned int)outNews.size()));
		outNews.resize(32000);
	}
}

//2003/07/30 added by ortana.
#ifdef WIN32
void HLDataBase::SetNewsFile(string inNews)
{
	if( inNews.length() > 32000)
	{
		DEBUG_CALL(printf("length: %u size: %u\n",
			(unsigned int)inNews.length(), (unsigned int)inNews.size()));
		inNews.resize(32000);
	}

	mNews = inNews;
}
#endif//WIN32

void HLDataBase::GetAgreement(string &outAgreement)
{
	DEBUG_CALL(printf("reading agreement file\n"));

	off_t agreementSize; 


	//2003/07/25 added by ortana.
#ifdef WIN32


	FILE* fp1 = fopen( gServer->Config().agreementPath.c_str() , "rb" );
	if( fp1 == NULL ){ return; }
	//get file size
	fseek( fp1, 0L, SEEK_END );
	agreementSize = ftell( fp1 );
	fseek( fp1, 0L, SEEK_SET );

	//typedef boost::shared_array<char> charPtr;
	//charPtr readBuf = charPtr( new char[size] );
	char * agreementBuf = new char[agreementSize + 1];
	if( fread( agreementBuf , agreementSize , 1 , fp1) < 1 )
	{
		return;
	}

	fclose( fp1 );
#else

	int fd = open(gServer->Config().agreementPath.c_str(), O_RDONLY);

	if( fd <= 0 )
	{
		DEBUG_CALL(printf("error opening agreement file\n"));
		return;
	}

	int err;
	off_t agreementSize = lseek(fd, 0, SEEK_END);
	lseek(fd, 0, SEEK_SET);
	char *agreementBuf = new char[agreementSize + 1];

	if ((err = read(fd, agreementBuf, agreementSize)) != agreementSize)
	{
		DEBUG_CALL(printf("error reading agreement file: %d\n", err));
		return;
	}
	close(fd);

#endif//WIN32

	agreementBuf[agreementSize] = '\0';
	outAgreement.assign(agreementBuf);
	fixLineEndings(outAgreement);

	delete[] agreementBuf;

}

bool HLDataBase::IsHostBanned(const string &inHost)
{
	HostBanList::iterator iter = mHostBanList.begin();
	time_t currentTime = time(0);
	bool isBanned = false;
	while (iter != mHostBanList.end())
	{
		if ((*iter).host == inHost)
		{
			if ((*iter).expire == 0 || (*iter).expire >= currentTime)
				isBanned = true;
			else if ((*iter).expire < currentTime)
			{
				// i maybe should add code here to remove old expired bans
				// do something like "delete from banlist_ip where expire < currentTime"
				DEBUG_CALL(printf("ban has expired for: %s\n", inHost.c_str()); fflush(stdout));
			}
			break;
		}
		iter++;
	}
	return isBanned;
}

bool HLDataBase::IsNameBanned(const string &/*inName*/)
{
	return false;
}

void HLDataBase::AddTransfer(HLTransfer &/*inTransfer*/)
{
}

void HLDataBase::EndTransfer(HLTransfer &/*inTransfer*/)
{
}

void HLDataBase::CreateAccount(HLAccount &inAccount)
{
	HLUserData newUserData(inAccount.Login());
	newUserData.SetName(inAccount.Name());
	newUserData.SetPassword(inAccount.Password());
	newUserData.SetAccessBits(inAccount.Access());
	newUserData.WriteTo(gServer->Config().accountsPath, true);
}

bool HLDataBase::FetchAccount(HLAccount &ioAccount, bool inVerifyPassword)
{
	HLUserData userData(ioAccount.Login());
	if (userData.ReadFrom(gServer->Config().accountsPath))
	{
		if (inVerifyPassword ? userData.Password() == ioAccount.Password() : true)
		{
			ioAccount.SetName(userData.Name());
			if (userData.RootPath() != "")
				ioAccount.SetFilesPath(userData.RootPath());
			else
				ioAccount.SetFilesPath(gServer->Config().rootPath);
			ioAccount.SetAccess(userData.AccessBits());
			ioAccount.SetMaxBps(userData.MaxBps());
			ioAccount.SetPassword(userData.Password());
			return true;
		}
	}
	return false;
}

void HLDataBase::ListAccounts(AccountList &outAccountList)
{
#ifdef WIN32
#pragma message("ListAccounts is not implemented on win32")
#else
	DIR *dir;
	struct dirent *de;
	struct stat sb;

	dir = opendir(gServer->Config().accountsPath.c_str());
	if (dir == NULL)
	{
		return;
	}

	while ((de = readdir(dir)) != NULL)
	{
		string fileName(de->d_name);
		if (fileName[0] != '.')
		{
			string filePath(gServer->Config().accountsPath);
			filePath.append(fileName);

			if (lstat(filePath.c_str(), &sb) == 0)
			{
				if (S_ISDIR(sb.st_mode))
				{
					HLAccount account;
					account.SetLogin(fileName);
					if (FetchAccount(account, false))
					{
						outAccountList.push_back(account);
					}
				}
			}
		}
	}
	closedir(dir);
#endif // WIN32
}

void HLDataBase::DeleteAccount(const string &inLogin)
{
	HLUserData deleteUserData(inLogin);
	deleteUserData.DeleteFrom(gServer->Config().accountsPath);
}

void HLDataBase::ModifyAccount(HLAccount &inAccount, bool inChangePassword)
{
	HLUserData modUserData(inAccount.Login());
	if (modUserData.ReadFrom(gServer->Config().accountsPath))
	{
		modUserData.SetName(inAccount.Name());
		modUserData.SetAccessBits(inAccount.Access());
		if (inChangePassword)
			modUserData.SetPassword(inAccount.Password());
		modUserData.WriteTo(gServer->Config().accountsPath, false);
	}
}

static void createDefaultAccounts()
{
	// create an admin account and a guest account
	struct fake_access_bits
	{
		u_int32_t hi, lo;
	};
	struct fake_access_bits accessBits;

	accessBits.hi = htonl(0xfff3cfef);
	accessBits.lo = htonl(0xff800000);
	HLUserData adminUserData("admin");
	adminUserData.SetName("Administrator");
	adminUserData.SetPassword("admin");
	adminUserData.SetAccessBits(*((struct hl_access_bits *)&accessBits));
	adminUserData.WriteTo(gServer->Config().accountsPath, true);

	accessBits.hi = htonl(0x60600C20);
	accessBits.lo = htonl(0x00000000);
	HLUserData guestUserData("guest");
	guestUserData.SetName("Guest");
	guestUserData.SetPassword("");
	guestUserData.SetAccessBits(*((struct hl_access_bits *)&accessBits));
	guestUserData.WriteTo(gServer->Config().accountsPath, true);
}

