#include "HLClient.h"
#include "HLChat.h"
#include "TCPSocket.h"
#include "HLProtocol.h"
#include <algorithm>
#include <time.h>
#include "FileUtils.h"
#include "ServerLog.h"

#if !defined(WIN32)
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#else
#include <imagehlp.h>
#endif

#if defined(CONFIG_THREADEDNEWS)
    #include "NewsUtils.h"
#endif // CONFIG_THREADEDNEWS

const string kTErrorGeneric = "Task Error";
const string kTErrorEmpty = "";

using namespace FileUtils;

#if defined(CONFIG_LINKING)
HLClient::HLClient(const string &inName, u_int32_t inGlobalID, u_int16_t inIcon,
	u_int16_t inStatus, HLAccount &inAccount)
	: mSocket(NULL), mVersion(0), mLastPacketTime(0),
    mLeechCounter(0), mFloodCounter(0), mListedFiles(false)
{
	// constructor for remote users
	mUser.LoggedIn(true);
	mUser.SetLoginTime();
    mUser.SetName(inName);
	mUser.SetGlobalID(inGlobalID);
	mUser.SetIcon(inIcon);
	(void)inStatus;
	//mUser.SetStatus(inStatus);
	mUser.SetAccount(inAccount);
	mUser.SetID(gServer->CreateNewUserID());
	
    // set the base times for flooding, idling, and connection timeout
    mUser.SetLastRecvTime(time(0));
    mLastPacketTime = mUser.LastRecvTime();
	
    gServer->mClientList.push_back(this);
	
	mUser.SetLoginTime();
	HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
	userChangePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
	userChangePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
	userChangePacket.AddUInt16Object(HTLS_DATA_ICON, mUser.Icon());
	userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, mUser.Status());
	gServer->SendToOthers(userChangePacket, *this);
}
#endif

HLClient::HLClient(HLSocket *inSocket)
	: mSocket(inSocket), mVersion(0), mLastPacketTime(0),
    mLeechCounter(0), mFloodCounter(0), mListedFiles(false)
{
    mSocket->SetCallback(this);
	mUser.SetID(gServer->CreateNewUserID());
	
    // set the base times for flooding, idling, and connection timeout
    mUser.SetLastRecvTime(time(0));
    mLastPacketTime = mUser.LastRecvTime();
	
    gServer->mClientList.push_back(this);
	//DEBUG_CALL(printf("adding client id: %u to the client list\n", mUser.ID()));
}	

HLClient::~HLClient()
{
	delete mSocket;
	mSocket = NULL;
}

void HLClient::NameLookup(const string &inHost, const string &inName)
{
	(void)inHost;
	DEBUG_CALL(printf("name lookup done\n"));
	mHostname = inName;


	//2003/08/01 added by ortana,
#ifdef WIN32
	if( gServer->getHLBan().isBan( mHostname.c_str() ) )
	{
		this->Disconnect();
		DEBUG_CALL(printf("host is banned\n"));
	}
#endif//WIN32
}

void HLClient::OnClose()
{
    StMutexLock lock(gServer->mLock);
		
		//2003/07/30 added by ortana.
#ifdef WIN32
		char log_buff[MAX_PATH + 50];
		sprintf( log_buff , "%s\t%s\t%s" , this->Address().c_str() , 
			this->User().Login().c_str(), this->User().Name().c_str());

		ServerLog::OtherLog( "logout" , log_buff );
#endif
		
    DEBUG_CALL(printf("HLClient::OnClose()\n"); fflush(stdout));
	Cleanup();
}

void HLClient::Cleanup()
{
	ChatList::iterator iter = gServer->mChatList.begin();
	HLChat *partChat;
	while (iter != gServer->mChatList.end())
	{
		if ((*iter)->IsClientInChat(*this))
		{
			partChat = (*iter);
			iter++;
			HLPacket chatUserPartPacket(HTLS_HDR_CHAT_USER_PART, 0, 0);
			chatUserPartPacket.AddUInt32Object(HTLS_DATA_CHAT_ID, ChatPtrToChatID(partChat));
			chatUserPartPacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
			partChat->RemoveClient(*this);
			partChat->SendToClients(chatUserPartPacket);
			if (partChat->IsEmpty())
				delete partChat;
		}
		else
			iter++;
	}
	
	if (mUser.IsLoggedIn())
	{
		mUser.LoggedIn(false);
        
        if (mUser.LoginTime())
        {
            HLPacket userLeavePacket(HTLS_HDR_USER_PART, 0, 0);
            userLeavePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
#if defined(CONFIG_LINKING)
			if (mSocket)
				gServer->Link().LinkUserLeave(*this);
#endif
			gServer->SendToAll(userLeavePacket);
        }
		
		if (mSocket)
		{
			// remove user from database
			gServer->mDataBase.RemoveUser(mUser);
			
			// transfer clean up
			gServer->mTransAM->CancelTransfersForUser(mUser);
		}
	}
}

void HLClient::Disconnect()
{
	if (mSocket)
		mSocket->Close();
}

void HLClient::Ban(const HLUser &inUser, const string &inReason, const u_int32_t inTimeout)
{
	if (mSocket)
	{
		gServer->mDataBase.AddHostBan(mSocket->RemoteHostName(), inUser, inReason, inTimeout);
		
		if (mUser.IsLoggedIn() && mUser.LoginTime() && !inReason.empty())
		{
			HLPacket chatKickPacket(HTLS_HDR_CHAT, 0, 0);

			//2003/07/30 added by ortana.
#ifdef _JAPAN_
			string chatString("\r               <<");
			chatString.append(mUser.Name());
			chatString.append(" ͈ȉ̗RɂBAN܂B : ");
			chatString.append(inReason);
			chatString.append(">>");
#else
			string chatString("\r<");
			chatString.append(mUser.Name());
			chatString.append(" was banned for ");
			chatString.append(inReason);
			chatString.append(">");
#endif//_JAPAN_
			chatKickPacket.AddStringObject(HTLS_DATA_CHAT, chatString);
			gServer->SendToOthers(chatKickPacket, *this);
		}
		
		Disconnect();
	}
	else
	{
		DEBUG_CALL(printf("attempt to ban remote user\n"));
	}
}

u_int32_t HLClient::RemoteHost()
{
	if (mSocket)
		return mSocket->RemoteHost();
	else
		return 0;
}

const string HLClient::Address() const
{
	if (mSocket)
		return mSocket->RemoteHostName();
	else
		return "remote";
}

void HLClient::SendKeepAlive()
{
    // sends an empty packet to avoid timeout in
    // masqueraded connections while download is queued
    if (mSocket)
	{
		char buf[2];
		int status = ::send(mSocket->Descriptor(), buf, 0, 0);
		if (status == -1)
		{
			DEBUG_CALL(printf("error sending keep alive: %s\n", strerror(errno)); fflush(stdout));
		}
	}
}

void HLClient::SendChatGreeting()
{
	/* removed by ortana.
    time_t currentTime = time(0);
    struct tm tm;
    char timeBuf[256];
    localtime_r(&currentTime, &tm);
	// Win32 has no %r - gah...
    //strftime(timeBuf, 255, " %A %m/%d/%Y %r", &tm);
	// try " %x %X"
	strftime(timeBuf, 255, " %A %m/%d/%Y %I:%M:%S %p", &tm);
	HLPacket serverTimePacket(HTLS_HDR_CHAT, 0, 0);
    string serverTimeChat("\r    Local Time:");
    serverTimeChat.append(timeBuf);
    serverTimeChat.append("\r       Uptime: ");
    string formattedTime;
    formatTime(gServer->Uptime(currentTime), formattedTime);
    serverTimeChat.append(formattedTime);
    serverTimePacket.AddStringObject(HTLS_DATA_CHAT, serverTimeChat);
    SendPacket(serverTimePacket);
	*/
}

void HLClient::SendSelfInfo(bool inFakeFilePrivs)
{
	HLPacket accessPacket(HTLS_HDR_USER_SELFINFO, 0, 0);
	accessPacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
	struct hl_access_bits userAccess = mUser.Access();
	userAccess.get_user_info = true; // everyone can get info on themselves
	
	if (inFakeFilePrivs)
	{
		// this is for telling HLclient 1.9 to enable the buttons in
		// the file window so a user can manage files in their drop box
		userAccess.rename_files = true;
		userAccess.move_files = true;
		userAccess.create_folders = true;
		userAccess.delete_folders = true;
		userAccess.rename_folders = true;
		userAccess.move_folders = true;
		userAccess.comment_files = true;
		userAccess.comment_folders = true;
		userAccess.make_aliases = true;
	}
	
	accessPacket.AddObject(HTLS_DATA_ACCESS,
		sizeof(struct hl_access_bits), (char *)&userAccess);
	SendPacket(accessPacket);
}

void HLClient::SendTaskError(const string &inError, u_int32_t inID)
{
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inID, 1);
	taskErrorPacket.AddStringObject(HTLS_DATA_TASKERROR, inError);
	SendPacket(taskErrorPacket);
}

void HLClient::HandlePacket(HLPacket &inPacket)
{
    StMutexLock lock(gServer->mLock);
    // handle packet
    time_t currentTime = time(0);
    if ((currentTime - mLastPacketTime) < FLOOD_INTERVAL)
        mFloodCounter++;
    else
        mFloodCounter = 0;
    
    if (!mUser.Access().cant_be_disconnected && mFloodCounter >= FLOOD_LIMIT)
    {
        gServer->AddClientToKickList(*this);
        if (IsConnected()) // make sure they didn't just get banned for strike rule
		{
			Disconnect();
			HLPacket chatKickPacket(HTLS_HDR_CHAT, 0, 0);

			//2003/07/30 addded by ortana.
#ifdef _JAPAN_
			string chatString("\r               <<");
			chatString.append(mUser.Name());
			chatString.append(" flooding attackׁAؒf܂B>>");
#else
			string chatString("\r<");
			chatString.append(mUser.Name());
			chatString.append(" was kicked for flooding>");
#endif//_JAPAN_
			chatKickPacket.AddStringObject(HTLS_DATA_CHAT, chatString);
			gServer->SendToOthers(chatKickPacket, *this);
		}
    }
    else
    {
		if (mUser.IsLoggedIn() && mUser.LoginTime() > 0)
		{
			// hotline 1.8.5 sends user getlist packets at a regular interval
			// which prevents them from going idle, i had to add the new variable
			// lastPacketTime to keep the packet time seperate from the idle timer
			// so i could prevent flooding of getuserlist requests (thanks kelly)
			// UPDATE: i'm pretty sure it only does this when it thinks the server
			// doesn't support the PING packet type, which is if it uses 1.2 login
			// instead of the newer 1.5+ login sequence.
			// UPDATE2: Also do a check for Frogblast 1.2.x, which sends get user info
			// packets as a keep-alive, except it uses 4 bytes for userID instead of 2
			if (inPacket.GetType() != HTLC_HDR_USER_GETLIST &&
				inPacket.GetType() != HTLC_HDR_USER_GETINFO)
			{
				mUser.SetLastRecvTime(mLastPacketTime);
				if (mUser.Status() & IDLE_MASK)
				{
					mUser.SetStatus(mUser.Status() ^ IDLE_MASK);
					// optimization - no need to send out two user change packets
					if (inPacket.GetType() != HTLC_HDR_USER_CHANGE)
					{
						HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
						userChangePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
						userChangePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
						userChangePacket.AddUInt16Object(HTLS_DATA_ICON, mUser.Icon());
						userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, mUser.Status());
						gServer->SendToAll(userChangePacket);
					}
				}
			}
		}
		else
			mUser.SetLastRecvTime(mLastPacketTime);

		ProcessPacket(inPacket);
        mLastPacketTime = currentTime;
	}
}

void HLClient::ProcessPacket(HLPacket &inPacket)
{
	// i check login time here because hl1.8.5 isn't fully logged in until
	// the client appears in the userlist when it sends the username
	if (mUser.IsLoggedIn() && mUser.LoginTime() > 0)
	{
		switch (inPacket.GetType())
		{
			case HTLC_HDR_NEWS_GETFILE:
				HandleNewsGetFile(inPacket);
				break;
				
			case HTLC_HDR_USER_GETLIST:
				HandleUserGetList(inPacket);
				break;
			
			case HTLC_HDR_USER_CHANGE:
				HandleUserChange(inPacket);
				break;
			
			case HTLC_HDR_CHAT:
				if (mUser.Access().send_chat)
					HandleChat(inPacket);
				break;
			
			case HTLC_HDR_CHAT_SUBJECT:
				if (mUser.Access().send_chat)
					HandleChatSubject(inPacket);
				break;
			
			case HTLC_HDR_MSG:
				HandleMessage(inPacket);
				break;
			
			case HTLC_HDR_CHAT_CREATE:
				HandleChatCreate(inPacket);
				break;
			
			case HTLC_HDR_CHAT_INVITE:
				HandleChatInvite(inPacket);
				break;
			
			case HTLC_HDR_CHAT_DECLINE:
				HandleChatDecline(inPacket);
				break;
			
			case HTLC_HDR_CHAT_JOIN:
				HandleChatJoin(inPacket);
				break;
			
			case HTLC_HDR_CHAT_PART:
				HandleChatPart(inPacket);
				break;
			
			case HTLC_HDR_NEWS_POST:
				HandleNewsPost(inPacket);
				break;
			
			case HTLC_HDR_USER_KICK:
				HandleUserKick(inPacket);
				break;
			
			case HTLC_HDR_USER_GETINFO:
				HandleUserGetInfo(inPacket);
				break;

            case HTLC_HDR_ACCOUNT_CREATE:
                HandleAccountCreate(inPacket);
				break;
                
			case HTLC_HDR_ACCOUNT_DELETE:
                HandleAccountDelete(inPacket);
				break;
                
			case HTLC_HDR_ACCOUNT_READ:
                HandleAccountRead(inPacket);
				break;
                
			case HTLC_HDR_ACCOUNT_MODIFY:
				HandleAccountModify(inPacket);
				break;
			
			case HTLC_HDR_ACCOUNT_LIST:
				HandleAccountList(inPacket);
				break;
			
			case HTLC_HDR_ACCOUNT_LIST_MODIFY:
				HandleAccountListModify(inPacket);
				break;

			case HTLC_HDR_FILE_LIST:
				HandleFileList(inPacket);
				break;
			
			case HTLC_HDR_FILE_GET:
				HandleFileGet(inPacket);
				break;
			
			case HTLC_HDR_FILE_PUT:
				HandleFilePut(inPacket);
				break;
				
			case HTLC_HDR_FILE_DELETE:
				HandleFileDelete(inPacket);
				break;
			
			case HTLC_HDR_FILE_MKDIR:
				HandleFileMkDir(inPacket);
				break;
			
			case HTLC_HDR_FILE_GETINFO:
				HandleFileGetInfo(inPacket);
				break;
				
			case HTLC_HDR_FILE_SETINFO:
				HandleFileSetInfo(inPacket);
				break;
				
			case HTLC_HDR_FILE_MOVE:
				HandleFileMove(inPacket);
				break;
				
			case HTLC_HDR_FILE_SYMLINK:
				HandleFileSymLink(inPacket);
				break;
            
            case HTLC_HDR_PING:
                HandlePing(inPacket);
                break;
            
            case HTLC_HDR_MSG_BROADCAST:
                HandleBroadcast(inPacket);
                break;
			
			case HTLC_HDR_XFER_STOP:
				HandleTransferStop(inPacket);
				break;
			
			case HTLC_HDR_FOLDER_GET:
				HandleFolderGet(inPacket);
				break;
			
			case HTLC_HDR_FOLDER_PUT:
				HandleFolderPut(inPacket);
				break;

#if defined(CONFIG_THREADEDNEWS)
			case HTLC_HDR_NEWS_LISTDIR:
                HandleNewsDirList(inPacket);
                break;
				
			case HTLC_HDR_NEWS_LISTCATEGORY:
				HandleNewsCatList(inPacket);
				break;
				
			case HTLC_HDR_NEWS_GETTHREAD:
				HandleNewsGetThread(inPacket);
				break;
				
			case HTLC_HDR_NEWS_MKDIR:
				HandleNewsMkDir(inPacket);
				break;
				
			case HTLC_HDR_NEWS_MKCATEGORY:
				HandleNewsMkCat(inPacket);
				break;
				
			case HTLC_HDR_NEWS_DELETECATDIR:
				HandleNewsDeleteCatDir(inPacket);
				break;
				
			case HTLC_HDR_NEWS_POSTTHREAD:
				HandleNewsPostThread(inPacket);
				break;
#endif // CONFIG_THREADEDNEWS
                
            default:
				DEBUG_CALL(printf("unhandled packet type: 0x%X\n", inPacket.GetType()));
				break;
		}
	}
	else
	{
		// these are the only packet types i will process if the user has not logged in
		switch (inPacket.GetType())
		{
			case HTLC_HDR_LOGIN:
                if (!mUser.IsLoggedIn())
                    HandleLogin(inPacket);
                else
                {
                    DEBUG_CALL(printf("user %s tried to login twice\n",
                        mUser.Name().c_str()); fflush(stdout));
                }
				break;

#if defined(CONFIG_THREADEDNEWS)
            case HTLC_HDR_AGREEMENTAGREE:
                if (mUser.IsLoggedIn())
                    HandleAgreementAgree(inPacket);
                break;
#endif // CONFIG_THREADEDNEWS

			// this is for 1.8.5 clients
			case HTLC_HDR_USER_CHANGE:
				if (mUser.IsLoggedIn())
					HandleUserChange(inPacket);
				break;

			// sometimes 1.8.5+ sends a userlist request before logging in
			case HTLC_HDR_USER_GETLIST:
				HandleUserGetList(inPacket);
				break;

            default:
				DEBUG_CALL(printf("out of order packet dropped: 0x%X\n", inPacket.GetType()));
				// disconnect if any other type of packet is recv'd
				//Disconnect();
				break;
		}
	}
}

void HLClient::HandleLogin(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	bool hopeLogin = false;
	string accountPassword;
    string accountLogin;
    string userName;
				
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_LOGIN:
				if ((*iter)->GetSize() && (*iter)->GetData()[0] != 0)
				{
					(*iter)->Encrypt(); // decrypt the login string
                	(*iter)->GetDataAsString(accountLogin, USERNAME_MAXLEN);
                }
				else
					hopeLogin = true;
				break;
			
			case HTLC_DATA_PASSWORD:
				(*iter)->Encrypt(); // decrypt the password
				(*iter)->GetDataAsString(accountPassword, PASSWORD_MAXLEN);
				break;
			
			case HTLC_DATA_NAME:
				(*iter)->GetDataAsString(userName, USERNAME_MAXLEN);
                mUser.SetName(userName);
				break;
			
			case HTLC_DATA_ICON:
				mUser.SetIcon((*iter)->GetDataAsUInt16());
				break;
            
            case HTLC_DATA_VERSION:
                mVersion = (*iter)->GetDataAsUInt16();
                DEBUG_CALL(printf("client version: %u\n", mVersion); fflush(stdout));
                break;
		}
		iter++;
	}
    
    /*
     * Hotline 1.5 sends a 1.2.3 style login, but then does a request for threaded news
     *   it uses a different object for threaded news also.
     * Hotline 1.5.5 sends a version number of 151 and uses the more recent revision
     *   of the threaded news protocol.
     * Hotline 1.9 sends 190 as the version number, and if the server sends a version
     *   then it will support both flat and threaded news. otherwise it just does flat news.
     *
     * Frogblast 1.2.3 sends 185 as the version number. It will only support threaded news
     *   if the server sends a version number, and flat news if it doesnt.
     * Heidrun 0.5 I think sends version 123 but will use threaded news if the server
     *   sends a version number in the login sequence (hl185 style).
     *   other wise, it will use flat news.
     */
	
	if (hopeLogin && (mUser.RealName().size() != 0 || mVersion > 0))
		// HOPE login doesn't send a user name
        // if it sends version then it isn't HOPE
        hopeLogin = false;
	
	if (!hopeLogin)
	{
        DEBUG_CALL(printf("user: %s login: %s id: %u\n",
            mUser.RealName().c_str(), mUser.Login().c_str(), mUser.ID());
            fflush(stdout));
        if (accountLogin.length() > 0)
        {
            mUser.Account().SetLogin(accountLogin);
            mUser.Account().SetPassword(accountPassword);
        }
		
        if (gServer->mDataBase.FetchAccount(mUser.Account()))
        {
			AsyncDNS dns;
			dns.ResolveHost(Address(), this);
            if (mUser.IsGuest())
			{
				if (gServer->mDataBase.IsNameBanned(mUser.RealName()))
				{
					DEBUG_CALL(printf("name is banned so banning ip\n"));
					gServer->mDataBase.AddHostBan(mSocket->RemoteHostName(),
                        gServer->serverUser, "name is banned", 0);
					Disconnect();
					return;
				}
				
				// check for the Leech client HotGrabber
				if (strstr(mUser.RealName().c_str(), "@ hgrab"))
				{
					Ban(gServer->serverUser, kBanReasonLeech);
					return;
				}
				
                if (gServer->Config().socksDetectLevel > 0)
                {
                    // here i spawn a thread to detect if
                    // the guest client is using a SOCKS firewall
                    // UPDATE: this causes the server to crash on linux kernel 2.4.x sometimes
                    InspectSOCKSThread *socksInspector =
                        new InspectSOCKSThread(Address(), *gServer);
                    socksInspector->Create();
                }
            }
			
			if (mUser.Access().disconnect_users)
				mUser.SetStatus(mUser.Status() | ADMIN_MASK);
			
			mUser.LoggedIn(true);
			// hl1.5.5 through 1.8.5 don't send a name, so don't show them in the userlist yet
			if (mUser.RealName().size())
			{
				// for hl1.5.5+ login i set the login time when it sends the username
				mUser.SetLoginTime();
				HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
				userChangePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
				userChangePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
				userChangePacket.AddUInt16Object(HTLS_DATA_ICON, mUser.Icon());
				userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, mUser.Status());
				gServer->SendToOthers(userChangePacket, *this);
                
#if defined(CONFIG_LINKING)
				gServer->Link().LinkUserJoin(*this);
#endif
                ServerLog::ConnectLog(*this);
			}
			
			// insert user into database
			gServer->mDataBase.AddUser(mUser, mSocket->RemoteHostName());
		}
        else
        {
            DEBUG_CALL(printf("fetch account failed\n"));
        }
	}
    
    if (mUser.IsLoggedIn())
    {
        HLPacket loginTaskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
				loginTaskPacket.AddStringObject( (u_int16_t)0x00a2 , gServer->Config().serverName );
				loginTaskPacket.AddUInt32Object( (u_int16_t)160		 , /*gServer->Config().serverVersion*/123 );//if version is 150 more then client can't login.added by ortana.

        
        // if (mVersion > 0 && mUser.Name().size() == 0) <--- this will detect frogblast
        // if i send 123 as the server version, then hl185 will fallback to 123 protocol
#if defined(CONFIG_THREADEDNEWS)
        // this hl 185 login protocol stuff
        // it can only be enabled when threaded news is used
        loginTaskPacket.AddUInt16Object(HTLS_DATA_VERSION, gServer->Config().serverVersion);
		// i think this is the size of the download queue
        loginTaskPacket.AddUInt16Object(HTLS_DATA_SERVERQUEUE, gServer->mTransAM->QueueSize());
		loginTaskPacket.AddStringObject(HTLS_DATA_SERVERNAME, gServer->Config().serverName);
#endif // CONFIG_THREADEDNEWS
        SendPacket(loginTaskPacket);
        
#if defined(CONFIG_THREADEDNEWS)
        // when hl185 login is enabled, an agreement packet should
        // always be send, the ChatGreeting shouldn't be sent until
        // after AgreementAgree
        HLPacket agreementPacket(HTLS_HDR_AGREEMENT, 0, 0);
        if (!mUser.Access().dont_show_agreement)
        {
            string agreementString;
            gServer->mDataBase.GetAgreement(agreementString);
            
            if (agreementString.length())
                agreementPacket.AddStringObject(HTLS_DATA_AGREEMENT, agreementString);
            else
                agreementPacket.AddUInt16Object(HTLS_DATA_NOAGREEMENT, 1);
        }
        
        // always send the agreement to hl15+ because they will
        // need to do an agreement agree
        if (mVersion >= 150)
            SendPacket(agreementPacket);
#else
        // hl1.2 protocol only sends agreement packet if there is
        // an agreement and the user is supposed to see it
        if (!mUser.Access().dont_show_agreement)
        {
            string agreementString;
            gServer->mDataBase.GetAgreement(agreementString);
            
            if (agreementString.length())
            {
                HLPacket agreementPacket(HTLS_HDR_AGREEMENT, 0, 0);
                agreementPacket.AddStringObject(HTLS_DATA_AGREEMENT, agreementString);
                
                SendPacket(agreementPacket);
            }
        }
#endif // CONFIG_THREADEDNEWS

#if defined(CONFIG_THREADEDNEWS)
        // if the version isn't send then the client won't
        // be doing an agreement agree, so we need to send this now
        if (mVersion == 0)
        {
            SendSelfInfo();
            SendChatGreeting();
        }
#else
        SendSelfInfo();
        SendChatGreeting();
#endif // CONFIG_THREADEDNEWS
    }
    else
    {
        SendTaskError(kTErrorEmpty, inPacket.GetID());
    }
}

#if defined(CONFIG_THREADEDNEWS)
void HLClient::HandleAgreementAgree(HLPacket &inPacket)
{
    DEBUG_CALL(printf("HLClient::HandleAgreementAgree\n"); fflush(stdout));
    HLObjectList::iterator iter = inPacket.begin();
    string userName;
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NAME:
				(*iter)->GetDataAsString(userName, USERNAME_MAXLEN);
                mUser.SetName(userName);
				break;
			
			case HTLC_DATA_ICON:
				mUser.SetIcon((*iter)->GetDataAsUInt16());
				break;
            
            case HTLC_DATA_BAN:
                DEBUG_CALL(printf("htlc_data_ban: %u\n",
                    (*iter)->GetDataAsUInt16()); fflush(stdout));
                break;
		}
		iter++;
	}
    
    if (mUser.LoginTime() != 0)
    {
        DEBUG_CALL(printf("got agreement agree and login time != 0\n"));
	}
    else
    	mUser.SetLoginTime();

		//TODO : BAN
	
	// modify user in database
    gServer->mDataBase.UpdateUser(mUser);
    
    HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
	userChangePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
	userChangePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
    userChangePacket.AddUInt16Object(HTLS_DATA_ICON, mUser.Icon());
    userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, mUser.Status());
    gServer->SendToOthers(userChangePacket, *this);

#if defined(CONFIG_LINKING)
	gServer->Link().LinkUserJoin(*this);
#endif

    ServerLog::ConnectLog(*this);
    
    HLPacket agreeTaskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
	SendPacket(agreeTaskPacket);
    
    SendSelfInfo();    
    SendChatGreeting();
}
#endif // CONFIG_THREADEDNEWS

void HLClient::HandleUserGetList(HLPacket &inPacket)
{
	ClientList::iterator iter = gServer->mClientList.begin();
	HLPacket userListPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
	char dataBuf[sizeof(HL_USER) + USERNAME_MAXLEN + 2];
	HL_USER *userData = (HL_USER *)dataBuf;
	u_int16_t dataSize;
	
	while (iter != gServer->mClientList.end())
	{
		if ((*iter)->mUser.IsLoggedIn() && (*iter)->mUser.LoginTime())
		{
			dataSize = sizeof(HL_USER) + (*iter)->mUser.Name().length();
			(*iter)->mUser.GetHLUser(userData);
			userListPacket.AddObject(HTLS_DATA_USER_LIST, dataSize, (char *)userData);
		}
		iter++;
	}
	
	string chatSubject;
	gServer->GetSubject(chatSubject);
	userListPacket.AddStringObject(HTLS_DATA_CHAT_SUBJECT, chatSubject);
	SendPacket(userListPacket);
}

void HLClient::HandleUserChange(HLPacket &inPacket)
{
	HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
	HLObjectList::iterator iter = inPacket.begin();
	string newName = mUser.Name();
	u_int16_t newIcon = mUser.Icon();
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NAME:
				(*iter)->GetDataAsString(newName, USERNAME_MAXLEN);
				//if (newName.length() == 0)
				//	newName.assign("unnamed");
				break;
			
			case HTLC_DATA_ICON:
				newIcon = (*iter)->GetDataAsUInt16();
				break;
			
			default:
				// allow clients to broadcast new types of data
				// through user change messages
				userChangePacket.AddObject(*(*iter));
				break;
		}
		iter++;
	}
		
	// code to detect bots that try to take admin names
	// will not kick an admin that connects
	// as a guest with the same name (it checks ips)
	if (mUser.IsGuest())
	{
		ClientList::iterator cIter = gServer->mClientList.begin();
		while (cIter != gServer->mClientList.end())
		{
			if ((*cIter)->mSocket)
			{
				if ((*cIter)->User().IsLoggedIn() && (*cIter)->User().ID() != mUser.ID() &&
					(*cIter)->mSocket->RemoteHost() != mSocket->RemoteHost())
				{
					if ((*cIter)->User().Name() == newName &&
						(*cIter)->User().Status() & ADMIN_MASK)
					{
						DEBUG_CALL(printf("bot trying to take admin's name\n"));
						Ban(gServer->serverUser, kBanReasonStealingAdminsName);
						return;
					}
				}
			}
			cIter++;
		}
	}

	//TODO : BAN
	
	mUser.SetName(newName);
	mUser.SetIcon(newIcon);

	// this is for setting the login time for 1.8.5 type clients
	// that send no username and get delayed before being
	// added to the userlist
	if (mUser.LoginTime() == 0)
	{
		mUser.SetLoginTime();

#if defined(CONFIG_LINKING)
		gServer->Link().LinkUserJoin(*this);
#endif

		ServerLog::ConnectLog(*this);
	}
#if defined(CONFIG_LINKING)
	else
	{
		// if the user is not a new user then send an update
		gServer->Link().LinkUserChange(*this);
	}
#endif

	// modify user in database
	gServer->mDataBase.UpdateUser(mUser);
	
	userChangePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
	userChangePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
	userChangePacket.AddUInt16Object(HTLS_DATA_ICON, mUser.Icon());
	userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, mUser.Status());
	
	// need to use SendToOthers if it is 185 (and this still applies)
	gServer->SendToAll(userChangePacket);
}

void HLClient::HandleUserKick(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	u_int16_t userID = 0;
	bool isBan = false;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_UID:
				userID = (*iter)->GetDataAsUInt16();
				break;
			
			case HTLC_DATA_BAN:
                isBan = true;
                break;
		}
		iter++;
	}
	
	bool kicked = false;
	if (mUser.Access().disconnect_users && userID != 0)
	{
        HLClient *kickClient = gServer->GetClientForID(userID);
		if (kickClient && !kickClient->User().Access().cant_be_disconnected)
		{
            char chatBuf[256];
            kicked = true;
			if (isBan)
            {

							//2003/07/30 added by ortana.
#ifdef _JAPAN_
							snprintf(chatBuf, 255, "\r               <<%s  %s BAN悤ƂĂ܂B>>",
                    mUser.Name().c_str(), kickClient->User().Name().c_str());
#else
                snprintf(chatBuf, 255, "\r<%s banning %s>",
                    mUser.Name().c_str(), kickClient->User().Name().c_str());
#endif//_JAPAN_
            }
			else
            {
#ifdef _JAPAN_
							snprintf(chatBuf, 255, "\r               <<%s  %sOAEg悤ƂĂ܂B>>",
								mUser.Name().c_str(), kickClient->User().Name().c_str());
#else
							snprintf(chatBuf, 255, "\r<%s kicking %s>",
								mUser.Name().c_str(), kickClient->User().Name().c_str());
#endif//_JAPAN_
            }
            
            HLPacket chatKickPacket(HTLS_HDR_CHAT, 0, 0);
            string chatString(chatBuf);
            chatKickPacket.AddStringObject(HTLS_DATA_CHAT, chatString);
            gServer->SendToAll(chatKickPacket, HLServer::CanKickFilter);
            
            isBan ? kickClient->Ban(mUser, "", gServer->Config().banTimeout) : kickClient->Disconnect();
		}
	}
	
    if (userID != mUser.ID())
    {
        // no need to complete the task if they kicked themselves
        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), !kicked);
        SendPacket(taskPacket);
    }
}

void HLClient::HandleUserGetInfo(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	u_int16_t userID = 0;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_UID:
				userID = (*iter)->GetDataAsUInt16();
				break;
		}
		iter++;
	}
	
	if (userID != 0)
	{
		HLClient *infoClient = gServer->GetClientForID(userID);
		// let users with get info privs get info on anyone
		// users without the priv can only get info on themselves
		if (infoClient)
		{
			string infoString;
			string transferStatsString;
			bool fullInfo =
                (mUser.Access().get_user_info || (mUser.ID() == userID));
            
			infoClient->User().GetInfo(infoClient->Address(), infoString, fullInfo);
            if (fullInfo)
            {
				if (infoClient->Hostname().size())
				{
					char hostnameBuf[1024];
					snprintf(hostnameBuf, 1024, "\rHostname: %s", infoClient->Hostname().c_str());
					infoString.append(hostnameBuf);
				
				}
				
				if (infoClient->mVersion)
				{
					char versionBuf[32];
					snprintf(versionBuf, 32, "\r Version: %u", infoClient->mVersion);
					infoString.append(versionBuf);
				}
				else
					infoString.append("\r Version: None sent");

                infoString.append("\r__________ Downloads ___________\r\r");
                // this is for reporting running and queued downloads
                TransferList::iterator iter = gServer->mTransAM->mDownloadList.begin();
                while (iter != gServer->mTransAM->mDownloadList.end())
                {
                    if ((*iter)->User().ID() == infoClient->User().ID())
                    {
                        (*iter)->GetStats(transferStatsString);
                        infoString.append(transferStatsString);
                    }
                    iter++;
                }
                
                iter = gServer->mTransAM->mDownloadQueue.begin();
                u_int32_t counter = 1;
                while (iter != gServer->mTransAM->mDownloadQueue.end())
                {
                    if ((*iter)->User().ID() == infoClient->User().ID())
                    {
                        (*iter)->GetStats(transferStatsString, counter);
                        infoString.append(transferStatsString);
                    }
                    counter++;
                    iter++;
                }
                
                infoString.append("___________ Uploads ____________\r\r");
                iter = gServer->mTransAM->mUploadList.begin();
                while (iter != gServer->mTransAM->mUploadList.end())
                {
                    if ((*iter)->User().ID() == infoClient->User().ID())
                    {
                        (*iter)->GetStats(transferStatsString);
                        infoString.append(transferStatsString);
                    }
                    iter++;
                }
            }
            
			HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
            // adding the userID and especially the user name here is pointless
            // this should be removed from any future protocol requirements
            taskPacket.AddUInt16Object(HTLS_DATA_UID, infoClient->User().ID());
			taskPacket.AddStringObject(HTLS_DATA_NAME, infoClient->User().Name());
			taskPacket.AddStringObject(HTLS_DATA_USER_INFO, infoString);
			SendPacket(taskPacket);
			return;
		}
	}
	
	// error, user not found
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleNewsGetFile(HLPacket &inPacket)
{
	if (mUser.Access().read_news)
	{
		string newsFile;
		gServer->mDataBase.GetNewsFile(newsFile);
		
		HLPacket newsFilePacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
		newsFilePacket.AddStringObject(HTLS_DATA_NEWS, newsFile);
		SendPacket(newsFilePacket);
	}
	else
	{
		// error, no news for you!
		SendTaskError(kTErrorGeneric, inPacket.GetID());
	}
}

void HLClient::HandleNewsPost(HLPacket &inPacket)
{
	HLPacket newsPostPacket(HTLS_HDR_NEWSFILE_POST, 0, 0);
	HLObjectList::iterator iter = inPacket.begin();
	string postString;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NEWSFILE_POST:
				(*iter)->GetDataAsString(postString, POST_LIMIT);
				break;
		}
		iter++;
	}
	
	if (postString.length() && mUser.Access().post_news)
	{
		gServer->mDataBase.AddNewsPost(mUser, postString);
		string tempString;
		HLServer::FormatNewsPost(mUser.Name(), time(0), postString, tempString);
		newsPostPacket.AddStringObject(HTLS_DATA_NEWS, tempString);
		gServer->SendToAll(newsPostPacket, HLServer::ReadNewsFilter);
		
		HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
		SendPacket(taskPacket);
		return;
	}
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleMessage(HLPacket &inPacket)
{
	HLPacket msgPacket(HTLS_HDR_MSG, 0, 0);
	HLObjectList::iterator iter = inPacket.begin();
	HLClient *msgClient = NULL;
	string msgString;
	
    if (mUser.Access().send_msgs)
    {
        msgPacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
		msgPacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
		while (iter != inPacket.end())
        {
            switch ((*iter)->GetType())
            {
                case HTLC_DATA_MSG:
                    // kelly was getting blank messages in hl185
                    // (i think i fixed this with the packet id counter)
                    (*iter)->GetDataAsString(msgString);
                    msgPacket.AddObject(*(*iter));
                    break;
                
                case HTLC_DATA_UID:
                    msgClient = gServer->GetClientForID((*iter)->GetDataAsUInt16());
                    break;
                
                default:
                    DEBUG_CALL(printf("msg fowarding object type: 0x%X\n", (*iter)->GetType()); fflush(stdout));
                    // allow clients to foward new types of data through messages
                    msgPacket.AddObject(*(*iter));
                    break;
            }
            iter++;
        }
        
        // more bot detection code for message bots
        if (mUser.IsGuest() && msgClient)
        {
            time_t currentTime = time(0);
            
            if ((currentTime - mUser.LoginTime() <= 1) ||
                (currentTime - msgClient->User().LoginTime() <= 1))
            {
                Ban(gServer->serverUser, kBanReasonBot);
                return;
            }
            
            if	(strstr(msgString.c_str(), "message delivered by") ||
                    strstr(msgString.c_str(), "message was sent by") ||
                    strstr(msgString.c_str(), "am a bot"))
            {
                Ban(gServer->serverUser, kBanReasonBot);
                return;
            }
        }
        
        if (msgClient)
		{
#if defined(CONFIG_LINKING)
			if (msgClient->mSocket)
				msgClient->SendPacket(msgPacket);
			else
				// this is a remote user
				gServer->Link().LinkMessage(*this, *msgClient, msgString);
#else
            msgClient->SendPacket(msgPacket);
#endif
		}
	}
    
	if (msgClient != NULL)
	{
		HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
		SendPacket(taskPacket);
	}
	else
		SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleChat(HLPacket &inPacket)
{
	HLPacket chatPacket(HTLS_HDR_CHAT, 0, 0);
	HLChat *chat = NULL;
	HLObjectList::iterator iter = inPacket.begin();
	u_int32_t style = 0;
	string chatString;
	string formattedChatString;
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_CHAT:
				(*iter)->GetDataAsString(chatString, CHAT_LIMIT);
                FileUtils::fixLineEndings(chatString);
				break;
			
			case HTLC_DATA_CHAT_ID:
				if ((chat = ChatIDToChatPtr((*iter)->GetDataAsUInt32())))
					chatPacket.AddObject(*(*iter));
				break;
			
			case HTLC_DATA_STYLE:
				style = (*iter)->GetDataAsUInt32();
				break;
		}
		iter++;
	}
	
	formattedChatString = chatString;
	
	unsigned int lineCount;
	HLServer::FormatChat(*this, style, formattedChatString, lineCount);
	
    // more detection for flooding
    if (!mUser.Access().cant_be_disconnected && lineCount >= FLOOD_LIMIT)
    {
		gServer->AddClientToKickList(*this);
        if (IsConnected()) // make sure client didn't just get banned for strike rule
		{
			Disconnect();
			HLPacket chatKickPacket(HTLS_HDR_CHAT, 0, 0);

			//2003/07/30 added by ortana.
#ifdef _JAPAN_
			string kickChatString("\r               <<");
			kickChatString.append(mUser.Name());
			kickChatString.append(" flooding attackׁAؒf܂B>>");
			
#else
			string kickChatString("\r<");
			kickChatString.append(mUser.Name());
			kickChatString.append(" was kicked for chat flooding>");
#endif//_JAPAN
			chatKickPacket.AddStringObject(HTLS_DATA_CHAT, kickChatString);
			gServer->SendToOthers(chatKickPacket, *this);
        }
		return;
    }

	chatPacket.AddStringObject(HTLS_DATA_CHAT, formattedChatString);
    
    // this is the way hotline chat should work:
	// send just the unformatted chat and the userid and let the client
	// do all the formatting. it would make parsing chat a lot easier.
    // UPDATE: i have to add this after the chat string because frogblast is broken
    chatPacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
	
	if (chat)
	{
		if (gServer->IsValidChat(*chat))
		{
			if (chat->IsClientInChat(*this))
				chat->SendToClients(chatPacket);
		}
	}
	else
	{
		gServer->SendToAll(chatPacket, HLServer::ReadChatFilter);
#if defined(CONFIG_LINKING)
		// send chat to hub so it will be broadcast to other servers
		gServer->Link().LinkChat(*this, style, chatString);
#endif
	}
}

void HLClient::HandleChatSubject(HLPacket &inPacket)
{
	HLPacket chatSubjectPacket(HTLS_HDR_CHAT_SUBJECT, 0, 0);
	HLChat *chat = NULL;
	HLObjectList::iterator iter = inPacket.begin();
	string chatSubject;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_CHAT_SUBJECT:
				(*iter)->GetDataAsString(chatSubject);
				chatSubjectPacket.AddObject(*(*iter));
				break;
			
			case HTLC_DATA_CHAT_ID:
				chat = ChatIDToChatPtr((*iter)->GetDataAsUInt32());
				chatSubjectPacket.AddObject(*(*iter));
				break;
		}
		iter++;
	}
	
	if (chat)
	{
		if (gServer->IsValidChat(*chat))
		{
			if (chat->IsClientInChat(*this))
			{
				chat->SetSubject(chatSubject);
				chat->SendToClients(chatSubjectPacket);
			}
		}
	}
	else
	{
		gServer->SetSubject(chatSubject);
		gServer->SendToAll(chatSubjectPacket);
	}
}

void HLClient::HandleChatCreate(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	u_int16_t userID = 0;
	string chatName;
	HLChat *newChat;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_UID:
				userID = (*iter)->GetDataAsUInt16();
				break;
		}
		iter++;
	}
	
    // at some point i guess i should be checking mUser.Access().create_pchats
    // but most people won't have this in their UserData files yet
	if (gServer->CountChatsForClient(*this) >= MAX_CHATSFORCLIENT ||
		gServer->TotalChats() >= MAX_CHATSFORSERVER ||
		(!mUser.Access().read_chat && !mUser.Access().send_chat && !mUser.Access().create_pchats))
	{
		// client is already in the maximum number of chats
		// or the server has reached its max number of chats
		// or the client can't read and send chat
		SendTaskError(kTErrorGeneric, inPacket.GetID());
		return;
	}
	
	HLPacket chatTaskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
	newChat = new HLChat(*this, chatName);
	chatTaskPacket.AddUInt32Object(HTLS_DATA_CHAT_ID, ChatPtrToChatID(newChat));
	chatTaskPacket.AddUInt16Object(HTLS_DATA_COLOUR, mUser.Status());
	chatTaskPacket.AddUInt16Object(HTLS_DATA_ICON, mUser.Icon());
	chatTaskPacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
	chatTaskPacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
	
	SendPacket(chatTaskPacket);
	
	if (userID != mUser.ID() && userID != 0)
	{
		HLClient *inviteClient = gServer->GetClientForID(userID);
		if (inviteClient)
		{
			HLPacket chatInvitePacket(HTLS_HDR_CHAT_INVITE, 0, 0);
			chatInvitePacket.AddUInt32Object(HTLS_DATA_CHAT_ID, ChatPtrToChatID(newChat));
			chatInvitePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
			chatInvitePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
			// add this chat id to a client list
			inviteClient->AddChatInvite(*newChat);
			inviteClient->SendPacket(chatInvitePacket);
		}
	}
}

void HLClient::HandleChatInvite(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	HLChat *inviteChat = NULL;
	u_int16_t userID = 0;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_CHAT_ID:
				inviteChat = ChatIDToChatPtr((*iter)->GetDataAsUInt32());
				break;
				
			case HTLC_DATA_UID:
				userID = (*iter)->GetDataAsUInt16();
				break;
		}
		iter++;
	}
	
	if (inviteChat && userID)
	{
		if (gServer->IsValidChat(*inviteChat))
		{
			if (gServer->IsClientInChat(*this, *inviteChat))
			{
				HLClient *inviteClient = gServer->GetClientForID(userID);
				if (inviteClient)
				{
					if (!gServer->IsClientInChat(*inviteClient, *inviteChat))
					{
						HLPacket chatInvitePacket(HTLS_HDR_CHAT_INVITE, 0, 0);
						chatInvitePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
						chatInvitePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
						chatInvitePacket.AddUInt32Object(HTLS_DATA_CHAT_ID, ChatPtrToChatID(inviteChat));
						inviteClient->AddChatInvite(*inviteChat);
						inviteClient->SendPacket(chatInvitePacket);
					}
				}
			}
		}
	}
}

void HLClient::HandleChatDecline(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	HLChat *declineChat = NULL;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_CHAT_ID:
				declineChat = ChatIDToChatPtr((*iter)->GetDataAsUInt32());
				break;
		}
		iter++;
	}
	
	if (declineChat)
	{
		if (RemoveChatInvite(*declineChat))
		{
			HLPacket declineChatPacket(HTLS_HDR_CHAT, 0, 0);
			declineChatPacket.AddUInt32Object(HTLS_DATA_CHAT_ID, ChatPtrToChatID(declineChat));

			//2003/07/30 added by ortana.
#ifdef _JAPAN_
			string declineString("\r               <<");
			declineString.append(mUser.Name());
			declineString.append(" ̓vCx[g`bgۂ܂B>>");
#else
			string declineString("\r<");
			declineString.append(mUser.Name());
			declineString.append(" has declined the invitation to chat>");
#endif//_JAPAN_
			declineChatPacket.AddStringObject(HTLS_DATA_CHAT, declineString);
			declineChat->SendToClients(declineChatPacket);
		}
	}
}

void HLClient::HandleChatJoin(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	HLChat *joinChat = NULL;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_CHAT_ID:
				joinChat = ChatIDToChatPtr((*iter)->GetDataAsUInt32());
				break;
				
		}
		iter++;
	}
	
	if (joinChat)
	{
		if (RemoveChatInvite(*joinChat))
		{
			// send chat userlist to new client
			HLPacket taskJoinPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
			joinChat->AddUserListToPacket(taskJoinPacket);
			string chatSubject;
			joinChat->GetSubject(chatSubject);
			taskJoinPacket.AddStringObject(HTLS_DATA_CHAT_SUBJECT, chatSubject);
			SendPacket(taskJoinPacket);
			
			// add user to chat userlist
			joinChat->AddClient(*this);
			
			// send user change for client
			HLPacket chatUserChangePacket(HTLS_HDR_CHAT_USER_CHANGE, 0, 0);
			chatUserChangePacket.AddUInt32Object(HTLS_DATA_CHAT_ID, ChatPtrToChatID(joinChat));
			chatUserChangePacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
			chatUserChangePacket.AddStringObject(HTLS_DATA_NAME, mUser.Name());
			chatUserChangePacket.AddUInt16Object(HTLS_DATA_ICON, mUser.Icon());
			chatUserChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, mUser.Status());
			joinChat->SendToClients(chatUserChangePacket);
			return;
		}
	}
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleChatPart(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	HLChat *partChat = NULL;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_CHAT_ID:
				partChat = ChatIDToChatPtr((*iter)->GetDataAsUInt32());
				break;
				
		}
		iter++;
	}
	
	if (partChat)
	{
		if (gServer->IsValidChat(*partChat))
		{
			if (partChat->IsClientInChat(*this))
			{
				partChat->RemoveClient(*this);
				HLPacket chatUserPartPacket(HTLS_HDR_CHAT_USER_PART, 0, 0);
				chatUserPartPacket.AddUInt32Object(HTLS_DATA_CHAT_ID, ChatPtrToChatID(partChat));
				chatUserPartPacket.AddUInt16Object(HTLS_DATA_UID, mUser.ID());
				partChat->SendToClients(chatUserPartPacket);
				if (partChat->IsEmpty())
					delete partChat;
			}
		}
	}
}

void HLClient::HandleFileList(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	PathVector listPathVector;

	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_DIR:
				//(*iter)->GetDataAsPath(relativePath);
				(*iter)->GetDataAsPathVector(listPathVector);
				break;
		}
		iter++;
	}
	
	string fullPath;
	if (PathVectorToRealPath(mUser.FilesPath(), listPathVector, fullPath))
	{
        bool isDropBoxesDir;
        bool isUsersDropBox;
		if (canUserAccessPath(mUser, listPathVector, isDropBoxesDir, isUsersDropBox))
		{
            // this is part of bot detection code
            // the download code makes sure the user has listed files
            mListedFiles = true;
            
			HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
			scanPathToPacket(fullPath, taskPacket);
			SendPacket(taskPacket);
			
			if (isUsersDropBox)
			{
				// we do this so HLclient 1.9 will enable the buttons in the file window
				// to allow the user to manage files in their drop box
				SendSelfInfo(true);
			}
			return;
		}
	}
	// notice the taskErrorPacket sends 0 instead of 1 for the flag
	// sending 1 caused a problem with zombie.
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
	SendPacket(taskErrorPacket);
}


void HLClient::HandleFileDelete(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	string fileName;
	PathVector filePathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(fileName);
                FileUtils::NetworkToHomeString(fileName);
				break;
				
			case HTLC_DATA_DIR:
				//(*iter)->GetDataAsPath(relativePath);
				(*iter)->GetDataAsPathVector(filePathVector);
				break;
		}
		iter++;
	}
	
	// make sure they have access to the complete path
	filePathVector.push_back(fileName);
    bool isDropBoxesDir;
    bool isUsersDropBox;
    if (canUserAccessPath(mUser, filePathVector, isDropBoxesDir, isUsersDropBox))
	{
#ifdef __GNUG__
#warning this function needs to be rewritten to check if the path is a file or dir
#endif
        if ((mUser.Access().delete_folders && !isDropBoxesDir) ||
            mUser.Access().modify_users || isUsersDropBox)
        {
            // don't resolve the last element
            filePathVector.pop_back();
            string rmPath;
            if (PathVectorToRealPath(mUser.FilesPath(), filePathVector, rmPath))
            {
                rmPath.append(fileName);
                if (deletePath(rmPath))
                {
                    HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                    SendPacket(taskPacket);
                    return;
                }
            }
        }
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleFileMkDir(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	string directoryName;
    PathVector relativePathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(directoryName);
                FileUtils::NetworkToHomeString(directoryName);
				break;
				
			case HTLC_DATA_DIR:
				//(*iter)->GetDataAsPath(relativePath);
                (*iter)->GetDataAsPathVector(relativePathVector);
				break;
		}
		iter++;
	}
	
    bool isDropBoxesDir;
    bool isUsersDropBox;
	if (canUserAccessPath(mUser, relativePathVector, isDropBoxesDir, isUsersDropBox))
    {
        if (mUser.Access().create_folders || isUsersDropBox)
        {
            string dirPath;
            if (PathVectorToRealPath(mUser.FilesPath(), relativePathVector, dirPath))
            {
                dirPath.append(directoryName);
                if (mkdir(dirPath.c_str(), gServer->mConf.dirPermissions) == 0)
                {
                    HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                    SendPacket(taskPacket);
                    return;
                }
                else
                {
                    DEBUG_CALL(printf("mkdir error: %s\n", strerror(errno)); fflush(stdout));
                }
            }
        }
    }
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleFileGetInfo(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	PathVector relativePathVector;
	string fileName;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(fileName);
                break;
				
			case HTLC_DATA_DIR:
				(*iter)->GetDataAsPathVector(relativePathVector);
				break;
		}
		iter++;
	}
	
    bool isDropBoxesDir;
    bool isUsersDropBox;
    // to make sure they have access to the entire path
    relativePathVector.push_back(fileName);
    if (canUserAccessPath(mUser, relativePathVector, isDropBoxesDir, isUsersDropBox))
    {
        // remove the fileName before resolving the path to allow
        // them to get info on an alias/symlink
        relativePathVector.pop_back();
        string fullPath;
        if (PathVectorToRealPath(mUser.FilesPath(), relativePathVector, fullPath))
        {
            HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
            if (addFileInfoToPacket(fullPath, fileName, taskPacket))
            {
                SendPacket(taskPacket);
                return;
            }
        }
	}
    // error case
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleFileSetInfo(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	string fileName;
	string newFileName;
	string fileComment;
    PathVector filePathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(fileName);
                break;
			
			case HTLC_DATA_DIR:
				//(*iter)->GetDataAsPath(relativePath);
				(*iter)->GetDataAsPathVector(filePathVector);
				break;
			
			case HTLC_DATA_FILE_RENAME:
				(*iter)->GetDataAsString(newFileName);
                break;
			
			case HTLC_DATA_FILE_COMMENT:
				(*iter)->GetDataAsString(fileComment);
				break;
		}
		iter++;
	}
    
    // make sure they have access to the complete path
	filePathVector.push_back(fileName);
    bool isDropBoxesDir;
    bool isUsersDropBox;
    if (canUserAccessPath(mUser, filePathVector, isDropBoxesDir, isUsersDropBox))
    {
#ifdef __GNUG__
#warning this function needs to be rewritten to check if the path is a file or dir
#endif // __GNUG__
        if ((mUser.Access().rename_folders && !isDropBoxesDir) ||
            mUser.Access().modify_users || isUsersDropBox)
        {
            // don't resolve the last element
            filePathVector.pop_back();
            string fullPath;
            if (PathVectorToRealPath(mUser.FilesPath(), filePathVector, fullPath))
            {
                if (fileName != newFileName)
                {
                    FileUtils::NetworkToHomeString(fileName);
                    FileUtils::NetworkToHomeString(newFileName);
                    string fromPath = fullPath;
                    fromPath.append(fileName);
                    
                    string toPath = fullPath;
                    toPath.append(newFileName);
                    
                    if (rename(fromPath.c_str(), toPath.c_str()) == 0)
                    {
                        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                        SendPacket(taskPacket);
                        return;
                    }
                }
            }
        }
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleFileMove(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	string fileName;
	//string newRelativePath;
	PathVector fromPathVector;
    PathVector toPathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(fileName);
                break;
				
			case HTLC_DATA_DIR:
				//(*iter)->GetDataAsPath(relativePath);
				(*iter)->GetDataAsPathVector(fromPathVector);
				break;
			
			case HTLC_DATA_DIR_RENAME:
				//(*iter)->GetDataAsPath(newRelativePath);
                (*iter)->GetDataAsPathVector(toPathVector);
				break;
		}
		iter++;
	}
	
	// make sure they have access to the complete path
	fromPathVector.push_back(fileName);
    
    bool fromIsDropBoxesDir;
    bool toIsDropBoxesDir;
    bool fromIsUsersDropBox;
    bool toIsUsersDropBox;
    
	if (canUserAccessPath(mUser, fromPathVector, fromIsDropBoxesDir, fromIsUsersDropBox) &&
        canUserAccessPath(mUser, toPathVector, toIsDropBoxesDir, toIsUsersDropBox))
	{
#ifdef __GNUG__
#warning this function needs to be rewritten to check if the path is a file or dir
#endif
        if ((mUser.Access().move_folders && !fromIsDropBoxesDir && !toIsDropBoxesDir) ||
            mUser.Access().modify_users || (fromIsUsersDropBox && toIsUsersDropBox))
        {
            // don't resolve the file, user might want to move an alias
            // and not the real file
            fromPathVector.pop_back();
            string fromPath;
            if (PathVectorToRealPath(mUser.FilesPath(), fromPathVector, fromPath))
            {
                string toPath;
                if (PathVectorToRealPath(mUser.FilesPath(), toPathVector, toPath))
                {
                    FileUtils::NetworkToHomeString(fileName);
                    if (movePath(fromPath, toPath, fileName))
                    {
                        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                        SendPacket(taskPacket);
						return;
                    }
                }
            }
        }
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleFileSymLink(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	string fromFileName;
	string toFileName;
	PathVector fromPathVector;
	PathVector toPathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(fromFileName);
                break;
				
			case HTLC_DATA_FILE_RENAME:
				(*iter)->GetDataAsString(toFileName);
                break;
			
			case HTLC_DATA_DIR:
				(*iter)->GetDataAsPathVector(fromPathVector);
				break;
			
			case HTLC_DATA_DIR_RENAME:
				(*iter)->GetDataAsPathVector(toPathVector);
				break;
		}
		iter++;
	}
	
	bool fromIsDropBoxesDir;
    bool toIsDropBoxesDir;
    bool fromIsUsersDropBox;
    bool toIsUsersDropBox;
    
    // make sure they have access to the complete path
    fromPathVector.push_back(fromFileName);
    if (canUserAccessPath(mUser, fromPathVector, fromIsDropBoxesDir, fromIsUsersDropBox) &&
        canUserAccessPath(mUser, toPathVector, toIsDropBoxesDir, toIsUsersDropBox))
    {
        if ((mUser.Access().make_aliases && !fromIsDropBoxesDir && !toIsDropBoxesDir) ||
            mUser.Access().modify_users || (fromIsUsersDropBox && toIsUsersDropBox))
        {
            string fromPath;
            if (PathVectorToRealPath(mUser.FilesPath(), fromPathVector, fromPath))
            {
                string toPath;
                if (PathVectorToRealPath(mUser.FilesPath(), toPathVector, toPath))
                {
                    
                    if (toFileName.empty())
                    {
                        FileUtils::NetworkToHomeString(fromFileName);
                        toPath.append(fromFileName);
                    }
                    else
                    {
                        FileUtils::NetworkToHomeString(toFileName);
                        toPath.append(toFileName);
                    }
                    
                    DEBUG_CALL(printf("symlink from: %s to %s\n",
                        fromPath.c_str(), toPath.c_str()); fflush(stdout));

										//create a shortcut file(WIN32). added by ortana.
#ifdef WIN32					
										if( strcmp( PathFindExtensionA( toPath.c_str() ) , ".lnk" ) != 0 )
										{
											toPath += ".lnk";
										}
										
										char from_buff[MAX_PATH];
										FileUtils::getAbsolutPath( fromPath , from_buff );
										BOOL ret = FileUtils::CreateShellLink( toPath.c_str() , from_buff ,
											NULL , NULL ,NULL );
										
										if( ret )
										{
											return;
										}
#else

                    if (symlink(fromPath.c_str(), toPath.c_str()) == 0)
                    {
                        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                        SendPacket(taskPacket);
                        return;
                    }

#endif//WIN32
                }
            }
        }
	}
    
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleFileGet(HLPacket &inPacket)
{
    HLObjectList::iterator iter = inPacket.begin();
	PathVector relativePathVector;
	string fileName;
	u_int32_t remoteDataForkSize = 0;
	u_int32_t remoteResourceForkSize = 0;
    string errorString;
    bool isPreview = false;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(fileName);
                break;
				
			case HTLC_DATA_DIR:
				(*iter)->GetDataAsPathVector(relativePathVector);
				break;
			
			case HTLC_DATA_RFLT:
				// resume info
				(*iter)->GetDataResumeSizes(remoteDataForkSize, remoteResourceForkSize);
				break;
			
			case HTLC_DATA_FILE_PREVIEW:
				DEBUG_CALL(printf("file preview for download\n"));
                isPreview = true;
				break;
		}
		iter++;
	}
	
	// i dunno what does this, but some bot connects and tries to download
	// this file from the root dir and then proceeds to scan all the rest
	// of the directories. i'm just gonna disconnect and ban these bitches
	// update: a newer version tries to connect and download "\0\1\0\0\8Bookmark"
    // so i've changed this code to be more general
	if (fileName.length() >= 4 && memcmp(fileName.data(), "\x00\x01\x00\x00", 4) == 0)
	{
		Ban(gServer->serverUser, kBanReasonBot);
		return;
	}
	else if (strcmp(fileName.c_str(), "L33chxT3st") == 0)
	{
		// this is for the SearchHotline bot
		Ban(gServer->serverUser, kBanReasonBot);
		return;
	}
	
	if (mUser.Access().download_files)
	{
        bool isDropBoxesDir;
        bool isUsersDropBox;
		if (canUserAccessPath(mUser, relativePathVector, isDropBoxesDir, isUsersDropBox))
        {
            string fullPath;
            // resolve the last element because we don't
            // want people downloading symlinks
            relativePathVector.push_back(fileName);
            if (PathVectorToRealPath(mUser.FilesPath(), relativePathVector, fullPath))
            {
                FileUtils::NetworkToHomeString(fileName);
                DEBUG_CALL(printf("file download (%u): %s\n",
                    (unsigned int)fileName.length(), fullPath.c_str()); fflush(stdout));
                
                // this code needs to be moved into FileUtils
                struct stat sb;
                bool noError = true;
                
                if (lstat(fullPath.c_str(), &sb) == 0)
                {
                    // i don't think this will even happen because PathVectorToRealPath
                    // should resolve all symlinks in the path
                    if (S_ISLNK(sb.st_mode))
                    {
                        // this is a link
                        if (stat(fullPath.c_str(), &sb) != 0)
                            // error resolving link
                            noError = false;
                    }
                }
                else
                    noError = false;
                
                if (noError)
                {
                    if (!S_ISDIR(sb.st_mode))
                    {
            
                        HLTransferInfo newTransferInfo;
                        newTransferInfo.user = mUser;
                        newTransferInfo.type = isPreview ? kPreviewTransfer : kDownloadTransfer;
                        newTransferInfo.fileName = fileName;
                        newTransferInfo.filePath = fullPath;
                        newTransferInfo.localDataForkSize = sb.st_size;
                        newTransferInfo.remoteDataForkSize = remoteDataForkSize;
                        newTransferInfo.transferSize = 0; // this is used for uploads
						newTransferInfo.ref = HLTransferManager::MakeRef(newTransferInfo);
						DEBUG_CALL(printf("download transfer ref: 0x%X\n", newTransferInfo.ref));
                        // resource fork sizes would also go here...
                        
                        if (newTransferInfo.localDataForkSize &&
                            newTransferInfo.localDataForkSize > newTransferInfo.remoteDataForkSize)
                        {
                            bool isLeech;
                            if (gServer->mTransAM->QueueTransferInfo(newTransferInfo, isLeech))
                            {
                                DEBUG_CALL(
                                    if (!mListedFiles)
                                    {
                                        printf("not listed files!\n");
                                        printf("user: %s login time: %u\n",
                                            mUser.Name().c_str(), (unsigned int)(time(0) - mUser.LoginTime()));
                                        fflush(stdout);
                                    }
                                );
                                
                                HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                                taskPacket.AddUInt32Object(HTLS_DATA_HTXF_REF, newTransferInfo.ref);
                                if (isPreview)
                                {
                                    taskPacket.AddUInt32Object(HTLS_DATA_HTXF_SIZE,
                                        newTransferInfo.localDataForkSize - newTransferInfo.remoteDataForkSize);
                                }
                                else
                                {
                                    taskPacket.AddUInt32Object(HTLS_DATA_HTXF_SIZE,
                                        130 /* transfer header size + 1 fork header size */ + fileName.length() +
                                        (newTransferInfo.localDataForkSize - newTransferInfo.remoteDataForkSize));
                                }
                                SendPacket(taskPacket);
                                return;
                            }
                            else // QueueTransferInfo
                            {
                                if (isLeech && mUser.IsGuest())
                                {
                                    mLeechCounter++;
                                    if (mLeechCounter == 3)
                                    {
                                        Ban(gServer->serverUser, kBanReasonLeech,
                                            gServer->Config().banTimeout);
                                        // don't return here because newTransferInfo needs to be freed
                                    }
                                }
                                DEBUG_CALL(printf("download failed to queue transfer info\n"); fflush(stdout));
                            }
                        }
                        else
                        {
                            if (newTransferInfo.localDataForkSize == 
                                newTransferInfo.remoteDataForkSize)
                            {
                                errorString.assign("You already have the complete file.");
                            }
                            
                            DEBUG_CALL(
								printf("download size test failed\n");
								printf("local size: %u resume size: %u user: %s\n",
									newTransferInfo.localDataForkSize,
									newTransferInfo.remoteDataForkSize,
									mUser.Name().c_str());
								fflush(stdout);
							);
                        }
                    }
                } // noError
                else if (mUser.LoginTime() < 3 && !mListedFiles)
                {
                    Ban(gServer->serverUser, kBanReasonBot);
                    return;
                }
                
                DEBUG_CALL(
                printf("download error user: %s login time: %u\n",
                        mUser.Name().c_str(), (unsigned int)(time(0) - mUser.LoginTime()));
                if (!mListedFiles)
                    printf("not listed files!\n");
                
                char hexBuf[513];
                u_int32_t counter = 0;
                while (counter < fileName.length())
                {
                    sprintf(&hexBuf[counter * 2], "%.2x", (u_int8_t)fileName[counter]);
                    counter++;
                }
                printf("filename in hex(%u): %s\n", (unsigned int)fileName.length(), hexBuf);
                fflush(stdout)); // end of debug code
            }
        } // canUserAccessPath
	}
	
    if (errorString.empty())
		errorString = kTErrorGeneric;
    SendTaskError(errorString, inPacket.GetID());
}

void HLClient::HandleFilePut(HLPacket &inPacket)
{
    HLObjectList::iterator iter = inPacket.begin();
	string fileName;
	u_int32_t transferSize = 0;
	PathVector uploadPathVector;
	bool resumeUpload = false;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(fileName);
                break;
				
			case HTLC_DATA_DIR:
				(*iter)->GetDataAsPathVector(uploadPathVector);
				break;
			
			case HTLC_DATA_HTXF_SIZE:
				transferSize = (*iter)->GetDataAsUInt32();
				break;
			
			case HTLC_DATA_FILE_PREVIEW:
				DEBUG_CALL(printf("file preview got in HandleFilePut()\n"));
				resumeUpload = true;
				break;
		}
		iter++;
	}
	
    if ((mUser.Access().upload_files && isValidUploadPath(uploadPathVector)) ||
        mUser.Access().upload_anywhere)
	{
        //bool isDropBoxesDir;
        //bool isUsersDropBox;
		//if (canUserAccessPath(mUser, uploadPathVector, isDropBoxesDir, isUsersDropBox))
        //{
            string uploadDirPath;
            if (PathVectorToRealPath(mUser.FilesPath(), uploadPathVector, uploadDirPath))
            {
                string fullPath;
                FileUtils::NetworkToHomeString(fileName);
                u_int32_t localDataForkSize = 0;
                if (checkUploadPathAndResumeSize(uploadDirPath,
                    fileName, fullPath, localDataForkSize))
                {
                    DEBUG_CALL(printf("file upload: %s\n", fullPath.c_str()));
                    // now check to see if it is resuming?
                    HLTransferInfo newTransferInfo;
                    newTransferInfo.user = mUser;
                    newTransferInfo.type = kUploadTransfer;
                    newTransferInfo.fileName = fileName;
                    newTransferInfo.filePath = fullPath;
                    newTransferInfo.localDataForkSize = localDataForkSize;
                    newTransferInfo.remoteDataForkSize = 0;
                    newTransferInfo.transferSize = transferSize;
					newTransferInfo.ref = HLTransferManager::MakeRef(newTransferInfo);
					DEBUG_CALL(printf("upload transfer ref: 0x%X\n", newTransferInfo.ref));
					
                    // resource fork sizes would also go here...
                    DEBUG_CALL(printf("recved transferSize: %u\n", transferSize));
                    
                    //getFileExtension(const string &inPath, string &outExtension)
                    
                    bool isLeech;
                    if (gServer->mTransAM->QueueTransferInfo(newTransferInfo, isLeech))
                    {
                        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                        taskPacket.AddUInt32Object(HTLS_DATA_HTXF_REF, newTransferInfo.ref);

												//2003/07/22 can't resume Upload. modified by ortana
                        if (newTransferInfo.localDataForkSize/* && resumeUpload*/ || resumeUpload)
                        {
                            char resumeBuf[SIZEOF_RESUME_DATA + SIZEOF_FORK_DATA];
                            struct resume_data *resumeHdr = (struct resume_data *)&resumeBuf[0];
                            struct fork_data *resumeData = (struct fork_data *)&resumeBuf[SIZEOF_RESUME_DATA];
                            memset(resumeBuf, 0, SIZEOF_RESUME_DATA + SIZEOF_FORK_DATA);
                            
                            resumeHdr->magic = htonl(0x52464C54); // 'RFLT'
                            resumeHdr->ref = htonl(0x00010000);
                            resumeHdr->hc = htons(1);
                            
                            resumeData->type = htonl(0x44415441); // 'DATA'
                            resumeData->len = htonl(newTransferInfo.localDataForkSize);
                            
                            taskPacket.AddObject(HTLS_DATA_RFLT, SIZEOF_RESUME_DATA + SIZEOF_FORK_DATA, resumeBuf);
                            
                            // if i was supporting resource forks
                            // i'd need to have resume data for 'MACR' type
                        }
                        else
                        {
                            // user does not want to resume so i'm setting
                            // the local size back to zero
                            newTransferInfo.localDataForkSize = 0;
                        }
                        SendPacket(taskPacket);
                        return;
                    } // QueueTransferInfo
                } // checkUploadPathAndResumeSize
            } // PathVectorToRealPath
        //} // canUserAccessPath
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleAccountCreate(HLPacket &inPacket)
{
    HLObjectList::iterator iter = inPacket.begin();
    string accountLogin;
    string accountPassword;
    string accountName;
    struct hl_access_bits accountAccess;
    
    DEBUG_CALL(printf("HandleAccountCreate\n"));
    while (iter != inPacket.end())
    {
        switch ((*iter)->GetType())
        {
            case HTLC_DATA_LOGIN:
                (*iter)->Encrypt();
                (*iter)->GetDataAsString(accountLogin, USERNAME_MAXLEN);
                break;
            
            case HTLC_DATA_PASSWORD:
                (*iter)->Encrypt();
                (*iter)->GetDataAsString(accountPassword, PASSWORD_MAXLEN);
                break;
            
            case HTLC_DATA_NAME:
                (*iter)->GetDataAsString(accountName, USERNAME_MAXLEN);
                break;
            
            case HTLC_DATA_ACCESS:
                if ((*iter)->GetSize() >= sizeof(accountAccess))
                {
                    accountAccess = *((struct hl_access_bits *)(*iter)->GetData());
                    DEBUG_CALL(printf("access bits: 0x%16llX\n",
                        *((u_int64_t *)&accountAccess)); fflush(stdout));
                }
                break;
            
            default:
                DEBUG_CALL(printf("obj type: 0x%X\n", (*iter)->GetType()));
                break;
        }
        iter++;
    }

    DEBUG_CALL(printf("login: %s pass: %s name: %s\n", accountLogin.c_str(),
        accountPassword.c_str(), accountName.c_str()); fflush(stdout));
    
    if (mUser.Access().create_users)
    {
#ifdef __GNUG__
#warning need to rewrite this function to make sure user isnt giving more privs than they have
#endif
        if (mVersion == 0) // because heidrun sends 123
        {
            // earlier client versions can't set these privs
            // so we use the defaults
            accountAccess.create_pchats = true;
            accountAccess.send_msgs = true;
        }
        HLAccount newAccount;
        newAccount.SetLogin(accountLogin);
        newAccount.SetPassword(accountPassword);
        newAccount.SetName(accountName);
        newAccount.SetAccess(accountAccess);
        gServer->mDataBase.CreateAccount(newAccount);
        ServerLog::AccountLog(*this, AccountActionCreate, accountLogin.c_str());
        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
        SendPacket(taskPacket);
        return;
    }

    SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleAccountDelete(HLPacket &inPacket)
{
    DEBUG_CALL(printf("HandleAccountDelete\n"));
    if (mUser.Access().delete_users)
    {
		ParseAccountDelete(inPacket);
        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
        SendPacket(taskPacket);
        return;
    }

    SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleAccountRead(HLPacket &inPacket)
{
    HLObjectList::iterator iter = inPacket.begin();
	string accountLogin;

    DEBUG_CALL(printf("HandleAccountRead\n"));
    while (iter != inPacket.end())
    {
        switch ((*iter)->GetType())
        {
            case HTLC_DATA_LOGIN:
                (*iter)->GetDataAsString(accountLogin, USERNAME_MAXLEN);
                break;
                
            default:
                DEBUG_CALL(printf("obj type: 0x%X\n", (*iter)->GetType()));
                break;
        }
        iter++;
    }
    
    if (mUser.Access().read_users)
    {
        HLAccount readAccount;
        readAccount.SetLogin(accountLogin);
        if (gServer->mDataBase.FetchAccount(readAccount, false))
        {
            HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
            taskPacket.AddStringObject(HTLC_DATA_LOGIN, readAccount.Login(), true); // login encrypted
            taskPacket.AddStringObject(HTLC_DATA_NAME, readAccount.Name());
			if (mVersion > 0 && !readAccount.Password().empty())
			{
				// it seems like newer clients want this
				taskPacket.AddUInt8Object(HTLC_DATA_PASSWORD, 0x78);
			}
            taskPacket.AddObject(HTLC_DATA_ACCESS, sizeof(struct hl_access_bits),
                (char *)&readAccount.Access());
            SendPacket(taskPacket);
            return;
        }
        /*
        HLUserData readUserData(accountLogin);
        if (readUserData.ReadFrom(gServer->Config().accountsPath))
        {
            HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
            taskPacket.AddStringObject(HTLC_DATA_LOGIN, readUserData.Login(), true); // login encrypted
            taskPacket.AddStringObject(HTLC_DATA_NAME, readUserData.Name());
            taskPacket.AddObject(HTLC_DATA_ACCESS, sizeof(struct hl_access_bits),
                (char *)&readUserData.AccessBits());
            SendPacket(taskPacket);
            return;
        }
        */
    }
    
    SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleAccountModify(HLPacket &inPacket)
{
	DEBUG_CALL(printf("HandleAccountModify\n"));
    if (mUser.Access().modify_users)
    {
		if (ParseAccountModify(inPacket, false))
		{
			HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
			SendPacket(taskPacket);
			return;
		}
    }
    
    SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleAccountList(HLPacket &inPacket)
{
	DEBUG_CALL(printf("HandleAccountList\n"));
    HLObjectList::iterator iter = inPacket.begin();
	// i'm pretty sure this packet type never contains any objects
	while (iter != inPacket.end())
    {
		switch ((*iter)->GetType())
        {
			default:
				DEBUG_CALL(printf("obj type: 0x%X\n", (*iter)->GetType()));
				break;
		}
		iter++;
	}
	
	if (mUser.Access().modify_users)
    {
		HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
		
		AccountList accountList;
		gServer->mDataBase.ListAccounts(accountList);
		AccountList::iterator accountIter = accountList.begin();
		while (accountIter != accountList.end())
		{
			HLObjectList accountObjectList;
			accountObjectList.AddStringObject(HTLC_DATA_NAME, (*accountIter).Name());
			accountObjectList.AddStringObject(HTLC_DATA_LOGIN, (*accountIter).Login(), true);
			if (!(*accountIter).Password().empty())
				accountObjectList.AddUInt8Object(HTLC_DATA_PASSWORD, 0x78);
			accountObjectList.AddObject(HTLC_DATA_ACCESS, sizeof(struct hl_access_bits),
                (char *)&((*accountIter).Access()));
			taskPacket.AddListObject(HTLC_DATA_DATA, accountObjectList);
			accountIter++;
		}
		
        SendPacket(taskPacket);
        return;
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleAccountListModify(HLPacket &inPacket)
{
	DEBUG_CALL(printf("HandleAccountListModify\n"));
	if (mUser.Access().modify_users &&
		mUser.Access().delete_users &&
		mUser.Access().create_users &&
		mUser.Access().read_users)
    {
		HLObjectList accountDataList;
		HLObjectList::iterator iter = inPacket.begin();
		bool error = false;
		while (iter != inPacket.end() && !error)
		{
			switch ((*iter)->GetType())
			{
				case HTLC_DATA_DATA:
					(*iter)->GetDataAsObjectList(accountDataList);
					// yes, thats right pepsi...i'm counting objects =P
					if (accountDataList.GetObjCount() == 1)
					{
						// this is a request to delete an account
						ParseAccountDelete(accountDataList);
					}
					else
					{
						// account modify or possibly account create
						if (!ParseAccountModify(accountDataList, true))
							error = true;
					}
					break;
				
				default:
					DEBUG_CALL(printf("halm uobj type: 0x%X\n", (*iter)->GetType()));
					break;
			}
			iter++;
		}
	
		if (!error)
		{
			HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
			SendPacket(taskPacket);
			return;
		}
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

bool HLClient::ParseAccountModify(HLObjectList &inList, bool inCanCreate)
{
	HLObjectList::iterator iter = inList.begin();
    string accountLogin;
    string accountPassword;
    string accountName;
	string oldAccountLogin;
    struct hl_access_bits accountAccess;
    bool modifyPass = false;
	bool gotPasswordObject = false;
    
    DEBUG_CALL(printf("ParseAccountModify\n"));
    while (iter != inList.end())
    {
        switch ((*iter)->GetType())
        {
			case HTLC_DATA_DATA:
				(*iter)->Encrypt();
                (*iter)->GetDataAsString(oldAccountLogin, USERNAME_MAXLEN);
				break;
				
            case HTLC_DATA_LOGIN:
                (*iter)->Encrypt();
                (*iter)->GetDataAsString(accountLogin, USERNAME_MAXLEN);
                break;
            
            case HTLC_DATA_PASSWORD:
				gotPasswordObject = true;
                if ((*iter)->GetSize() >= 1 && (*iter)->GetData()[0] != 0)
                {
                    modifyPass = true;
                    (*iter)->Encrypt();
                    (*iter)->GetDataAsString(accountPassword, PASSWORD_MAXLEN);
				}
                else if ((*iter)->GetSize() == 0)
                    modifyPass = true;
                break;
            
            case HTLC_DATA_NAME:
                (*iter)->GetDataAsString(accountName, USERNAME_MAXLEN);
                break;
            
            case HTLC_DATA_ACCESS:
                if ((*iter)->GetSize() >= sizeof(accountAccess))
                {
                    accountAccess = *((struct hl_access_bits *)(*iter)->GetData());
                    DEBUG_CALL(printf("access bits: 0x%16llX\n",
                        *((u_int64_t *)&accountAccess)); fflush(stdout));
                }
                break;
            
            default:
                DEBUG_CALL(printf(" unhandled obj type: 0x%X\n", (*iter)->GetType()));
                break;
        }
        iter++;
    }
	
	if (!oldAccountLogin.empty() && oldAccountLogin != accountLogin)
	{
		// TODO FIXME - need to implement this - jjt
		DEBUG_CALL(printf("request to change login: %s %s\n",
			oldAccountLogin.c_str(), accountLogin.c_str()));
		return false;
	}
	else
	{
		if (mVersion == 0) // because heidrun sends 123
		{
			// earlier client versions can't set these privs
			// so we use the defaults
			accountAccess.create_pchats = true;
			accountAccess.send_msgs = true;
		}
		
		HLAccount modAccount;
		modAccount.SetLogin(accountLogin);
		
		if (mVersion > 0 && !gotPasswordObject)
		{
			// Hotline 1.9 doesn't send a password object when it wants the password to be blank
			modifyPass = true;
		}
		
		if (modifyPass)
			modAccount.SetPassword(accountPassword);
		modAccount.SetName(accountName);
		modAccount.SetAccess(accountAccess);
		gServer->mDataBase.ModifyAccount(modAccount, modifyPass);
		// need to read the account here to fill in the other
		// fields that aren't changed by the account modify
		if (gServer->mDataBase.FetchAccount(modAccount, false))
		{
			ServerLog::AccountLog(*this, AccountActionModify, accountLogin.c_str());
			gServer->UpdateAccountForClients(modAccount);
			return true;
		}
		else if (inCanCreate)
		{
			gServer->mDataBase.CreateAccount(modAccount);
			ServerLog::AccountLog(*this, AccountActionCreate, accountLogin.c_str());
			return true;
		}
		else
			return false;
	}
}

void HLClient::ParseAccountDelete(HLObjectList &inList)
{
    HLObjectList::iterator iter = inList.begin();
    string accountLogin;
    
    DEBUG_CALL(printf("ParseAccountDelete\n"));
    while (iter != inList.end())
    {
        switch ((*iter)->GetType())
        {
			case HTLC_DATA_DATA:
            case HTLC_DATA_LOGIN:
                (*iter)->Encrypt();
                (*iter)->GetDataAsString(accountLogin, USERNAME_MAXLEN);
                break;
        
            default:
                DEBUG_CALL(printf("obj type: 0x%X\n", (*iter)->GetType()));
                break;
        }
        iter++;
    }
    DEBUG_CALL(printf("login: %s\n", accountLogin.c_str()); fflush(stdout));
	gServer->mDataBase.DeleteAccount(accountLogin);
	ServerLog::AccountLog(*this, AccountActionDelete, accountLogin.c_str());
	gServer->KickClientsWithAccount(accountLogin);
	return;
}

void HLClient::HandlePing(HLPacket &inPacket)
{
    DEBUG_CALL(printf("hl pinging %s\n", mUser.Name().c_str()));
	HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
	SendPacket(taskPacket);
}

void HLClient::HandleBroadcast(HLPacket &inPacket)
{
    if (mUser.Access().can_broadcast)
    {
        HLObjectList::iterator iter = inPacket.begin();
        HLPacket broadcastPacket(HTLS_HDR_MSG_BROADCAST, 0, 0);
        string broadcastString;
        while (iter != inPacket.end())
        {
            switch ((*iter)->GetType())
            {
                case HTLC_DATA_MSG:
                    (*iter)->GetDataAsString(broadcastString);
                    break;
            }
            iter++;
        }
        broadcastPacket.AddStringObject(HTLS_DATA_MSG, broadcastString);
        gServer->SendToOthers(broadcastPacket, *this);
#if defined(CONFIG_LINKING)
		// send to hub so it will be broadcast to other servers
		gServer->Link().LinkBroadcast(broadcastString);
#endif
    }
    
    HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
    SendPacket(taskPacket);
}

void HLClient::HandleFolderGet(HLPacket &inPacket)
{
	DEBUG_CALL(printf("HandleFolderGet\n"));
	HLObjectList::iterator iter = inPacket.begin();
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				break;
			
			case HTLC_DATA_DIR:
				break;
			
			default:
				DEBUG_CALL(printf("obj type: 0x%X size: %u\n",
					(*iter)->GetType(), (*iter)->GetSize()));
				break;
		}
		iter++;
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleFolderPut(HLPacket &inPacket)
{
	DEBUG_CALL(printf("HandleFolderPut\n"));
	HLObjectList::iterator iter = inPacket.begin();
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				break;
			
			case HTLC_DATA_DIR:
				break;
			
			case HTLC_DATA_HTXF_SIZE:
				break;
			
			case HTLC_DATA_FILE_NFILES:
				// short
				break;
			
			default:
				DEBUG_CALL(printf("obj type: 0x%X size: %u\n",
					(*iter)->GetType(), (*iter)->GetSize()));
				break;
		}
		iter++;
	}
	
	SendTaskError(kTErrorGeneric, inPacket.GetID());
}

void HLClient::HandleTransferStop(HLPacket &inPacket)
{
	DEBUG_CALL(printf("HandleTransferStop\n"));
	HLObjectList::iterator iter = inPacket.begin();
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			default:
				DEBUG_CALL(printf("obj type: 0x%X size: %u\n",
					(*iter)->GetType(), (*iter)->GetSize()));
				break;
		}
		iter++;
	}
}

#if defined(CONFIG_THREADEDNEWS)
/* --- News Shite --- */

void HLClient::HandleNewsDirList(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	PathVector listPathVector;

	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NEWS_DIR:
				(*iter)->GetDataAsPathVector(listPathVector);
				break;
		}
		iter++;
	}
	
	string fullPath;
	if (PathVectorToRealPath(gServer->Config().newsDirPath, listPathVector, fullPath))
	{
        HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
		try
		{
			NewsUtils::scanPathToPacket(fullPath, taskPacket);
			SendPacket(taskPacket);
			return;
		}
		catch (ios::failure &error)
		{
			DEBUG_CALL(printf("exception scanning news to packet: %s\n", error.what()));
		}
	}

	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 1);
	SendPacket(taskErrorPacket);
}

void HLClient::HandleNewsCatList(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	PathVector listPathVector;

	DEBUG_CALL(printf("HLClient:HandleNewsCatList()\n"));
	
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NEWS_DIR:
				(*iter)->GetDataAsPathVector(listPathVector);
				break;
		}
		iter++;
	}
	
	string fullPath;
	if (PathVectorToRealPath(gServer->Config().newsDirPath, listPathVector, fullPath))
	{
		try
		{
			HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
			if (NewsUtils::getPostHeaders(fullPath, taskPacket))
			{
				SendPacket(taskPacket);
				return;
			}
		}
		catch (ios::failure &error)
		{
			DEBUG_CALL(printf("exception getting post headers: %s\n", error.what()));
		}
	}
	
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 1);
	SendPacket(taskErrorPacket);
}

void HLClient::HandleNewsGetThread(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	PathVector listPathVector;
	u_int16_t threadid;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NEWS_DIR:
				(*iter)->GetDataAsPathVector(listPathVector);
				break;
				
			case HTLC_DATA_NEWS_THREADID:
				threadid = (*iter)->GetDataAsUInt16();
				break;
		}
		iter++;
	}
	
	string fullPath;
	if (PathVectorToRealPath(gServer->Config().newsDirPath, listPathVector, fullPath) && threadid != 0)
	{
		try
		{
			HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
			if (NewsUtils::getPostData(fullPath, threadid, taskPacket))
			{
				SendPacket(taskPacket);
				return;
			}
		}
		catch (ios::failure &error)
		{
			DEBUG_CALL(printf("exception getting post data: %s\n", error.what()));
		}
	}
	
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 1);
	SendPacket(taskErrorPacket);
}

void HLClient::HandleNewsMkCat(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	string categoryName;
    PathVector relativePathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NEWS_CAT_NAME:
				(*iter)->GetDataAsString(categoryName);
                FileUtils::NetworkToHomeString(categoryName);
				break;
				
			case HTLC_DATA_NEWS_DIR:
				//(*iter)->GetDataAsPath(relativePath);
                (*iter)->GetDataAsPathVector(relativePathVector);
				break;
		}
		iter++;
	}

/* it would be pretty cool if users had access to certain news dirs, which could be checked here - jcb */	
//    bool isDropBoxesDir;
//    bool isUsersDropBox;
//	if (canUserAccessPath(mUser, relativePathVector, isDropBoxesDir, isUsersDropBox))
//	{
		if (mUser.Access().create_news_bundles)
		{
            string dirPath;
            if (PathVectorToRealPath(gServer->Config().newsDirPath, relativePathVector, dirPath))
            {
                if (NewsUtils::createNewsFile(dirPath, categoryName))
                {
                    HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                    SendPacket(taskPacket);
                    return;
                }
                else
                {
                    DEBUG_CALL(printf("mkdir error: %s\n", strerror(errno)); fflush(stdout));
                }
            }
		}
//	}
	
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 1);
	SendPacket(taskErrorPacket);
}

void HLClient::HandleNewsMkDir(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	string directoryName;
    PathVector relativePathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_FILE_NAME:
				(*iter)->GetDataAsString(directoryName);
                FileUtils::NetworkToHomeString(directoryName);
				break;
				
			case HTLC_DATA_NEWS_DIR:
				//(*iter)->GetDataAsPath(relativePath);
                (*iter)->GetDataAsPathVector(relativePathVector);
				break;
		}
		iter++;
	}

/* it would be pretty cool if users had access to certain news dirs, which could be checked here - jcb */	
//    bool isDropBoxesDir;
//    bool isUsersDropBox;
//	if (canUserAccessPath(mUser, relativePathVector, isDropBoxesDir, isUsersDropBox))
//	{
		if (mUser.Access().create_news_bundles)
		{
            string dirPath;
            if (PathVectorToRealPath(gServer->Config().newsDirPath, relativePathVector, dirPath))
            {
                dirPath.append(directoryName);
#ifdef WIN32
								char abPath[MAX_PATH];
								FileUtils::getAbsolutPath( dirPath.c_str() , abPath );
								strcat( abPath , "\\" );
								if(MakeSureDirectoryPathExists(abPath))
								{
#else
                if (mkdir(dirPath.c_str(), gServer->mConf.dirPermissions) == 0)
                {
#endif//WIN32
                    HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                    SendPacket(taskPacket);
                    return;
                }
                else
                {
                    DEBUG_CALL(printf("mkdir error: %s\n", strerror(errno)); fflush(stdout));
                }
            }
		}
//	}
	
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 1);
	SendPacket(taskErrorPacket);
}

void HLClient::HandleNewsPostThread(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	
	string fileName;
	PathVector filePathVector;
	
	u_int32_t threadID;
	string postTitle;
	u_int16_t unknown;
	string mimeType;
	
	HLObject *postData = NULL;
	
	//bool hasPostData;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{
			case HTLC_DATA_NEWS_DIR:
				(*iter)->GetDataAsPathVector(filePathVector);
				break;
			
			case HTLC_DATA_NEWS_THREADID:
				threadID = (*iter)->GetDataAsUInt16();
				break;
				
			case HTLC_DATA_NEWS_SUBJECT:
				(*iter)->GetDataAsString(postTitle);
				break;
				
			case HTLC_DATA_NEWS_UNKNOWN:
				unknown = (*iter)->GetDataAsUInt16();
				break;
				
			case HTLC_DATA_NEWS_MIMETYPE:
				(*iter)->GetDataAsString(mimeType);
				break;
				
			case HTLC_DATA_NEWS_POST:
				postData = (*iter);
				break;
		}
		iter++;
	}
	
	string fullPath;
	if (PathVectorToRealPath(gServer->Config().newsDirPath, filePathVector, fullPath) && postData != NULL)
	{
		try
		{
			HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
			if (NewsUtils::addPost(fullPath, threadID, postTitle, mUser.Name(), postData->GetData(), postData->GetSize()))
			{
				SendPacket(taskPacket);
				return;
			}
		}
		catch (ios::failure &error)
		{
			DEBUG_CALL(printf("exception returning post data: %s\n", error.what()));
		}
	}
	
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 1);
	SendPacket(taskErrorPacket);
}

void HLClient::HandleNewsDeleteCatDir(HLPacket &inPacket)
{
	HLObjectList::iterator iter = inPacket.begin();
	//string relativePath;
	string fileName;
	PathVector filePathVector;
	
	while (iter != inPacket.end())
	{
		switch ((*iter)->GetType())
		{				
			case HTLC_DATA_NEWS_DIR:
				//(*iter)->GetDataAsPath(relativePath);
				(*iter)->GetDataAsPathVector(filePathVector);
				break;
		}
		iter++;
	}
	
	// make sure they have access to the complete path
/* it would be pretty cool if users had access to certain news dirs, which could be checked here - jcb */	
//    bool isDropBoxesDir;
//    bool isUsersDropBox;
//	if (canUserAccessPath(mUser, filePathVector, isDropBoxesDir, isUsersDropBox))
//	{
#ifdef __GNUG__
#warning this function needs to be rewritten to check if the path is a file or dir
#endif
/* and when it gets rewritten, fix this this line... - jcb */
        if (mUser.Access().delete_categories || mUser.Access().delete_news_bundles)
        {
            // don't resolve the last element
			fileName = filePathVector.back();
			filePathVector.pop_back();
            string rmPath;
            if (PathVectorToRealPath(gServer->Config().newsDirPath, filePathVector, rmPath))
            {
                rmPath.append(fileName);
                if (deletePath(rmPath))
                {
                    HLPacket taskPacket(HTLS_HDR_TASK, inPacket.GetID(), 0);
                    SendPacket(taskPacket);
                    return;
                }
            }
        }
//	}
	
	HLPacket taskErrorPacket(HTLS_HDR_TASK, inPacket.GetID(), 1);
	SendPacket(taskErrorPacket);
}

/* --- End of News Shite --- */
#endif // CONFIG_THREADEDNEWS

bool HLClient::AddChatInvite(HLChat &inChat)
{
	ChatList::iterator iter	=
		find(mChatInviteList.begin(), mChatInviteList.end(), &inChat);
	if (iter == mChatInviteList.end())
	{
		mChatInviteList.push_back(&inChat);
		return true;
	}
	return false;
}

bool HLClient::RemoveChatInvite(HLChat &inChat)
{
	ChatList::iterator iter	=
		find(mChatInviteList.begin(), mChatInviteList.end(), &inChat);
	if (iter != mChatInviteList.end())
	{
		mChatInviteList.remove(&inChat);
		return gServer->IsValidChat(inChat);
	}
	return false;
}


