/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
 *    RemoveType.java
 *    Copyright (C) 2002 Richard Kirkby
 *
 */

package weka.filters.unsupervised.attribute;

import weka.filters.*;

import weka.core.*;
import java.util.Enumeration;
import java.util.Vector;

/** 
 * A filter that removes attributes of a given type.<p>
 *
 * Valid filter-specific options are: <p>
 *
 * -T type <br>
 * Attribute type to delete.
 * Options are "nominal", "numeric", "string" and "date". (default "string")<p>
 *
 * -V<br>
 * Invert matching sense (i.e. only keep specified columns)<p>
 *
 * @author Richard Kirkby (rkirkby@cs.waikato.ac.nz)
 * @version $Revision: 1.4 $
 */
public class RemoveType extends Filter
  implements UnsupervisedFilter, StreamableFilter, OptionHandler {

  /** The attribute filter used to do the filtering */
  protected Remove m_attributeFilter = new Remove();

  /** The type of attribute to delete */
  protected int m_attTypeToDelete = Attribute.STRING;

  /** Whether to invert selection */
  protected boolean m_invert = false;
  
  /** Tag allowing selection of attribute type to delete */
  public static final Tag [] TAGS_ATTRIBUTETYPE = {
    new Tag(Attribute.NOMINAL, weka.LocalString.get("Delete nominal attributes")),
    new Tag(Attribute.NUMERIC, weka.LocalString.get("Delete numeric attributes")),
    new Tag(Attribute.STRING, weka.LocalString.get("Delete string attributes")),
    new Tag(Attribute.DATE, weka.LocalString.get("Delete date attributes"))
      };

  /**
   * Sets the format of the input instances.
   *
   * @param instanceInfo an Instances object containing the input instance
   * structure (any instances contained in the object are ignored - only the
   * structure is required).
   * @return true if the outputFormat may be collected immediately
   * @exception Exception if the inputFormat can't be set successfully 
   */ 
  public boolean setInputFormat(Instances instanceInfo) throws Exception {
    
    super.setInputFormat(instanceInfo);

    int[] attsToDelete = new int[instanceInfo.numAttributes()];
    int numToDelete = 0;
    for (int i=0; i<instanceInfo.numAttributes(); i++) {
      if ((i == instanceInfo.classIndex() && !m_invert)) {
	continue; // skip class
      }
      if (instanceInfo.attribute(i).type() == m_attTypeToDelete)
	attsToDelete[numToDelete++] = i;
    }

    int[] finalAttsToDelete = new int[numToDelete];
    System.arraycopy(attsToDelete, 0, finalAttsToDelete, 0, numToDelete);
    
    m_attributeFilter.setAttributeIndicesArray(finalAttsToDelete);
    m_attributeFilter.setInvertSelection(m_invert);
    
    boolean result = m_attributeFilter.setInputFormat(instanceInfo);
    Instances afOutputFormat = m_attributeFilter.getOutputFormat();
    
    // restore old relation name to hide attribute filter stamp
    afOutputFormat.setRelationName(instanceInfo.relationName());

    setOutputFormat(afOutputFormat);
    return result;
  }

  /**
   * Input an instance for filtering.
   *
   * @param instance the input instance
   * @return true if the filtered instance may now be
   * collected with output().
   */
  public boolean input(Instance instance) {
    
    return m_attributeFilter.input(instance);
  }

  /**
   * Signify that this batch of input to the filter is finished.
   *
   * @return true if there are instances pending output
   */  
  public boolean batchFinished() throws Exception {

    return m_attributeFilter.batchFinished();
  }

  /**
   * Output an instance after filtering and remove from the output queue.
   *
   * @return the instance that has most recently been filtered (or null if
   * the queue is empty).
   */
  public Instance output() {

    return m_attributeFilter.output();
  }

  /**
   * Output an instance after filtering but do not remove from the
   * output queue.
   *
   * @return the instance that has most recently been filtered (or null if
   * the queue is empty).
   */
  public Instance outputPeek() {

    return m_attributeFilter.outputPeek();
  }

  /**
   * Returns the number of instances pending output
   *
   * @return the number of instances  pending output
   */  
  public int numPendingOutput() {
  
    return m_attributeFilter.numPendingOutput();
  }
  
  /**
   * Returns whether the output format is ready to be collected
   *
   * @return true if the output format is set
   */  
  public boolean isOutputFormatDefined() {

    return m_attributeFilter.isOutputFormatDefined();
  }

  /**
   * Returns an enumeration describing the available options.
   *
   * @return an enumeration of all the available options.
   */
  public Enumeration listOptions() {

    Vector newVector = new Vector(2);

    newVector.addElement(new Option(
				    weka.LocalString.get("\tAttribute type to delete. Valid options are \"nominal\", ")
				    + weka.LocalString.get("\"numeric\", \"string\" and \"date\". (default \"string\")"),
				    "T", 1, weka.LocalString.get("-T <nominal|numeric|string|date>")));
    newVector.addElement(new Option(
	      weka.LocalString.get("\tInvert matching sense (i.e. only keep specified columns)"),
              "V", 0, "-V"));


    return newVector.elements();
  }

  /**
   * Parses the options for this object. Valid options are: <p>
   *
   * -T type <br>
   * Attribute type to delete.
   * Options are "nominal", "numeric", "string" and "date". (default "string")<p>
   *
   * -V<br>
   * Invert matching sense (i.e. only keep specified columns)<p>
   *
   *
   * @param options the list of options as an array of strings
   * @exception Exception if an option is not supported
   */
  public void setOptions(String[] options) throws Exception {
    
    String tString = Utils.getOption('T', options);
    if (tString.length() != 0) setAttributeTypeString(tString);
    setInvertSelection(Utils.getFlag('V', options));

    if (getInputFormat() != null) {
      setInputFormat(getInputFormat());
    }
  }

  /**
   * Gets the current settings of the filter.
   *
   * @return an array of strings suitable for passing to setOptions
   */
  public String [] getOptions() {

    String [] options = new String [3];
    int current = 0;

    if (getInvertSelection()) {
      options[current++] = "-V";
    }
    options[current++] = "-T";
    options[current++] = getAttributeTypeString();
    
    while (current < options.length) {
      options[current++] = "";
    }
    return options;
  }

  /**
   * Returns a string describing this filter
   *
   * @return a description of the filter suitable for
   * displaying in the explorer/experimenter gui
   */
  public String globalInfo() {

    return weka.LocalString.get("Removes attributes of a given type.");
  }

  /**
   * Returns the tip text for this property
   *
   * @return tip text for this property suitable for
   * displaying in the explorer/experimenter gui
   */
  public String attributeTypeTipText() {

    return weka.LocalString.get("The type of attribute to remove.");
  }

  /**
   * Sets the attribute type to be deleted by the filter.
   *
   * @param type a TAGS_ATTRIBUTETYPE of the new type the filter should delete
   */
  public void setAttributeType(SelectedTag type) {
    
    if (type.getTags() == TAGS_ATTRIBUTETYPE) {
      m_attTypeToDelete = type.getSelectedTag().getID();
    }
  }

  /**
   * Gets the attribute type to be deleted by the filter.
   *
   * @return the attribute type as a selected tag TAGS_ATTRIBUTETYPE
   */
  public SelectedTag getAttributeType() {

    return new SelectedTag(m_attTypeToDelete, TAGS_ATTRIBUTETYPE);
  }

  /**
   * Returns the tip text for this property
   *
   * @return tip text for this property suitable for
   * displaying in the explorer/experimenter gui
   */
  public String invertSelectionTipText() {

    return weka.LocalString.get("Determines whether action is to select or delete.")
      + weka.LocalString.get(" If set to true, only the specified attributes will be kept;")
      + weka.LocalString.get(" If set to false, specified attributes will be deleted.");
  }

  /**
   * Get whether the supplied columns are to be removed or kept
   *
   * @return true if the supplied columns will be kept
   */
  public boolean getInvertSelection() {

    return m_invert;
  }

  /**
   * Set whether selected columns should be removed or kept. If true the 
   * selected columns are kept and unselected columns are deleted. If false
   * selected columns are deleted and unselected columns are kept.
   *
   * @param invert the new invert setting
   */
  public void setInvertSelection(boolean invert) {

    m_invert = invert;
  }

  /**
   * Gets the attribute type to be deleted by the filter as a string.
   *
   * @return the attribute type as a String
   */
  protected String getAttributeTypeString() {

    if (m_attTypeToDelete == Attribute.NOMINAL) return "nominal";
    else if (m_attTypeToDelete == Attribute.NUMERIC) return "numeric";
    else if (m_attTypeToDelete == Attribute.STRING) return "string";
    else if (m_attTypeToDelete == Attribute.DATE) return "date";
    else return "unknown";
  }

  /**
   * Sets the attribute type to be deleted by the filter.
   *
   * @param type a String representing the new type the filter should delete
   */
  protected void setAttributeTypeString(String typeString) {

    typeString = typeString.toLowerCase();
    if (typeString.equals("nominal")) m_attTypeToDelete = Attribute.NOMINAL;
    else if (typeString.equals("numeric")) m_attTypeToDelete = Attribute.NUMERIC;
    else if (typeString.equals("string")) m_attTypeToDelete = Attribute.STRING;
    else if (typeString.equals("date")) m_attTypeToDelete = Attribute.DATE;
  }

  /**
   * Main method for testing this class.
   *
   * @param argv should contain arguments to the filter: use -h for help
   */
  public static void main(String [] argv) {

    try {
      if (Utils.getFlag('b', argv)) {
 	Filter.batchFilterFile(new RemoveType(), argv); 
      } else {
	Filter.filterFile(new RemoveType(), argv);
      }
    } catch (Exception ex) {
      System.out.println(ex.getMessage());
    }
  }
}
