// --------------------------------------------------------------------
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

package tinylib.primitives;

// Cube class
//
// # of Vertexes: 8
// # of Faces(triangle): 12
//
// UV mapping:
//  +--------+
//  |\ |\ |\ |  
//  | \| \| \|
//  +--------+
//  |\ |\ |\ |
//  | \| \| \|
//  +--------+
//
// Faces and UV positions:
//    0- 1- 2  1- 3- 2 - top-left      (0-1-2-3)  0  1  2  3
//    4- 5- 6  6- 5- 7 - top-middle    (0-4-1-5)  4  5  6  7
//    8- 9-10 10- 9-11 - top-right     (0-4-2-6)  8  9 10 11 
//   12-13-14 13-15-14 - bottom-left   (2-3-6-7) 12 13 14 15
//   16-17-18 18-17-19 - bottom-middle (1-5-3-7) 16 17 18 19
//   20-21-22 22-21-23 - bottom-right  (4-6-5-7) 20 21 22 23

/**
  A Cube
**/

class Cube3D extends Polygon {
  public function new( ?x:Float = 1.0,
                       ?y:Float = 1.0,
                       ?z:Float = 1.0 ) {
    super();
    x *= 0.5;
    y *= 0.5;
    z *= 0.5;

    // vertexes
    var v:Array< Point3D > = new Array< Point3D >();
    v.push( new Point3D( -x, -y, -z ) ); // 0: ---
    v.push( new Point3D( -x, -y,  z ) ); // 1: --+
    v.push( new Point3D( -x,  y, -z ) ); // 2: -+-
    v.push( new Point3D( -x,  y,  z ) ); // 3: -++
    v.push( new Point3D(  x, -y, -z ) ); // 4: +--
    v.push( new Point3D(  x, -y,  z ) ); // 5: +-+
    v.push( new Point3D(  x,  y, -z ) ); // 6: ++-
    v.push( new Point3D(  x,  y,  z ) ); // 7: +++
    for ( i in [ 0, 1, 2, 3, 0, 4, 1, 5, 0, 4, 2, 6,
                 2, 3, 6, 7, 1, 5, 3, 7, 4, 6, 5, 7 ] ) {
      addVertexPos( v[i] );
    }

    // uv vertexes
    var uv00:UVCoord = new UVCoord(  0.0, 0.0 );
    var uv01:UVCoord = new UVCoord( 0.33, 0.0 );
    var uv02:UVCoord = new UVCoord( 0.66, 0.0 );
    var uv03:UVCoord = new UVCoord( 1.00, 0.0 );
    var uv10:UVCoord = new UVCoord(  0.0, 0.5 );
    var uv11:UVCoord = new UVCoord( 0.33, 0.5 );
    var uv12:UVCoord = new UVCoord( 0.66, 0.5 );
    var uv13:UVCoord = new UVCoord( 1.00, 0.5 );
    var uv20:UVCoord = new UVCoord(  0.0, 1.0 );
    var uv21:UVCoord = new UVCoord( 0.33, 1.0 );
    var uv22:UVCoord = new UVCoord( 0.66, 1.0 );
    var uv23:UVCoord = new UVCoord( 1.00, 1.0 );

    // register faces
    addFace( new Face(  0,  1,  2, uv00, uv01, uv11 ) );
    addFace( new Face(  1,  3,  2, uv00, uv10, uv11 ) );
    addFace( new Face(  4,  5,  6, uv01, uv11, uv12 ) );
    addFace( new Face(  6,  5,  7, uv01, uv02, uv12 ) );
    addFace( new Face(  8, 10,  9, uv02, uv13, uv12 ) );
    addFace( new Face( 10, 11,  9, uv02, uv13, uv03 ) );
    addFace( new Face( 12, 13, 14, uv10, uv20, uv21 ) );
    addFace( new Face( 13, 15, 14, uv10, uv11, uv21 ) );
    addFace( new Face( 16, 17, 18, uv11, uv21, uv22 ) );
    addFace( new Face( 18, 17, 19, uv11, uv12, uv22 ) );
    addFace( new Face( 20, 21, 22, uv12, uv22, uv23 ) );
    addFace( new Face( 22, 21, 23, uv12, uv13, uv23 ) );
  }
}
