// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

/**
  Container for the parameters used in wm3d.
  Almost all variables are referenced by parent class, Watermelon
  Thus, most variables should be defined as public.

  Variables in this class may not be modified by standard usage unlike
  those in WMStates.
**/

class WMParams {
  private var wm:Watermelon;

  /**
    culling
  **/
  @:isVar public var doCulling( get, set ):Bool;
    /**
      getter for `doCulling`
    **/
    public function get_doCulling():Bool { return( doCulling ); }
    /**
      setter for `doCulling`
    **/
    public function set_doCulling( c:Bool ) {
      doCulling = c;
      return( doCulling );
    }

  /**
    debug message color
  **/
  @:isVar public var traceColor( get, set ):Int;
    /**
      getter for `traceColor`
    **/
    public function get_traceColor():Int { return( traceColor ); }
    /**
      setter for `traceColor`
    **/
    public function set_traceColor( c:Int ) {
      traceColor = c;
      return( traceColor );
    }

  /**
    auto-rotation degrees per frame
  **/
  @:isVar public var arDegree( get, set ):Float;
    /**
      getter of `arDegree`
    **/
    public function get_arDegree():Float { return( arDegree ); }
    /**
      setter of `arDegree`
    **/
    public function set_arDegree( d:Float ):Float {
      arDegree = Math.max( d, 0.0 );
      if ( arDegree == 0.0 && wm.states != null ) {
        wm.states.stopAutoRotation();
      }
      return( arDegree );
    }

  /**
    mouse wheel scaling coefficient
  **/
  @:isVar public var scaleWheel( get, set ):Float;
    /**
      getter of `scaleWheel`
    **/
    public function get_scaleWheel():Float { return( scaleWheel ); }
    /**
      setter of `scaleWheel`
    **/
    public function set_scaleWheel( s:Float ) {
      scaleWheel = Math.max( s, 0.0 );
      return( scaleWheel );
    }

  /**
    coordinate auto-scaling coefficient
  **/
  @:isVar public var scaleFactorAuto( get, set ):Float;
    /**
      getter for `scaleFactorAuto`
    **/
    public function get_scaleFactorAuto():Float { return( scaleFactorAuto ); }
    /**
      setter for `scaleFactorAuto`
    **/
    public function set_scaleFactorAuto( f:Float ) {
      scaleFactorAuto = Math.max( f, 0.0 );
      return( scaleFactorAuto );
    }

  /**
    coodrinate manually scaling factor
  **/
  @:isVar public var scaleFactorManual( get, set ):Float;
    /**
      getter of `scaleFactorManual`
    **/
    public function get_scaleFactorManual():Float { return( scaleFactorManual ); }
    /**
      setter of `scaleFactorManual`
    **/
    public function set_scaleFactorManual( f:Float ):Float {
      scaleFactorManual = Math.max( f, 0.0 );
      return( scaleFactorManual );
    }

  /**
    unit length for camera position translation
  **/
  @:isVar public var depthWheel( get, set ):Float;
    /**
      getter for `depthWheel`
    **/
    public function get_depthWheel():Float { return( depthWheel ); }
    /**
      setter for `depthWheel`
    **/
    public function set_depthWheel( f:Float ) {
      depthWheel = Math.max( f, 0.0 );
      return( depthWheel );
    }

  /**
    framerate for playing; meaningful only if there are multiple scenes
  **/
  @:isVar public var playFrameRate( get, set ):Float;
    /**
      getter of `playFrameRate`
    **/
    public function get_playFrameRate():Float { return( playFrameRate ); }
    /**
      setter of `playFrameRate`
    **/
    public function set_playFrameRate( r:Float ):Float {
      playFrameRate = Math.max( r, 0.0 );
      return( playFrameRate );
    }

  /**
    number of frames displayed for a scene
    this value is determined by the framerate of the flash and framerate of
    `playFrameRate` variable.
  **/
  @:isVar public var numFramesPerScene( get, set ):Int;
    /**
      getter of `numFramesPerScene`
    **/
    public function get_numFramesPerScene():Int { return( numFramesPerScene ); }
    /**
      setter of `numFramesPerScene`
    **/
    public function set_numFramesPerScene( n:Int ) {
      numFramesPerScene = n;
      return( numFramesPerScene );
    }

  /**
    coefficient for rotation degrees in mouse dragging (only in rotation mode)
  **/
  @:isVar public var mouseMoveRotatScale( get, set ):Float;
    /**
      getter of `mouseMoveRotatScale`
    **/
    public function get_mouseMoveRotatScale():Float { return( mouseMoveRotatScale ); }
    /**
      setter of `mouseMoveRotatScale`
    **/
    public function set_mouseMoveRotatScale( s:Float ) {
      mouseMoveRotatScale = s;
      return( mouseMoveRotatScale );
    }

  /**
    coefficient for scaling in mouse dragging (only in scale mode)
  **/
  @:isVar public var mouseMoveScaleScale( get, set ):Float;
    /**
      getter of `mouseMoveScaleScale`
    **/
    public function get_mouseMoveScaleScale():Float { return( mouseMoveScaleScale ); }
    /**
      setter of `mouseMoveScaleScale`
    **/
    public function set_mouseMoveScaleScale( s:Float ):Float {
      mouseMoveScaleScale = s;
      return( mouseMoveScaleScale );
    }

  /**
    Whether disable, saving, editing xml data.
  **/
  @:isVar public var protectData( get, set ):Bool;
    /**
      getter for `protectData`
    **/
    public function get_protectData():Bool { return( protectData ); }
    /**
      setter for `protectData`
    **/
    public function set_protectData( f:Bool ) {
      protectData = f;
      return( protectData );
    }

  /**
    whether read molecules in the background
  **/
  @:isVar public var readBackground( get, set ):Bool;
    /**
      getter of `readBackground`
    **/
    public function get_readBackground():Bool { return( readBackground ); }
    /**
      setter of `readBackground`
    **/
    public function set_readBackground( f:Bool ) {
      readBackground = f;
      return( readBackground );
    }

  /**
    whether inherit scaling parameter from the first SCENE
  **/
  @:isVar public var inheritScale( get, set ):Bool;
    /**
      getter of `inheritScale`
    **/
    public function get_inheritScale():Bool { return( inheritScale ); }
    /**
      setter of `inheritScale`
    **/
    public function set_inheritScale( f:Bool ) {
      inheritScale = f;
      return( inheritScale );
    }

  /**
    whether activate auto scaling when loading
  **/
  @:isVar public var doAutoScale( get, set ):Bool;
    /**
      getter for `doAutoScale`
    **/
    public function get_doAutoScale():Bool { return( doAutoScale ); }
    /**
      setter for `doAutoScale`
    **/
    public function set_doAutoScale( f:Bool ):Bool {
      doAutoScale = f;
      return( doAutoScale );
    }

  /**
    is depth cueing active?
  **/
  @:isVar public var dcActive( get, set ):Bool;
    /**
      getter of `dcActive`
    **/
    public function get_dcActive():Bool { return( dcActive ); }
    /**
      setter of `dcActive`
    **/
    public function set_dcActive( a:Bool ):Bool {
      dcActive = a;
      return( dcActive );
    }

  /**
    depth cueing coefficient
  **/
  @:isVar public var dcCoeff( get, set ):Float;
    /**
      getter of `dcCoeff`
    **/
    public function get_dcCoeff():Float { return( dcCoeff ); }
    /**
      setter of `dcCoeff`
    **/
    public function set_dcCoeff( c:Float ):Float {
      dcCoeff = c;
      return( dcCoeff );
    }

  /**
    depth cueing characteristic length
  **/
  @:isVar public var dcLength( get, set ):Float;
    /**
      getter of `dcLength`
    **/
    public function get_dcLength():Float { return( dcLength ); }
    /**
      setter of `dcLength`
    **/
    public function set_dcLength( l:Float ):Float {
      dcLength = l;
      return( dcLength );
    }

  // ######################################################################
  /**
    Constructor.

    Default values:

    - doCulling; true
    - traceColor: 0xFF0000
    - arDegree: 1.5
    - scaleWheel: 0.01
    - depthWheel: 0.04 * max( stageWidth, stageHeight )
    - playFrameRate: 3.0 // 3 frames per sec (at most)
    - mouseMoveRotatScale: 30.0
    - mouseMoveScaleScale: 15.0
    - protectData: false
    - readBackground: false
    - numFramesPerScene: -1
    - inheritScale: true
    - doAutoScale: true
    - scaleFactorAuto: 0.3
    - scaleFactorManual: 10.0
    - dcActive: true
    - dcCoeff: 4.0 / min( stageWidth, stageHeight )
    - dcLength: -0.2 * min( stageWidth, stageHeight )
  **/
  public function new( w:Watermelon ) {
    wm = w;
    // default values
    doCulling = true;
    traceColor = 0xFF0000;

    arDegree = 1.5;
    scaleWheel = 0.01;
    depthWheel = 0.04 * Math.max( wm.stage.stageWidth, wm.stage.stageHeight );
    playFrameRate = 3.0; // 3 frames per sec (at most)

    mouseMoveRotatScale = 30.0;
    mouseMoveScaleScale = 15.0;

    protectData = false;
    readBackground = false;
    numFramesPerScene = -1;

    inheritScale = true;
    doAutoScale = true;
    scaleFactorAuto = 0.3;
    scaleFactorManual = 10.0;

    dcActive = true;
    dcCoeff = 4.0 / Math.min( wm.stage.stageWidth, wm.stage.stageHeight );
    dcLength = -0.2 * Math.min( wm.stage.stageWidth, wm.stage.stageHeight );
  }

  /**
    actual setter of numFramesPerScene
  **/
  public function assignNumFramesPerScene():Void {
    numFramesPerScene = Std.int( wm.stage.frameRate / playFrameRate );
  }
}
