// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

import flash.text.TextField;
import flash.text.TextFormat;
import flash.display3D.Context3D;

import flash.xml.XML;
import flash.xml.XMLList;
import flash.xml.XMLParser;

import tinylib.Point3D;
import tinylib.primitives.Text3D;

/**
  Label object; a text

  See `WMBase` and `WMObjBase` classes for inheritted parameters.
**/

class WMLabel extends WMObjBase {
  /**
    font name used for text. Default is "Arial"
  **/
  @:isVar public var font( get, set ):String;
    /**
      getter for `font`
    **/
    public function get_font():String { return( font ); }
    /**
      setter for `font`
    **/
    public function set_font( fn:String ):String {
      font = fn;
      return( font );
    }
  /**
    font size. default is 32.0
  **/
  @:isVar public var fontsize( get, set ):Float;
    /**
      getter for `fontsize`
    **/
    public function get_fontsize():Float { return( fontsize ); }
    /**
      setter for `fontsize`
    **/
    public function set_fontsize( fs:Float ):Float {
      fontsize = fs;
      return( fontsize );
    }
  /**
    text to be shown. no default.
  **/
  @:isVar public var text( get, set ):String;
    /**
      getter of `text`
    **/
    public function get_text():String { return( text ); }
    /**
      setter of `text`
    **/
    public function set_text( t:String ):String {
      text = t;
      return( t );
    }
  /**
    position of the text. no default.
  **/
  @:isVar public var pos( get, set ):Point3D;
    /**
      getter of `pos`
    **/
    public function get_pos():Point3D { return( pos ); }
    /**
      setter of `pos`
    **/
    public function set_pos( p:Point3D ):Point3D {
      if ( p != null ) pos = p.clone();
      return( pos );
    }
  /**
    original position; not intensively used.
  **/
  @:isVar public var opos( get, set ):Point3D;
    /**
      getter of `opos`
    **/
    public function get_opos():Point3D { return( opos ); }
    /**
      setter of `opos`
    **/
    public function set_opos( p:Point3D ):Point3D {
      if ( p != null ) opos = p.clone();
      return( opos );
    }

  // ####################################################################

  /**
    Constructor. Parameters are listed below, where values in parenthesis
    are default values.

    - r: width of the textfield (130.0)
    - c0: text color (0xFFFFFF, white)
    - c1: text color (0xFFFFFF, white), not used now
    - a: alpha (1.0, opaque)
    - o: offset (0)
    - q: quality (0), not used
    - ag: ambient intensity (0), not used
    - d: diffuse intensity (0), not used
    - sp: specular (0), not used
    - gl: gloss (20.0), not used
    - sh: shader type ("2DUV"), should not be changed
  **/
  public function new( ?r:Float = 130.0,
                       ?c0:Int = 0xFFFFFF,
                       ?c1:Int = 0xFFFFFF,
                       ?a:Float = 1.0,
                       ?o:Float = 0.0,
                       ?q:Int = 0,
                       ?ag:Float = 0.0,
                       ?d:Float = 0.0,
                       ?sp:Float = 0.0,
                       ?gl:Float = 20.0,
                       ?sh:String = "2DUV" ) {
    super( r, c0, c1, a, o, q, ag, d, sp, gl, sh );
    pos = null;
    opos = null;
    font = "Arial";
    fontsize = 32.0;
    text = "";
  }

  /**
    returns a copy of `this` instance
  **/
  public function clone():WMLabel {
    var ret:WMLabel = new WMLabel( radius, color0, color1, alpha, offset, quality, ambient, diffuse, specular, gloss, shader );
    ret.font = font;
    ret.fontsize = fontsize;
    return( ret );
  }

  /**
    initialize values; if `def` is given, that is used as initial value.
  **/
  public override function clear( ?def:WMDefaults = null ):Void {
    if ( def != null ) {
      copyFrom( def.Label );
      font = def.LabelFont;
      fontsize = def.LabelSize;
    } else {
      radius = 130.0;
      alpha = 1.0;
      font = "Arial";
      fontsize = 32.0;
    }
    pos = null;
    opos = null;
    text = "";
    _polygon = null;
  }

  /**
    load from XML; if `def` is given, that is used as initial value. See also
    `clear` function of this class.
  **/
  public override function loadFromXml( x:Xml,
                                        ?def:WMDefaults = null ):Void {
    clear( def );
    loadFromXmlWOClear( x );
  }

  /**
    read XML while overriding default values; this function does not read
    WMAtom specific fields. Use `loadFromXmlWOClear` when loading WMAtom
    specific fields.
  **/
  public function loadFromXmlOverwrite( x:Xml ):Void {
    super.loadFromXml( x );
    if ( x.exists( "font" ) ) font = x.get( "font" );
    if ( x.exists( "fontsize" ) ) fontsize = Std.parseFloat( x.get( "fontsize" ) );
    shader = "2DUV"; // only applicable shader
  }

  /**
    load XML data; usually this function is called by loadFromXml function
  **/
  public function loadFromXmlWOClear( x:Xml ):Void {
    loadFromXmlOverwrite( x );
    if ( !x.exists( "pos" ) || !x.exists( "text" ) ) {
      trace( "pos and text attributes are required for a LABEL element." );
      return;
    }
    pos = Point3D.fromStringInverted( x.get( "pos" ) );
    opos = pos;
    text = x.get( "text" );
  }

  /**
    generate polygon and set corresponding shaders;
    this function calls `pregen` and `gen2`.
  **/
  public function gen( c:Context3D,
                       ?is_dc_active = false ):Void {
    pregen();
    gen2( c, is_dc_active );
  }

  /**
    preparation for generating polygon. This function is explicitly used in
    non-primordial Worker, where Context3D of the primordial Worker is not
    available. If Worker is not used, this function should not be called
    explicitly.
  **/
  public function pregen() {}

  /**
    generate polygon and shader. Call `pregen` function before.
    This function should not be called explicitly unless you are using
    multiple Workers.
  **/
  public function gen2( c:Context3D,
                        ?is_dc_active = false ) {
    var tf:TextField = new TextField();
    tf.text = text;
    tf.autoSize = flash.text.TextFieldAutoSize.LEFT;
    tf.selectable = false;
    // larger font size might be better to draw smooth character
    tf.setTextFormat( new TextFormat( font, fontsize, color0, true, null, null, null, flash.text.TextFormatAlign.LEFT ) );
    var aspect:Float = tf.width / tf.height;
    //_polygon = new Text3D( tf.width * scale, tf.height * scale );
    // ad hoc scaling for width
    _polygon = new Text3D( 0.75 * aspect * radius, radius );
    _polygon.translate( pos.x, pos.y, pos.z );
    _polygon.createTextTexture( c, tf );
    _polygon.allocate( c, is_dc_active, shader, color0, alpha );
  }

  /**
    number of elements inside; return always 1.
  **/
  public function num():Int { return(1); }
  /**
    sum of position
  **/
  public function sumPos():Point3D { return( pos ); }
  /**
    translate coordinate by `p`
  **/
  public function translate( p:Point3D ):Void { pos.add( p ); }
  /**
    absolute max of position: max(abs(x),abs(y),abs(z))
  **/
  public function absmax():Point3D {
    return( new Point3D( Math.abs( pos.x ), Math.abs( pos.y ), Math.abs( pos.z ) ) );
  }
  /**
    scale position by `scale`
  **/
  public function scaleCoord( scale:Float ):Void {
    pos.multiply( scale );
  }
  /**
    predefined cost for drawing; returns always 3.
  **/
  public override function getDataSize():Int { return(3); }
}
