// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2014, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

import flash.display3D.Context3D;

import flash.xml.XML;
import flash.xml.XMLList;
import flash.xml.XMLParser;

import tinylib.Point3D;
import tinylib.primitives.Sphere3D;

/**
  Atom object; a sphere.

  See `WMBase` and `WMObjBase` classes for inheritted parameters.
**/

class WMAtom extends WMObjBase {
  // variables and their getter/header
  /**
    position of atom; no default
  **/
  @:isVar public var pos( get, set ):Point3D;
    /**
      getter for `pos`
    **/
    public function get_pos():Point3D { return( pos ); }
    /**
      setter for `pos`
    **/
    public function set_pos( p:Point3D ):Point3D {
      if ( p != null ) {
        pos = p.clone();
      }
      return( pos );
    }
  /**
    old position; currently unused, no default
  **/
  @:isVar public var opos( get, set ):Point3D;
    /**
      getter for `opos`
    **/
    public function get_opos():Point3D { return( opos ); }
    /**
      setter for `opos`
    **/
    public function set_opos( p:Point3D ):Point3D {
      if ( p != null ) opos = p.clone();
      return( opos );
    }

  // #######################################################################

  /**
    Constructor. Parameters are listed below, where values in parenthesis
    are default values.

    - r: radius (1.0)
    - c0: color (0x00FF00; lime)
    - c1: color, currently not used (0x00FF00, lime)
    - a: alpha (1.0, opaque)
    - o: offset, not used (0.0)
    - q: quality, higher is better (3)
    - ag: ambient intensity (0.35)
    - d: diffuse intensity (0.6)
    - sp: specular intensity (0.3)
    - gl: gloss (30.0)
    - sh: shader type ("Phong")
  **/
  public function new( ?r:Float = 1.0,
                       ?c0:Int = 0x00FF00,
                       ?c1:Int = 0x00FF00,
                       ?a:Float = 1.0,
                       ?o:Float = 0.0,
                       ?q:Int = 3,
                       ?ag:Float = 0.35,
                       ?d:Float = 0.6,
                       ?sp:Float = 0.3,
                       ?gl:Float = 30.0,
                       ?sh:String = "Phong" ) {
    super( r, c0, c1, a, o, q, ag, d, sp, gl, sh );
    pos = null;
    opos = null;
    _polygon = null;
  }

  /**
    returns clone of `this` instance
  **/
  public function clone():WMAtom {
    var ret:WMAtom = new WMAtom( radius, color0, color1, alpha, offset, quality, ambient, diffuse, specular, gloss, shader );
    return( ret );
  }

  /**
    initialize values; if `def` is given, that is used as initial value.
  **/
  public override function clear( ?def:WMDefaults = null ):Void {
    if ( def != null ) {
      copyFrom( def.Atom );
    } else {
      radius = 1.0;
      color0 = 0x00FF00;
      color1 = 0x00FF00;
      alpha = 1.0;
      offset = 0.0;
      quality = 3;
      ambient = 0.35;
      diffuse = 0.6;
      specular = 0.3;
      gloss = 30.0;
      shader = "Phong";
    }
    pos = null;
    opos = null;
    _polygon = null;
  }

  /**
    load from XML; if `def` is given, that is used as initial value. See also
    `clear` function of this class.
  **/
  public override function loadFromXml( x:Xml,
                                        ?def:WMDefaults = null ):Void {
    clear( def );
    loadFromXmlWOClear( x );
  }

  /**
    read XML while overriding default values; this function does not read
    WMAtom specific fields. Use `loadFromXmlWOClear` when loading WMAtom
    specific fields.
  **/
  public function loadFromXmlOverwrite( x:Xml ):Void {
    super.loadFromXml( x );
  }

  /**
    load XML data; usually this function is called by loadFromXml function
  **/
  public function loadFromXmlWOClear( x:Xml ):Void {
    loadFromXmlOverwrite( x );
    if ( !x.exists( "pos" ) ) {
      trace( "pos attribute is required for an ATOM" );
      return;
    }
    pos = Point3D.fromStringInverted( x.get( "pos" ) );
    opos = pos;
  }

  /**
    generate polygon and set corresponding shaders;
    this function calls `pregen` and `gen2`.
  **/
  public function gen( c:Context3D,
                       ?is_dc_active:Bool = false ):Void {
    pregen();
    gen2( c, is_dc_active );
  }

  /**
    preparation for generating polygon. This function is explicitly used in
    non-primordial Worker, where Context3D of the primordial Worker is not
    available. If Worker is not used, this function should not be called
    explicitly.
  **/
  public function pregen():Void {
    _polygon = new Sphere3D( WMBase.getRelative( radius ), quality );
    _polygon.translate( pos.x, pos.y, pos.z );
    setShaderParams();
  }

  /**
    generate polygon and shader. Call `pregen` function before.
    This function should not be called explicitly unless you are using
    multiple Workers.
  **/
  public function gen2( c:Context3D,
                        ?is_dc_active:Bool = false ):Void {
    _polygon.allocate( c, is_dc_active, shader, color0, alpha );
  }

  /**
    number of elements inside; return always 1.
  **/
  public function num():Int { return(1); }
  /**
    sum of position
  **/
  public function sumPos():Point3D { return( pos ); }
  /**
    translate coordinate by `p`
  **/
  public function translate( p:Point3D ):Void { pos.add( p ); }
  /**
    absolute max of position: max(abs(x),abs(y),abs(z))
  **/
  public function absmax():Point3D {
    var ret:Point3D = new Point3D( 0, 0, 0 );
    ret.x = Math.abs( pos.x );
    ret.y = Math.abs( pos.y );
    ret.z = Math.abs( pos.z );
    return( ret );
  }
  /**
    scale position by `scale`
  **/
  public function scaleCoord( scale:Float ):Void {
    pos.multiply( scale );
  }
  /**
    predefined cost for drawing; return always 10.
  **/
  public override function getDataSize():Int { return( 10 ); }
}
