// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

class PdbResidue extends ResidueBase {
  public var atoms( get, null ):Map< String, PdbAtom >;
    public function get_atoms():Map< String, PdbAtom > { return( atoms ); }

  @:isVar public var secondary( get, set ):Int;
    public function get_secondary():Int { return( secondary ); }
    public function set_secondary( s:Int ):Int {
      secondary = s;
      return( secondary );
    }

  public var rama( get, null ):RamaAngles;
    public function get_rama():RamaAngles { return( rama ); }

  // ########################################################################

  public function new( ?n:String = "",
                       ?i:Int = 1,
                       ?isnt:Bool = false,
                       ?isct:Bool = false,
                       ?sec:Int = 0 ) {
    super( n, i, isnt, isct );
    clearAtoms();
    secondary = sec;
    rama = new RamaAngles();
  }

  public function clearAtoms() {
    atoms = new Map< String, PdbAtom >();
  }

  public function addAtom( a:PdbAtom ):Void {
    if ( a == null ) return;
    if ( !atoms.exists( a.name ) ) {
      atoms.set( a.name, a.clone() );
    }
    // duplicated atoms are simply ignored
  }

  public function getAtom( a:String ):PdbAtom {
    if ( !atoms.exists( a ) ) return( null );
    return( atoms.get( a ) );
  }

  public function isAmino():Bool {
    if ( atoms.exists( "N" ) && atoms.exists( "CA" ) &&
         atoms.exists( "C" ) && atoms.exists( "O" ) ) return( true );
    return( false );
  }

  // write position of alpha-carbon, "CA"
  public function genPoint():String {
    if ( !isAmino() ) return( "" );
    var atom = atoms.get( "CA" );
    if ( atom == null ) return( "" );
    // atom.pos.toString() must not be used;
    // that function flips z-coordinate
    return( "      <POINT index=\"" + index +
            "\" pos=\"" + atom.pos.x +
            " " + atom.pos.y +
            " " + atom.pos.z + "\" />\n" );
  }

  // generate "H" atom, which binds to backbone "N" atom.
  // the parameter c corresponds to the position of "C" atom in the previous
  // residue. thus, "H" atom never be created for first residue
  public function genH( ?r:PdbResidue = null ):Void {
    if ( atoms.exists( "H" ) ) return;
    if ( r == null ) return;
    if ( r.getAtom( "C" ) == null ) return;
    genHatom( r.getAtom( "C" ) );
  }
  public function genHatom( a:PdbAtom ):Void {
    if ( atoms.exists( "H" ) ) return;
    if ( a == null ) return;
    genHpos( a.pos );
  }
  public function genHpos( c:Point3D ):Void {
    if ( atoms.exists( "H" ) ) return;
    // if not an amino acid, do nothing.
    if ( !isAmino() ) return;
    var pos_nc:Point3D = Point3D.getSub( atoms.get( "N" ).pos, c );
    var pos_nca:Point3D = Point3D.getSub( atoms.get( "N" ).pos, atoms.get( "CA" ).pos );
    pos_nc.normalize();
    pos_nca.normalize();
    pos_nc.add( pos_nca );
    pos_nc.normalize();
    pos_nc.multiply( HBParams.NH_dist );
    pos_nc.add( atoms.get( "N" ).pos );
    // add created hydrogen atom to this residue
    addAtom( new PdbAtom( "H", 0, pos_nc, 1.0, 0.0, "H" ) );
  }
}
