// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

// data container for pdb file

// data hierarchy
// Pdb - PdbModel - PdbChain(use chain id) - PdbResidue - PdbAtom
//
// PdbModel   : MODELs which are often used for NMR structure
// PdbChain   : A chain
// PdbResidue : A residue
// PdbAtom    : An atom

// keywords to be interpretted are:
// TER, END, ENDMDL, HELIX, SHEET, ATOM, HETATM

class Pdb {
  // global parameters
  static public var coil_radius:String = "3.5";
  static public var ribbon_radius:String = "10.0";
  // helix params
  static public var helix_color:String = "red";
  static public var helix_thickness:String = "0.2";
  // strand params
  static public var strand_color:String = "blue";
  static public var strand_thickness:String = "0.2";

  // ######################################################################

  public var models( get, null ):Map< Int, PdbModel >;
    public function get_models():Map< Int, PdbModel > { return( models ); }

  // ######################################################################
  // local variables
  private var modelnum:Int;
  private var secs:Array< PdbSecondary >;
  private var prev:Dynamic;

  // ######################################################################

  public function new( ?pdbText:String ) {
    initialize();
    if ( pdbText != null ) readAtOnce( pdbText );
  }

  public function initialize() {
    models = new Map< Int, PdbModel >();

    modelnum = 0;
    secs = new Array< PdbSecondary >();
    prev = null;
  }

  public function empty():Bool {
    if ( !models.iterator().hasNext() ) return( true );
    return( false );
  }

  public function readLine( ?line:String ):Void {
    if ( line.length >= 6 ) {
      var fieldname:String = StringTools.trim( line.substr( 0, 6 ) );
      switch ( fieldname ) {
        case "HELIX":
          var sec:PdbSecondary = PdbSecondary.readHelixFromText( line );
          secs.push( sec );
        case "SHEET":
          var sec:PdbSecondary = PdbSecondary.readStrandFromText( line );
          secs.push( sec );
        case "ATOM", "HETATM":
          var an = PdbAtom.readFromText( line );
          addAtom( modelnum, an.ci, an.rn, an.ri, an.at );
          prev = an;
        case "END", "ENDMDL":
          modelnum++;
      }
    } else if ( line.length >= 3 ) {
      // END statement is assumed to be equivalent to ENDMDL
      if ( line.substr( 0, 3 ) == "END" ) {
        modelnum++;
      }
    }
  }

  public function readAtOnce( ?pdbText:String,
                              ?doPostProcess:Bool = true ):Void {
    if ( pdbText == null ) return;
    for ( line in pdbText.split( '\n' ) ) {
      readLine( line );
    }
    if ( doPostProcess ) postProcess();
  }

  public function postProcess():Void {
    for ( model in models ) {
      model.initializeSecondary();
    }
    // if no secondary structure information is described in the pdb file,
    // assign it.
    if ( secs.length == 0 ) {
      calcSecondaryStructures();
    }
    // assign secondary structure type
    for ( sec in secs ) {
      // secondary structures are assumed to be common for all the models
      for ( model in models ) {
        model.chains.get( sec.chainid ).assignSec( sec );
      }
    }
  }

  public function calcSecondaryStructures() {
    // calculate secondary structure for each chain
    for ( model in models ) {
      for ( chain in model.chains ) {
        chain.calcSecondaryStructures();
      }
    }
  }

  public function addAtom( mn:Int,
                           cid:String,
                           resname:String,
                           resid:Int,
                           atom:PdbAtom ) {
    // add models if necessary
    if ( !models.exists( mn ) ) {
      models.set( mn, new PdbModel( mn ) );
    }
    var mymodel:PdbModel = models.get( mn );
    mymodel.addAtom( cid, resname, resid, atom );
  }

  // generate XML string from PDB data
  public function genXml():String {
    var ret:String = "<WMXML>\n";
    ret += "  <GLOBAL readatonce=\"1\" readchainatonce=\"1\">\n";
    ret += "    <RADIUS method=\"absolute\" />\n";
    ret += "    <AUTOSCALE manual=\"1\" />\n";
    ret += "    <RIBBON radius=\"" + Pdb.ribbon_radius + "\" />\n";
    ret += "    <COIL radius=\"" + Pdb.coil_radius + "\" />\n";
    ret += "  </GLOBAL>\n";
    ret += "  <ALIAS>\n";
    ret += "    <STRAND elem=\"RIBBON\" color=\"" + Pdb.strand_color +
                                "\" thickness=\"" + Pdb.strand_thickness +
                                "\" smoothing=\"1\" />\n";
    ret += "    <HELIX elem=\"RIBBON\" color=\"" + Pdb.helix_color +
                               "\" thickness=\"" + Pdb.helix_thickness +
                               "\" />\n";
    ret += "  </ALIAS>\n";
    for ( model in models ) {
      ret += model.genXml();
    }
    ret += "</WMXML>";
    return( ret );
  }

  public function genSecondaryString():String {
    var ret:String = "";
    for ( model in models ) {
      ret += model.genSecondaryString();
      break; // secondary structures are common for all models
    }
    return( ret );
  }
}
