// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

// Container for the status information used in wm3d.
// Almost all variables are referenced by parent Watermelon.
// So, most variables are defined as public.

import flash.geom.Vector3D;
import flash.geom.Matrix3D;

import flash.events.Event;
import flash.events.MouseEvent;
import flash.events.TimerEvent;

import flash.utils.Timer;

class WMStates {
  // static information
  static public var MOUSE_L_ROTAT_MODE:Int = 0;
  static public var MOUSE_L_TRANS_MODE:Int = 1;
  static public var MOUSE_W_SCALE_MODE:Int = 0;
  static public var MOUSE_W_DEPTH_MODE:Int = 1;

  // parent Watermelon
  private var wm:Watermelon;

  // camera related
  //// camera itself
  @:isVar public var camera( get, set ):Camera;
    public function get_camera():Camera { return( camera ); }
    public function set_camera( c:Camera ):Camera {
      camera = c.clone();
      return( camera );
    }

  //// current coordinate
  @:isVar public var mpos( get, set ):Matrix3D;
    public function get_mpos():Matrix3D { return( mpos ); } // not clone()
    public function set_mpos( m:Matrix3D ):Matrix3D {
      mpos = m.clone();
      updateMPos = true;
      return( mpos );
    }

  //// direction of light
  @:isVar public var light( get, set ):Vector3D;
    public function get_light():Vector3D { return( light ); }
    public function set_light( l:Vector3D ) {
      light = l.clone();
      return( light );
    }

  //// camera position
  @:isVar public var cpos( get, set ):Vector3D;
    public function get_cpos():Vector3D { return( cpos ); }
    public function set_cpos( p:Vector3D ):Vector3D {
      cpos = p.clone();
      return( cpos );
    }

  //// offset of view; i forgot what this is
  @:isVar public var view_offset( get, set ):Vector3D;
    public function get_view_offset():Vector3D { return( view_offset ); }
    public function set_view_offset( o:Vector3D ):Vector3D {
      view_offset = o.clone();
      return( view_offset );
    }

  // auto-rotation status
  //// x
  @:isVar public var arDegreeX( get, set ):Float;
    public function get_arDegreeX():Float { return( arDegreeX ); }
    public function set_arDegreeX( d:Float ):Float {
      arDegreeX = d;
      return( arDegreeX );
    }

  //// y
  @:isVar public var arDegreeY( get, set ):Float;
    public function get_arDegreeY():Float { return( arDegreeY ); }
    public function set_arDegreeY( d:Float ):Float {
      arDegreeY = d;
      return( arDegreeY );
    }

  // mouse mode
  //// left
  @:isVar public var mouseModeL( get, set ):Int;
    public function get_mouseModeL():Int { return( mouseModeL ); }
    public function set_mouseModeL( m:Int ) {
      mouseModeL = m;
      return( mouseModeL );
    }

  //// wheel
  @:isVar public var mouseModeW( get, set ):Int;
    public function get_mouseModeW():Int { return( mouseModeW ); }
    public function set_mouseModeW( m:Int ) {
      mouseModeW = m;
      return( mouseModeW );
    }

  // flags often referenced
  //// whether auto-rotating
  @:isVar public var arNow( get, set ):Bool;
    public function get_arNow():Bool { return( arNow ); }
    public function set_arNow( f:Bool ):Bool {
      arNow = f;
      return( arNow );
    }

  //// whether update of camera position is required
  @:isVar public var updateCameraPos( get, set ):Bool;
    public function get_updateCameraPos():Bool { return( updateCameraPos ); }
    public function set_updateCameraPos( f:Bool ) {
      updateCameraPos = f;
      return( updateCameraPos );
    }

  //// whether update coordinate
  @:isVar public var updateMPos( get, set ):Bool;
    public function get_updateMPos():Bool { return( updateMPos ); }
    public function set_updateMPos( f:Bool ):Bool {
      updateMPos = f;
      return( updateMPos );
    }

  //// whether update offset of view
  @:isVar public var updateViewOffset( get, set ):Bool;
    public function get_updateViewOffset():Bool { return( updateViewOffset ); }
    public function set_updateViewOffset( f:Bool ) {
      updateViewOffset = f;
      return( updateViewOffset );
    }

  //// whether update of scene is needed
  @:isVar public var updateScene( get, set ):Bool;
    public function get_updateScene():Bool { return( updateScene ); }
    public function set_updateScene( f:Bool ):Bool {
      updateScene = f;
      return( updateScene );
    }

  // flags
  //// general flag; whether i am busy now?
  @:isVar public var busyNow( get, set ):Bool;
    public function get_busyNow():Bool { return( busyNow ); }
    public function set_busyNow( f:Bool ):Bool {
      busyNow = f;
      return( busyNow );
    }

  //// whether playing now
  @:isVar public var playingNow( get, set ):Bool;
    public function get_playingNow():Bool { return( playingNow ); }
    public function set_playingNow( f:Bool ):Bool {
      playingNow = f;
      return( playingNow );
    }

  //// whether playing is in remove mode
  @:isVar public var playReverse( get, set ):Bool;
    public function get_playReverse():Bool { return( playReverse ); }
    public function set_playReverse( f:Bool ):Bool {
      playReverse = f;
      return( playReverse );
    }

  // timer
  @:isVar public var myLoadSysTimer( get, null ):Timer;
    public function get_myLoadSysTimer():Timer { return( myLoadSysTimer ); }

  // counter
  //// frame counter for play mode
  @:isVar public var frameCounter( get, set ):Int;
    public function get_frameCounter():Int { return( frameCounter ); }
    public function set_frameCounter( c:Int ):Int {
      frameCounter = c;
      return( frameCounter );
    }

  //// current frame
  @:isVar public var frameIndex( get, set ):Int;
    public function get_frameIndex():Int { return( frameIndex ); }
    public function set_frameIndex( i:Int ):Int {
      frameIndex = i;
      return( frameIndex );
    }

  // #######################################################################
  public function new( w:Watermelon ) {
    wm = w;

    // default values
    arDegreeX = 0.0;
    arDegreeY = wm.params.arDegree; // params must be initialized first
    mouseModeL = WMStates.MOUSE_L_ROTAT_MODE; // rotation mode
    //mouseModeL = WMStates.MOUSE_L_TRANS_MODE; // translation mode
    mouseModeW = WMStates.MOUSE_W_SCALE_MODE; // scale mode
    //mouseModeW = WMStates.MOUSE_W_DEPTH_MODE; // depth mode

    camera = new Camera();
    camera.pos.z = -Math.max( wm.stage.stageWidth, wm.stage.stageHeight ) * 2;
    camera.ratio = wm.stage.stageWidth / wm.stage.stageHeight;
    camera.determineFov( wm.stage.stageHeight, Math.abs( camera.pos.z ) );

    light = new Vector3D( 1, -1, -1 );
    light.normalize();

    cpos = new Vector3D();
    mpos = new Matrix3D();
    mpos.identity();

    view_offset = new Vector3D();

    arNow = true;
    busyNow = false;
    updateCameraPos = true;
    updateMPos = true;
    updateViewOffset = false;
    updateScene = true;

    playingNow = false;
    playReverse = false;

    frameCounter = 0;
    frameIndex = 0;
  }

  public function needToUpdate():Bool {
    return( updateCameraPos || updateMPos || arNow || updateViewOffset ||
            updateScene );
  }

  public function updateCPos():Void {
    cpos.x = camera.pos.x;
    cpos.y = camera.pos.y;
    cpos.z = camera.pos.z;
    updateCameraPos = false;
  }

  public function resetFlags():Void {
    updateCameraPos = false;
    updateMPos = false;
    updateViewOffset = false;
    updateScene = false;
  }

  public function setCameraPosZ( z:Float ):Void {
    camera.pos.z = z;
    updateCameraPos = true;
  }

  public function setLightDirection( p:Point3D ):Void {
    light.x = p.x;
    light.y = p.y;
    light.z = p.z;
    light.normalize();
  }

  // mouse actions relatives
  //// scaling
  public function changeScale( d:Int ):Void {
    var sc:Float = 1.0 + wm.params.scaleWheel * d;
    mpos.appendScale( sc, sc, sc );
    updateMPos = true;
  }

  //// depth change
  public function changeDepth( d:Int ):Void {
    camera.pos.z += wm.params.depthWheel * d;
    camera.pos.z = Math.max( 0, camera.pos.z );
    camera.determineFov( wm.stage.stageHeight, Math.abs( camera.pos.z ) );
    updateCameraPos = true;
  }

  // auto-rotation related
  public function applyAutoRotation() {
    mpos.appendRotation( arDegreeY, flash.geom.Vector3D.Y_AXIS );
    mpos.appendRotation( arDegreeX, flash.geom.Vector3D.X_AXIS );
  }

  public function beginAutoRotation():Void { arNow = true; }
  public function stopAutoRotation():Void { arNow = false; }

  // play mode
  public function pausePlay( ?e:MouseEvent = null ):Void { playingNow = false; }

  public function initPlay():Void {
    if ( wm.systems.length <= 1 ) return;
    wm.params.assignNumFramesPerScene();
    playReverse = false;
    playingNow = true;
    frameCounter = 0;
  }

  public function playForward( ?e:MouseEvent = null ):Void {
    if ( wm.systems.length <= 1 || myLoadSysTimer.running ) return;
    initPlay();
  }

  public function playBackward( ?e:MouseEvent = null ):Void {
    if ( wm.systems.length <= 1 || myLoadSysTimer.running ) return;
    initPlay();
    playReverse = true;
  }

  // scene migration
  public function forwardScene( ?e:MouseEvent = null ):Void {
    if ( myLoadSysTimer.running ) return;
    frameIndex = ( frameIndex + 1 ) % wm.systems.length;
    updateScene = true;
  }

  public function gotoLastScene( ?e:MouseEvent ):Void {
    if ( myLoadSysTimer.running ) return;
    frameIndex = wm.systems.length - 1;
    updateScene = true;
  }

  public function backScene( ?e:MouseEvent = null ):Void {
    if ( myLoadSysTimer.running ) return;
    if ( --frameIndex < 0 ) frameIndex = wm.systems.length + frameIndex;
    updateScene = true;
  }

  public function gotoInitScene( ?e:MouseEvent = null ):Void {
    if ( myLoadSysTimer.running ) return;
    frameIndex = 0;
    updateScene = true;
  }

  // timer (load systems)
  public function beginLoadSysTimer( ?delay:Float = 2.0 ):Void {
    myLoadSysTimer = new Timer( delay );
    myLoadSysTimer.addEventListener( TimerEvent.TIMER, wm.loadSystems );
    myLoadSysTimer.start();
  }

  public function stopLoadSysTimer():Void { myLoadSysTimer.stop(); }
  public function isLoadSysTimerRunning():Bool {
    if ( myLoadSysTimer != null ) {
      if ( myLoadSysTimer.running ) return( true );
    }
    return( false );
  }

  // frame counter
  //// count
  public function processCounter():Void {
    // this function is meaningful only when in play mode
    if ( isLoadSysTimerRunning() || wm.params.numFramesPerScene <= 0 ) return;
    if ( playingNow ) {
      if ( ++frameCounter >= wm.params.numFramesPerScene ) {
        frameCounter = 0;
        if ( playReverse ) {
          backScene();
        } else {
          forwardScene();
        }
      }
    }
  }
}
