// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

// Container for the parameters used in wm3d.
// Almost all variables are referenced by parent class, Watermelon
// Thus, most variables should be defined as public one.

// variables in this class may not be modified by standard usage unlike
// those in WMStates.

class WMParams {
  private var wm:Watermelon;

  // culling
  @:isVar public var doCulling( get, set ):Bool;
    public function get_doCulling():Bool { return( doCulling ); }
    public function set_doCulling( c:Bool ) {
      doCulling = c;
      return( doCulling );
    }

  // trace color
  @:isVar public var traceColor( get, set ):Int;
    public function get_traceColor():Int { return( traceColor ); }
    public function set_traceColor( c:Int ) {
      traceColor = c;
      return( traceColor );
    }

  // auto-rotation parameter
  @:isVar public var arDegree( get, set ):Float;
    public function get_arDegree():Float { return( arDegree ); }
    public function set_arDegree( d:Float ):Float {
      arDegree = Math.max( d, 0.0 );
      if ( arDegree == 0.0 && wm.states != null ) {
        wm.states.stopAutoRotation();
      }
      return( arDegree );
    }

  // scaling parameters
  //// scaling parameter when scaled by mouse wheel
  @:isVar public var scaleWheel( get, set ):Float;
    public function get_scaleWheel():Float { return( scaleWheel ); }
    public function set_scaleWheel( s:Float ) {
      scaleWheel = Math.max( s, 0.0 );
      return( scaleWheel );
    }

  //// scaling parameter for the automatic scaling during loading mol
  @:isVar public var scaleFactorAuto( get, set ):Float;
    public function get_scaleFactorAuto():Float { return( scaleFactorAuto ); }
    public function set_scaleFactorAuto( f:Float ) {
      scaleFactorAuto = Math.max( f, 0.0 );
      return( scaleFactorAuto );
    }

  //// manually defined scaling factor during loading mol
  @:isVar public var scaleFactorManual( get, set ):Float;
    public function get_scaleFactorManual():Float { return( scaleFactorManual ); }
    public function set_scaleFactorManual( f:Float ):Float {
      scaleFactorManual = Math.max( f, 0.0 );
      return( scaleFactorManual );
    }

  // depth change parameter
  @:isVar public var depthWheel( get, set ):Float;
    public function get_depthWheel():Float { return( depthWheel ); }
    public function set_depthWheel( f:Float ) {
      depthWheel = Math.max( f, 0.0 );
      return( depthWheel );
    }

  // framerate for playing (meaningful only if there are multiple scenes)
  @:isVar public var playFrameRate( get, set ):Float;
    public function get_playFrameRate():Float { return( playFrameRate ); }
    public function set_playFrameRate( r:Float ):Float {
      playFrameRate = Math.max( r, 0.0 );
      return( playFrameRate );
    }

  // number of frames displayed for a scene
  // this value is determined by the framerate of the flash and framerate of
  // playFrameRate.
  @:isVar public var numFramesPerScene( get, set ):Int;
    public function get_numFramesPerScene():Int { return( numFramesPerScene ); }
    public function set_numFramesPerScene( n:Int ) {
      numFramesPerScene = n;
      return( numFramesPerScene );
    }

  // scale factor used in rotation with mouse move
  @:isVar public var mouseMoveRotatScale( get, set ):Float;
    public function get_mouseMoveRotatScale():Float { return( mouseMoveRotatScale ); }
    public function set_mouseMoveRotatScale( s:Float ) {
      mouseMoveRotatScale = s;
      return( mouseMoveRotatScale );
    }

  // scale factor used in scaling with mouse move
  @:isVar public var mouseMoveScaleScale( get, set ):Float;
    public function get_mouseMoveScaleScale():Float { return( mouseMoveScaleScale ); }
    public function set_mouseMoveScaleScale( s:Float ):Float {
      mouseMoveScaleScale = s;
      return( mouseMoveScaleScale );
    }

  // parameter-like flags
  // these params may not be modified while viewing structure.
  //// whether prevent edit/save
  @:isVar public var protectData( get, set ):Bool;
    public function get_protectData():Bool { return( protectData ); }
    public function set_protectData( f:Bool ) {
      protectData = f;
      return( protectData );
    }

  //// whether read molecules in the background
  @:isVar public var readBackground( get, set ):Bool;
    public function get_readBackground():Bool { return( readBackground ); }
    public function set_readBackground( f:Bool ) {
      readBackground = f;
      return( readBackground );
    }

  //// whether inherit scaling parameter from the first SCENE
  @:isVar public var inheritScale( get, set ):Bool;
    public function get_inheritScale():Bool { return( inheritScale ); }
    public function set_inheritScale( f:Bool ) {
      inheritScale = f;
      return( inheritScale );
    }

  //// whether activate auto scaling when loading
  @:isVar public var doAutoScale( get, set ):Bool;
    public function get_doAutoScale():Bool { return( doAutoScale ); }
    public function set_doAutoScale( f:Bool ):Bool {
      doAutoScale = f;
      return( doAutoScale );
    }

  //// depth cueing
  @:isVar public var dcActive( get, set ):Bool;
    public function get_dcActive():Bool { return( dcActive ); }
    public function set_dcActive( a:Bool ):Bool {
      dcActive = a;
      return( dcActive );
    }

  @:isVar public var dcCoeff( get, set ):Float;
    public function get_dcCoeff():Float { return( dcCoeff ); }
    public function set_dcCoeff( c:Float ):Float {
      dcCoeff = c;
      return( dcCoeff );
    }

  @:isVar public var dcLength( get, set ):Float;
    public function get_dcLength():Float { return( dcLength ); }
    public function set_dcLength( l:Float ):Float {
      dcLength = l;
      return( dcLength );
    }

  // ######################################################################
  public function new( w:Watermelon ) {
    wm = w;
    // default values
    doCulling = true;
    traceColor = 0xFF0000;

    arDegree = 1.5;
    scaleWheel = 0.01;
    depthWheel = 0.04 * Math.max( wm.stage.stageWidth, wm.stage.stageHeight );
    playFrameRate = 3.0; // 3 frames per sec (at most)

    mouseMoveRotatScale = 30.0;
    mouseMoveScaleScale = 15.0;

    protectData = false;
    readBackground = false;
    numFramesPerScene = -1;

    inheritScale = true;
    doAutoScale = true;
    scaleFactorAuto = 0.3;
    scaleFactorManual = 10.0;

    dcActive = true;
    dcCoeff = 4.0 / Math.min( wm.stage.stageWidth, wm.stage.stageHeight );
    dcLength = -0.2 * Math.min( wm.stage.stageWidth, wm.stage.stageHeight );
  }

  public function assignNumFramesPerScene():Void {
    numFramesPerScene = Std.int( wm.stage.frameRate / playFrameRate );
  }
}
