// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

import flash.display3D.Context3D;

import flash.xml.XML;
import flash.xml.XMLList;
import flash.xml.XMLParser;

// available objects and their params
// Cone: radius, height, quality
// Cube: x, y, z
// Cylinder: radius, height, quality
// Square: width, height

class WMObject3D extends WMObjBase {
  @:isVar public var type( get, set ):String;
    public function get_type():String { return( type ); }
    public function set_type( s:String ):String {
      type = s;
      return( s );
    }
  @:isVar public var xsize( get, set ):Float;
    public function get_xsize():Float { return( xsize ); }
    public function set_xsize( x:Float ):Float {
      xsize = x;
      return( xsize );
    }
  @:isVar public var ysize( get, set ):Float;
    public function get_ysize():Float { return( ysize ); }
    public function set_ysize( y:Float ):Float {
      ysize = y;
      return( ysize );
    }
  @:isVar public var zsize( get, set ):Float;
    public function get_zsize():Float { return( zsize ); }
    public function set_zsize( z:Float ):Float {
      zsize = z;
      return( zsize );
    }
  @:isVar public var pos( get, set ):Point3D;
    public function get_pos():Point3D { return( pos ); }
    public function set_pos( p:Point3D ):Point3D {
      if ( p != null ) pos = p.clone();
      return( pos );
    }
  @:isVar public var dir( get, set ):Point3D;
    public function get_dir():Point3D { return( dir ); }
    public function set_dir( d:Point3D ):Point3D {
      if ( d != null ) dir = d.clone();
      return( dir );
    }

  // ##################################################################

  public function new( ?r:Float = 1.0,
                       ?c0:Int = 0x00FF00,
                       ?c1:Int = 0x00FF00,
                       ?a:Float = 1.0,
                       ?o:Float = 0.0,
                       ?q:Int = 3,
                       ?ag:Float = 0.4,
                       ?d:Float = 0.5,
                       ?sp:Float = 0.2,
                       ?gl:Float = 30.0,
                       ?sh:String = "Phong" ) {
    super( r, c0, c1, a, o, q, ag, d, sp, gl, sh );
    type = "cone";
    xsize = ysize = zsize = 0.0;
    pos = null;
    dir = null;
    _polygon = null;
  }

  public function clone():WMObject3D {
    var ret:WMObject3D = new WMObject3D( radius, color0, color1, alpha, offset, quality, ambient, diffuse, specular, gloss, shader );
    ret.xsize = xsize;
    ret.ysize = ysize;
    ret.zsize = zsize;
    ret.type = type;
    return( ret );
  }

  public override function clear( ?def:WMDefaults = null ):Void {
    if ( def != null ) {
      copyFrom( def.Object3D );
      type = def.Object3DType;
    } else {
      radius = 1.0;
      color0 = 0x00FF00;
      color1 = 0x00FF00;
      alpha = 1.0;
      offset = 0.0;
      quality = 3;
      ambient = 0.4;
      diffuse = 0.5;
      specular = 0.2;
      gloss = 30.0;
      shader = "Phong";
      type = "cone";
    }
    pos = null;
    dir = null;
    _polygon = null;
  }

  public override function loadFromXml( x:Xml,
                                        ?def:WMDefaults = null ):Void {
    clear( def );
    loadFromXmlWOClear( x );
  }

  public function loadFromXmlOverwrite( x:Xml ):Void {
    super.loadFromXml( x );
    if ( x.exists( "type" ) ) type = x.get( "type" );
  }

  public function loadFromXmlWOClear( x:Xml ):Void {
    loadFromXmlOverwrite( x );
    if ( !x.exists( "pos" ) ) {
      trace( "pos attribute is required for an <OBJ3D>" );
      return;
    }
    pos = Point3D.fromStringInverted( x.get( "pos" ) );
    if ( x.exists( "dir" ) ) dir = Point3D.fromStringInverted( x.get( "dir" ) );
    if ( x.exists( "x" ) ) xsize = Std.parseFloat( x.get( "x" ) );
    if ( x.exists( "y" ) ) ysize = Std.parseFloat( x.get( "y" ) );
    if ( x.exists( "z" ) ) zsize = Std.parseFloat( x.get( "z" ) );
  }

  public function gen( c:Context3D,
                       ?is_dc_active:Bool = false ):Void {
    pregen();
    gen2( c, is_dc_active );
  }

  public function pregen() {
    switch( type ) {
      case "cone":
        var myrad:Float = ( xsize == 0.0 ) ? radius : xsize;
        _polygon = new Cone3D( myrad, ysize, quality );
      case "cube":
        _polygon = new Cube3D( xsize, ysize, zsize );
      case "cylinder":
        var myrad:Float = ( xsize == 0.0 ) ? radius : xsize;
        _polygon = new Cylinder( myrad, ysize, quality );
      case "square":
        _polygon = new Square3D( xsize, ysize, true );
    }
    if ( dir != null ) _polygon.lookAt( dir );
    _polygon.translate( pos.x, pos.y, pos.z );
  }

  public function gen2( c:Context3D,
                        ?is_dc_active:Bool = false ):Void {
    _polygon.allocate( c, is_dc_active, shader, color0, alpha );
  }

  public function num():Int { return(1); }
  public function sumPos():Point3D { return( pos ); }
  public function translate( p:Point3D ):Void { pos.add( p ); }
  public function absmax():Point3D {
    return( new Point3D( Math.abs( pos.x ), Math.abs( pos.y ), Math.abs( pos.z ) ) );
  }
  public function scaleCoord( scale:Float ):Void {
    pos.multiply( scale );
    xsize *= scale;
    ysize *= scale;
    zsize *= scale;
  }
  public override function getDataSize():Int { return(5); }
}
