// 
// Copyright (c) 2003-2010, MIST Project, Nagoya University
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice,
// this list of conditions and the following disclaimer in the documentation
// and/or other materials provided with the distribution.
// 
// 3. Neither the name of the Nagoya University nor the names of its contributors
// may be used to endorse or promote products derived from this software
// without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
// IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
// THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 

/// @file mist/vector.h
//!
//! @brief eɑΉxNgZs߂̃Cu
//!
#ifndef __INCLUDE_MIST_VECTOR__
#define __INCLUDE_MIST_VECTOR__


#ifndef __INCLUDE_MIST_CONF_H__
#include "config/mist_conf.h"
#endif

#ifndef __INCLUDE_MIST_TYPE_TRAIT_H__
#include "config/type_trait.h"
#endif

#include <cmath>

// mistOԂ̎n܂
_MIST_BEGIN


//! @addtogroup vector_group xNgCu
//!
//! @code ̃wb_CN[h
//! #include <mist/vector.h>
//! @endcode
//!
//!  @{



/// @brief 3xNgNX
//! 
//! xNg̓ρEOϓȕւɈ߂̃NX
//! 
//! @param T c xNgɊeW\f[^^
//! 
template < class T >
class vector3
{
public:
	typedef T value_type;										///< @brief MIST̃ReiɊi[f[^^Dmist::array< data >  data Ɠ
	typedef size_t size_type;									///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
	typedef ptrdiff_t difference_type;							///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
	typedef typename float_type< T >::value_type float_type;	///< @brief ȂǂvZƂɗp镂_^

public:
	value_type x;		///< @brief XWl
	value_type y;		///< @brief YWl
	value_type z;		///< @brief ZWl

	/// @brief ftHgRXgN^D( 0, 0, 0 ) ɏ
	vector3( ) : x( 0 ), y( 0 ), z( 0 ){ }

	/// @brief ( xx, yy, zz ) ̃xNg쐬
	vector3( const value_type &xx, const value_type &yy, const value_type &zz ) : x( xx ), y( yy ), z( zz ){ }

	/// @brief ( vv, vv, vv ) ̃xNg쐬
	explicit vector3( const value_type &vv ) : x( vv ), y( vv ), z( vv ){ }


	/// @brief 3xNgŗvf̌^قȂ̂瓯vfxNg쐬
	//!
	//! @attention قȂvf^ԂŃf[^̕ϊ\łȂĂ͂ȂȂ
	//!
	template < class TT >
	vector3( const vector3< TT > &v ) : x( static_cast< value_type >( v.x ) ), y( static_cast< value_type >( v.y ) ), z( static_cast< value_type >( v.z ) ){ }


	/// @brief 3xNgœvf^̂̂pď
	vector3( const vector3< T > &v ) : x( v.x ), y( v.y ), z( v.z ){ }


	/// @brief f[^^3xNg
	//! 
	//! Rs[łxNg v ƑSxNg쐬D
	//! 
	//! @param[in] v c Rs[̃xNg
	//! 
	//! @return g
	//! 
	template < class TT >
	const vector3 &operator =( const vector3< TT > &v )
	{
		x = static_cast< value_type >( v.x );
		y = static_cast< value_type >( v.y );
		z = static_cast< value_type >( v.z );
		return ( *this );
	}

	/// @brief 3xNg
	//! 
	//! Rs[łxNg v ƑSxNg쐬D
	//! 
	//! @param[in] v c Rs[̃xNg
	//! 
	//! @return g
	//! 
	const vector3 &operator =( const vector3< T > &v )
	{
		if( &v != this )
		{
			x = v.x;
			y = v.y;
			z = v.z;
		}
		return ( *this );
	}


	/// @brief ]xNgԂ
	vector3 operator -( ) const { return ( vector3( -x, -y, -z ) ); }


	/// @brief xNga
	template < class TT >
	vector3 &operator +=( const vector3< TT > &v )
	{
		x = static_cast< value_type >( x + v.x );
		y = static_cast< value_type >( y + v.y );
		z = static_cast< value_type >( z + v.z );
		return( *this );
	}

	/// @brief xNg
	template < class TT >
	vector3 &operator -=( const vector3< TT > &v )
	{
		x = static_cast< value_type >( x - v.x );
		y = static_cast< value_type >( y - v.y );
		z = static_cast< value_type >( z - v.z );
		return( *this );
	}

	/// @brief xNg̊O
	template < class TT >
	vector3 &operator *=( const vector3< TT > &v )
	{
		value_type xx = static_cast< value_type >( y * v.z - z * v.y );
		value_type yy = static_cast< value_type >( z * v.x - x * v.z );
		value_type zz = static_cast< value_type >( x * v.y - y * v.x );
		x = xx;
		y = yy;
		z = zz;
		return( *this );
	}

	/// @brief xNg̒萔{
#if defined( __MIST_MSVC__ ) && __MIST_MSVC__ < 7
	vector3 &operator *=( const double &a )
#else
	template < class TT >
	vector3 &operator *=( const TT &a )
#endif
	{
		x = static_cast< value_type >( x * a );
		y = static_cast< value_type >( y * a );
		z = static_cast< value_type >( z * a );
		return( *this );
	}


	/// @brief xNg萔Ŋ
#if defined( __MIST_MSVC__ ) && __MIST_MSVC__ < 7
	vector3 &operator /=( const double &a )
#else
	template < class TT >
	vector3 &operator /=( const TT &a )
#endif
	{
		x = static_cast< value_type >( x / a );
		y = static_cast< value_type >( y / a );
		z = static_cast< value_type >( z / a );
		return( *this );
	}


	/// @brief 2̃xNgiSvfljǂ𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} = \mbox{\boldmath q} \rightarrow p_x = q_x \; \wedge \; p_y = q_y \; \wedge \; p_z = q_z \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c 2̃xNgꍇ
	//! @retval false c 2̃xNgقȂꍇ
	//! 
	bool operator ==( const vector3 &v ) const { return( x == v.x && y == v.y && z == v.z ); }

	/// @brief 2̃xNgȂiǂꂩPłقȂvfjǂ𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \neq \mbox{\boldmath q} \rightarrow \overline{ p_x = q_x \; \wedge \; p_y = q_y \; \wedge \; p_z = q_z} \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c 2̃xNgقȂꍇ
	//! @retval false c 2̃xNgꍇ
	//! 
	bool operator !=( const vector3 &v ) const { return( !( *this == v ) ); }

	/// @brief 2̃xNg < 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \ge \mbox{\boldmath q} \rightarrow \overline{ p_x \ge q_x \; \wedge \; p_y \ge q_y \; \wedge \; p_z \ge q_z } \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 <  v2 ̏ꍇ
	//! @retval false c v1 >= v2 ̏ꍇ
	//! 
	bool operator < ( const vector3 &v ) const
	{
		if( x == v.x )
		{
			if( y == v.y )
			{
				return( z < v.z );
			}
			else
			{
				return( y < v.y );
			}
		}
		else
		{
			return( x < v.x );
		}
	}

	/// @brief 2̃xNg <= 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \le \mbox{\boldmath q} \rightarrow p_x \le q_x \; \wedge \; p_y \le q_y \; \wedge \; p_z \le q_z \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 <= v2 ̏ꍇ
	//! @retval false c v1 >  v2 ̏ꍇ
	//! 
	bool operator <=( const vector3 &v ) const { return( v >= *this ); }

	/// @brief 2̃xNg > 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \le \mbox{\boldmath q} \rightarrow \overline{ p_x \le q_x \; \wedge \; p_y \le q_y \; \wedge \; p_z \le q_z } \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 >  v2 ̏ꍇ
	//! @retval false c v1 <= v2 ̏ꍇ
	//! 
	bool operator > ( const vector3 &v ) const { return( v < *this ); }

	/// @brief 2̃xNg >= 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \ge \mbox{\boldmath q} \rightarrow p_x \ge q_x \; \wedge \; p_y \ge q_y \; \wedge \; p_z \ge q_z \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 >= v2 ̏ꍇ
	//! @retval false c v1 <  v2 ̏ꍇ
	//! 
	bool operator >=( const vector3 &v ) const { return( !( *this < v ) ); }


	/// @brief PʃxNgvZ
	//! 
	//! \f[ \frac{\mbox{\boldmath v}}{\left\|{\mbox{\boldmath v}}\right\|} \f]
	//! 
	vector3 unit( ) const
	{
		float_type length_ = length( );
		if( length_ > 0 )
		{
			return( vector3( static_cast< value_type >( x / length_ ), static_cast< value_type >( y / length_ ), static_cast< value_type >( z / length_ ) ) );
		}
		else
		{
			return( *this );
		}
	}


	/// @brief xNg̓ςvZ
	//! 
	//! \f[ \mbox{\boldmath p} \cdot \mbox{\boldmath q} = p_x \times q_x + p_y \times q_y + p_z \times q_z \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	template < class TT >
	typename promote_trait< T, TT >::value_type inner( const vector3< TT > &v ) const
	{
		return( static_cast< typename promote_trait< T, TT >::value_type >( x * v.x + y * v.y + z * v.z ) );
	}


	/// @brief xNg̊OςvZ
	//! 
	//! \f[ \mbox{\boldmath p} \times \mbox{\boldmath q} = \left( p_y \times q_z - p_z \times q_y \;,\; p_z \times q_x - p_x \times q_z \;,\; p_x \times q_y - p_y \times q_x \right)^T \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	template < class TT >
	vector3< typename promote_trait< value_type, TT >::value_type > outer( const vector3< TT > &v ) const
	{
		return( vector3< typename promote_trait< value_type, TT >::value_type >( y * v.z - z * v.y, z * v.x - x * v.z, x * v.y - y * v.x ) );
	}


	/// @brief xNg̑傫vZ
	//! 
	//! \f[ \left\|\mbox{\boldmath v}\right\| = \sqrt{v_x^2 + v_y^2 + v_z^2} \f]
	//! 
	//! @return xNg̑傫
	//! 
	float_type length( ) const { return ( static_cast< float_type >( std::sqrt( static_cast< double >( x * x + y * y + z * z ) ) ) ); }

	/// @brief Cӎ̃xNg̉]
	//! 
	//! @param[in] v     c ]
	//! @param[in] theta c E˂Hő]px
	//! 
	//! @return ]̃xNg
	//! 
	vector3 rotate( const vector3 &v, double theta ) const
	{
		theta *= 3.1415926535897932384626433832795 / 180.0;
		double cs = std::cos( theta ), sn = std::sin( theta );
		value_type xx = static_cast< value_type >( ( v.x * v.x * ( 1.0 - cs ) + cs ) * x + ( v.x * v.y * ( 1.0 - cs ) - v.z * sn ) * y + ( v.x * v.z * ( 1.0 - cs ) + v.y * sn ) * z );
		value_type yy = static_cast< value_type >( ( v.x * v.y * ( 1.0 - cs ) + v.z * sn ) * x + ( v.y * v.y * ( 1.0 - cs ) + cs ) * y + ( v.y * v.z * ( 1.0 - cs ) - v.x * sn ) * z );
		value_type zz = static_cast< value_type >( ( v.x * v.z * ( 1.0 - cs ) - v.y * sn ) * x + ( v.y * v.z * ( 1.0 - cs ) + v.x * sn ) * y + ( v.z * v.z * ( 1.0 - cs ) + cs ) * z );
		return ( vector3( xx, yy, zz ) );
	}
};


/// @brief xNg̓
template < class T1, class T2 >
inline typename promote_trait< T1, T2 >::value_type operator ^( const vector3< T1 > &v1, const vector3< T2 > &v2 )
{
	typedef typename promote_trait< T1, T2 >::value_type value_type;
	return( vector3< value_type >( v1 ).inner( v2 ) );
}

// ^̏isZ̒`

/// @brief xNg̘a
DEFINE_PROMOTE_BIND_OPERATOR1( vector3, + )

/// @brief xNg̍
DEFINE_PROMOTE_BIND_OPERATOR1( vector3, - )

/// @brief xNg̊O
DEFINE_PROMOTE_BIND_OPERATOR1( vector3, * )

/// @brief xNgƒ萔̐
DEFINE_PROMOTE_BIND_OPERATOR2( vector3, * )

/// @brief 萔ƃxNg̐
DEFINE_PROMOTE_BIND_OPERATOR3( vector3, * )

/// @brief xNg萔Ŋ
DEFINE_PROMOTE_BIND_OPERATOR2( vector3, / )



/// @brief w肳ꂽXg[ɁCRei̗vf𐮌`ďo͂
//! 
//! @param[in,out] out c ͂Əo͂sXg[
//! @param[in]     v   c 3xNg
//! 
//! @return ͂ꂽXg[
//! 
//! @code o͗
//! ( 1, 2, 3 )
//! @endcode
//!
template < class T > inline std::ostream &operator <<( std::ostream &out, const vector3< T > &v )
{
	out << v.x << ", ";
	out << v.y << ", ";
	out << v.z;
	return( out );
}

/// @brief Method for reading the vector comma separated from an istream
//! 
//! @param[in,out] in c input stream
//! @param[out]    v  c 3-dimensional vector
//! 
//! @return the input stream
//! 
//! @code Marco Feuerstein
//! ( 1, 2, 3 )
//! @endcode
//!
template < class T > inline std::istream &operator >>( std::istream &in, vector3< T > &v )
{
	std::string comma;
	in >> v.x >> comma;
	in >> v.y >> comma;
	in >> v.z;
	return( in );
}



/// @brief 2xNgNX
//! 
//! xNg̓ρEOϓȕւɈ߂̃NX
//! 
//! @param T c xNgɊeW\f[^^
//! 
template < class T >
class vector2
{
public:
	typedef T value_type;										///< @brief MIST̃ReiɊi[f[^^Dmist::array< data >  data Ɠ
	typedef size_t size_type;									///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
	typedef ptrdiff_t difference_type;							///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
	typedef typename float_type< T >::value_type float_type;	///< @brief ȂǂvZƂɗp镂_^

public:
	value_type x;		///< @brief XWl
	value_type y;		///< @brief YWl

	/// @brief ftHgRXgN^D( 0, 0 ) ɏ
	vector2( ) : x( 0 ), y( 0 ){ }

	/// @brief ( xx, yy ) ̃xNg쐬
	vector2( const value_type &xx, const value_type &yy ) : x( xx ), y( yy ){ }

	/// @brief ( vv, vv ) ̃xNg쐬
	vector2( const value_type &vv ) : x( vv ), y( vv ){ }


	/// @brief 2xNgŗvf̌^قȂ̂瓯vfxNg쐬
	//!
	//! @attention قȂvf^ԂŃf[^̕ϊ\łȂĂ͂ȂȂ
	//!
	template < class TT >
	vector2( const vector2< TT > &v ) : x( static_cast< value_type >( v.x ) ), y( static_cast< value_type >( v.y ) ){ }


	/// @brief 2xNgœvf^̂̂pď
	vector2( const vector2< T > &v ) : x( v.x ), y( v.y ){ }


	/// @brief ̃f[^^2xNg
	//! 
	//! Rs[łxNg v ƑSxNg쐬D
	//! 
	//! @param[in] v c Rs[̃xNg
	//! 
	//! @return g
	//! 
	template < class TT >
	const vector2 &operator =( const vector2< TT > &v )
	{
		x = static_cast< value_type >( v.x );
		y = static_cast< value_type >( v.y );
		return ( *this );
	}

	/// @brief 2xNg
	//! 
	//! Rs[łxNg v ƑSxNg쐬D
	//! 
	//! @param[in] v c Rs[̃xNg
	//! 
	//! @return g
	//! 
	const vector2 &operator =( const vector2< T > &v )
	{
		if( &v != this )
		{
			x = v.x;
			y = v.y;
		}
		return ( *this );
	}


	/// @brief ]xNgԂ
	vector2 operator -( ) const { return ( vector2( -x, -y ) ); }


	/// @brief xNga
	template < class TT >
	vector2 &operator +=( const vector2< TT > &v ){ x = static_cast< value_type >( x + v.x ); y = static_cast< value_type >( y + v.y ); return( *this ); }

	/// @brief xNg
	template < class TT >
	vector2 &operator -=( const vector2< TT > &v ){ x = static_cast< value_type >( x - v.x ); y = static_cast< value_type >( y - v.y ); return( *this ); }


	/// @brief xNg̒萔{
#if defined( __MIST_MSVC__ ) && __MIST_MSVC__ < 7
	vector2 &operator *=( const double &a )
#else
	template < class TT >
	vector2 &operator *=( const TT &a )
#endif
	{
		x = static_cast< value_type >( x * a );
		y = static_cast< value_type >( y * a );
		return( *this );
	}


	/// @brief xNg萔Ŋ
#if defined( __MIST_MSVC__ ) && __MIST_MSVC__ < 7
	vector2 &operator /=( const double &a )
#else
	template < class TT >
	vector2 &operator /=( const TT &a )
#endif
	{
		x = static_cast< value_type >( x / a );
		y = static_cast< value_type >( y / a );
		return( *this );
	}


	/// @brief 2̃xNgiSvfljǂ𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} = \mbox{\boldmath q} \rightarrow p_x = q_x \; \wedge \; p_y = q_y \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c 2̃xNgꍇ
	//! @retval false c 2̃xNgقȂꍇ
	//! 
	bool operator ==( const vector2 &v ) const { return( x == v.x && y == v.y ); }

	/// @brief 2̃xNgȂiǂꂩPłقȂvfjǂ𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \neq \mbox{\boldmath q} \rightarrow \overline{ p_x = q_x \; \wedge \; p_y = q_y } \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c 2̃xNgقȂꍇ
	//! @retval false c 2̃xNgꍇ
	//! 
	bool operator !=( const vector2 &v ) const { return( !( *this == v ) ); }

	/// @brief 2̃xNg < 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \ge \mbox{\boldmath q} \rightarrow \overline{ p_x \ge q_x \; \wedge \; p_y \ge q_y } \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 <  v2 ̏ꍇ
	//! @retval false c v1 >= v2 ̏ꍇ
	//! 
	bool operator < ( const vector2 &v ) const
	{
		if( x == v.x )
		{
			return( y < v.y );
		}
		else
		{
			return( x < v.x );
		}
	}

	/// @brief 2̃xNg <= 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \ge \mbox{\boldmath q} \rightarrow p_x \le q_x \; \wedge \; p_y \le q_y \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 <= v2 ̏ꍇ
	//! @retval false c v1 >  v2 ̏ꍇ
	//! 
	bool operator <=( const vector2 &v ) const { return( v >= *this ); }

	/// @brief 2̃xNg > 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \ge \mbox{\boldmath q} \rightarrow \overline{ p_x \le q_x \; \wedge \; p_y \le q_y } \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 >  v2 ̏ꍇ
	//! @retval false c v1 <= v2 ̏ꍇ
	//! 
	bool operator > ( const vector2 &v ) const { return( v < *this ); }

	/// @brief 2̃xNg >= 𔻒肷
	//! 
	//! \f[ \mbox{\boldmath p} \ge \mbox{\boldmath q} \rightarrow p_x \ge q_x \; \wedge \; p_y \ge q_y \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @retval true  c v1 >= v2 ̏ꍇ
	//! @retval false c v1 <  v2 ̏ꍇ
	//! 
	bool operator >=( const vector2 &v ) const { return( !( *this < v ) ); }


	/// @brief PʃxNgvZ
	//! 
	//! \f[ \frac{\mbox{\boldmath v}}{\left\|\mbox{\boldmath v}\right\|} \f]
	//! 
	vector2 unit( ) const
	{
		float_type length_ = length( );
		if( length_ > 0 )
		{
			return( vector2( static_cast< value_type >( x / length_ ), static_cast< value_type >( y / length_ ) ) );
		}
		else
		{
			return( *this );
		}
	}


	/// @brief xNg̓ςvZ
	//! 
	//! \f[ \mbox{\boldmath p} \cdot \mbox{\boldmath q} = p_x \times q_x + p_y \times q_y \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	template < class TT >
	typename promote_trait< T, TT >::value_type inner( const vector2< TT > &v ) const { return( static_cast< typename promote_trait< T, TT >::value_type >( x * v.x + y * v.y ) ); }


	/// @brief xNg̊OςvZ
	//! 
	//! \f[ \mbox{\boldmath p} \times \mbox{\boldmath q} = p_x \times q_y - p_y \times q_x \f]
	//! 
	//! @param[in] v c EӒl
	//! 
	template < class TT >
	typename promote_trait< T, TT >::value_type outer( const vector2< TT > &v ) const { return( static_cast< typename promote_trait< T, TT >::value_type >( x * v.y - y * v.x ) ); }


	/// @brief xNg̑傫vZ
	//! 
	//! \f[ \left\|\mbox{\boldmath v}\right\| = \sqrt{v_x^2+v_y^2} \f]
	//! 
	//! @return xNg̑傫
	//! 
	float_type length( ) const { return ( static_cast< float_type >( std::sqrt( static_cast< double >( x * x + y * y ) ) ) ); }

	//// xNg̉]
	//vector2 rotate( const vector2 &a, double theta ) const
	//{
	//	theta *= 3.1415926535897932384626433832795 / 180.0;
	//	double cs = std::cos( theta ), sn = std::sin( theta );
	//	value_type xx = static_cast< value_type >( ( v.x * v.x * ( 1.0 - cs ) + cs ) * x + ( v.x * v.y * ( 1.0 - cs ) - v.z * sn ) * y + ( v.x * v.z * ( 1.0 - cs ) + v.y * sn ) * z );
	//	value_type yy = static_cast< value_type >( ( v.x * v.y * ( 1.0 - cs ) + v.z * sn ) * x + ( v.y * v.y * ( 1.0 - cs ) + cs ) * y + ( v.y * v.z * ( 1.0 - cs ) - v.x * sn ) * z );
	//	value_type zz = static_cast< value_type >( ( v.x * v.z * ( 1.0 - cs ) - v.y * sn ) * x + ( v.y * v.z * ( 1.0 - cs ) + v.x * sn ) * y + ( v.z * v.z * ( 1.0 - cs ) + cs ) * z );
	//	return ( vector2( xx, yy, zz ) );
	//}

};


/// @brief xNg̓
template < class T1, class T2 >
inline typename promote_trait< T1, T2 >::value_type operator ^( const vector2< T1 > &v1, const vector2< T2 > &v2 )
{
	typedef typename promote_trait< T1, T2 >::value_type value_type;
	return( vector2< value_type >( v1 ).inner( v2 ) );
}

/// @brief xNg̊O
template < class T1, class T2 >
inline typename promote_trait< T1, T2 >::value_type operator *( const vector2< T1 > &v1, const vector2< T2 > &v2 )
{
	typedef typename promote_trait< T1, T2 >::value_type value_type;
	return( vector2< value_type >( v1 ).outer( v2 ) );
}


// ^̏isZ̒`
/// @brief xNg̘a
DEFINE_PROMOTE_BIND_OPERATOR1( vector2, + )

/// @brief xNg̍
DEFINE_PROMOTE_BIND_OPERATOR1( vector2, - )

/// @brief xNgƒ萔̐
DEFINE_PROMOTE_BIND_OPERATOR2( vector2, * )

/// @brief 萔ƃxNg̐
DEFINE_PROMOTE_BIND_OPERATOR3( vector2, * )

/// @brief xNg萔Ŋ
DEFINE_PROMOTE_BIND_OPERATOR2( vector2, / )


/// @brief w肳ꂽXg[ɁCRei̗vf𐮌`ďo͂
//! 
//! @param[in,out] out c ͂Əo͂sXg[
//! @param[in]     v   c 2xNg
//! 
//! @return ͂ꂽXg[
//! 
//! @code o͗
//! ( 1, 2 )
//! @endcode
//!
template < class T > inline std::ostream &operator <<( std::ostream &out, const vector2< T > &v )
{
	out << v.x << ", ";
	out << v.y;
	return( out );
}

/// @brief Method for reading the vector comma separated from an istream
//! 
//! @param[in,out] in c input stream
//! @param[out]    v  c 2-dimensional vector
//! 
//! @return the input stream
//! 
//! @code Marco Feuerstein
//! ( 1, 2 )
//! @endcode
//!
template < class T > inline std::istream &operator >>( std::istream &in, vector2< T > &v )
{
	std::string comma;
	in >> v.x >> comma;
	in >> v.y;
	return( in );
}



#if defined( _MIST_VECTOR_SUPPORT_ ) && _MIST_VECTOR_SUPPORT_ != 0


/// @brief NxNgNX
//! 
//! - _USE_EXPRESSION_TEMPLATE_  }N1̏ꍇ́CExpression Template 𗘗pāCsExNgZ
//! - _CHECK_MATRIX_OPERATION_   }N1̏ꍇ́CsExNgZ̍ӂƉEӂŃxNg̃TCYK؂ł邩`FbN
//! - _CHECK_ACCESS_VIOLATION_   }N1̏ꍇ́CxNg̗vfփANZXۂɁC͈͊Oǂ𔻒肷
//! 
//! @param T         c sɊi[f[^^
//! @param Allocator c MISTReipAP[^^Dȗꍇ́CSTL̃ftHgAP[^gp
//! 
template < class T, class Allocator = ::std::allocator< T > >
class vector : public matrix< T, Allocator >
{
private:
	typedef matrix< T, Allocator > base;		///< @brief NXsNX

public:
	typedef typename base::allocator_type allocator_type;		///< @brief MISTReipAP[^^
	typedef typename base::reference reference;					///< @brief MIST̃ReiɊi[f[^^̎QƁDmist::array< data > ̏ꍇCdata & ƂȂ
	typedef typename base::const_reference const_reference;		///< @brief MIST̃ReiɊi[f[^^ const QƁDmist::array< data > ̏ꍇCconst data & ƂȂ
	typedef typename base::value_type value_type;				///< @brief MIST̃ReiɊi[f[^^Dmist::array< data >  data Ɠ
	typedef typename base::size_type size_type;					///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
	typedef typename base::difference_type difference_type;		///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
	typedef typename base::pointer pointer;						///< @brief MIST̃ReiɊi[f[^^̃|C^[^Dmist::array< data > ̏ꍇCdata * ƂȂ
	typedef typename base::const_pointer const_pointer;			///< @brief MIST̃ReiɊi[f[^^ const |C^[^Dmist::array< data > ̏ꍇCconst data * ƂȂ

	typedef typename base::iterator iterator;								///< @brief MIST̃Rei𑀍삷CSTLŌ̃_ANZXCe[^
	typedef typename base::const_iterator const_iterator;					///< @brief MIST̃Rei𑀍삷C̃_ANZXCe[^̃RXg
	typedef typename base::reverse_iterator reverse_iterator;				///< @brief MIST̃Rei𑀍삷Ct̃_ANZXCe[^
	typedef typename base::const_reverse_iterator const_reverse_iterator;	///< @brief MIST̃Rei𑀍삷Ct̃_ANZXCe[^̃RXg

	/// @brief f[^^̕ϊs
	template < class TT, class AAllocator = std::allocator< TT > > 
	struct rebind
	{
		typedef vector< TT, AAllocator > other;
	};


	//template < class TT, class AAlocator >
	//const vector& operator +=( const vector< TT, AAlocator > &v2 )
	//{
	//	base::operator +=( v2 );
	//	return( v1 );
	//}

	//template < class TT, class AAlocator >
	//const vector& operator -=( const vector< TT, AAlocator > &v2 )
	//{
	//	base::operator -=( v2 );
	//	return( v1 );
	//}


	/// @brief xNg̊O
	//! 
	//! @attention OςvZ2̃xNg̎ȂĂ͂ȂȂ
	//! 
	//! @param[in] v2 c EӒl
	//! 
	//! @return g
	//! 
	template < class TT, class AAlocator >
	const vector& operator *=( const vector< TT, AAlocator > &v2 )
	{
#if defined( _CHECK_MATRIX_OPERATION_ ) && _CHECK_MATRIX_OPERATION_ != 0
		if( base::size( ) != v2.size( ) || base::size( ) < 3 )
		{
			// Oς̌vZł܂O
			::std::cerr << "can't calculate outer product of two vectors." << ::std::endl;
			return( *this );
		}
#endif

		typedef typename vector< T, Allocator >::size_type size_type;
		vector &v1 = *this;
		vector v( v1.size( ) );

		size_type i;
		for( i = 0 ; i < v.size( ) - 2 ; i++ )
		{
			v[ i ] = v1[ i + 1 ] * v2[ i + 2 ] - v1[ i + 2 ] * v2[ i + 1 ];
		}

		v[ i ] = v1[ i + 1 ] * v2[ 0 ] - v1[ 0 ] * v2[ i + 1 ];
		v[ i + 1 ] = v1[ 0 ] * v2[ 1 ] - v1[ 1 ] * v2[ 0 ];

		v1.swap( v );

		return( *this );
	}


	/// @brief xNg̓
	//! 
	//! @attention ςvZ2̃xNg̎ȂĂ͂ȂȂ
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @return όʂ̃XJ[l
	//! 
	value_type operator ^( const vector &v ) const { return( inner( v ) ); }


	/// @brief xNg̓
	//! 
	//! @attention ςvZ2̃xNg̎ȂĂ͂ȂȂ
	//! 
	//! @param[in] v2 c EӒl
	//! 
	//! @return όʂ̃XJ[l
	//! 
	value_type inner( const vector &v2 ) const
	{
#if defined( _CHECK_MATRIX_OPERATION_ ) && _CHECK_MATRIX_OPERATION_ != 0
		if( base::size( ) != v2.size( ) )
		{
			// ς̌vZł܂O
			::std::cerr << "can't calculate inner product of two vectors." << ::std::endl;
			return( value_type( 0 ) );
		}
#endif

		typedef typename vector< T, Allocator >::size_type size_type;
		const vector &v1 = *this;
		value_type v = value_type( 0 );
		for( size_type i = 0 ; i < base::size( ) ; i++ )
		{
			v += v1[ i ] * v2[ i ];
		}

		return( v );
	}


	/// @brief xNg̊O
	//! 
	//! @attention OςvZ2̃xNg̎ȂĂ͂ȂȂ
	//! 
	//! @param[in] v c EӒl
	//! 
	//! @return g
	//! 
	vector outer( const vector &v ) const
	{
		return( vector( *this ) *= v );
	}


public:
	/// @brief fBtHgRXgN^D0 ̃xNg쐬
	vector( ) : base( ) {}

	/// @brief AP[^ a ̃Rs[𗘗pC0 ̃xNg쐬
	explicit vector( const Allocator &a ) : base( a ) {}


	/// @brief num ̃xNg쐬CftHglŗvf
	vector( size_type num ) : base( num, 1 ) {}

	/// @brief num ŁCAP[^ a ̃Rs[𗘗pxNg쐬CftHglŗvf
	vector( size_type num, const Allocator &a ) : base( num, 1, a ) {}


	/// @brief num ̃xNg쐬CSvfl val ŗvf
	vector( size_type num, const T &val ) : base( num, 1, val ) {}

	/// @brief num ŁCAP[^ a ̃Rs[𗘗pxNg쐬CSvfl val ŗvf
	vector( size_type num, const T &val, const Allocator &a ) : base( num, 1, val, a ) {}


	/// @brief ptr ẅɁCnum ̃xNg쐬쐬iptr w̗p\ȃʂ mem_available j
	vector( size_type num, pointer ptr, size_type mem_available ) : base( num, 1, ptr, mem_available ) {}

	/// @brief ptr ẅɁCnum ̃xNg쐬쐬CSvfl val ŗvfiptr w̗p\ȃʂ mem_available j
	vector( size_type num, const T &val, pointer ptr, size_type mem_available ) : base( num, 1, val, ptr, mem_available ) {}


	/// @brief ̃xNgŗvf̌^قȂ̂瓯TCỸxNg쐬
	//!
	//! @attention قȂvf^ԂŃf[^̕ϊ\łȂĂ͂ȂȂ
	//!
	template < class TT, class AAlocator >
	vector( const vector< TT, AAlocator > &o ) : base( o ){ }


	/// @brief ̃xNgœvf^̂̂pď
	vector( const vector< T, Allocator > &o ) : base( o ){ }

#if _USE_EXPRESSION_TEMPLATE_ != 0
	/// @brief ̍sExNgŽʂpăxNg
	template < class Expression >
	vector( const matrix_expression< Expression > &expression ) : base( expression.size( ), 1 )
	{
		vector &v = *this;
		for( size_type indx = 0 ; indx < v.size( ) ; indx++ )
		{
			v[ indx ] = expression[ indx ];
		}
	}
#endif
};



/// @brief w肳ꂽXg[ɁCs̗vf𐮌`ďo͂
//! 
//! @param[in,out] out c ͂Əo͂sXg[
//! @param[in]     v   c o͂s
//! 
//! @return ͂ꂽXg[
//! 
//! @code o͗
//! 1, 2, 3, 4
//! @endcode
//! 
template < class T, class Allocator >
inline ::std::ostream &operator <<( ::std::ostream &out, const vector< T, Allocator > &v )
{
	typename vector< T, Allocator >::size_type indx;
	for( indx = 0 ; indx < v.size( ) ; indx++ )
	{
		out << v[ indx ];
		if( indx != v.size( ) - 1 ) out << ", ";
	}

	return( out );
}

#if 0
#else

/// @brief xNg̊OςvZ
//! 
//! @attention OςvZ2̃xNg̎ȂĂ͂ȂȂ
//! 
//! @param[in] v1 c ӒlxNg
//! @param[in] v2 c EӒlxNg
//! 
//! @return Oό
//! 
template < class T, class Allocator >
inline vector< T, Allocator > operator *( const vector< T, Allocator > &v1, const vector< T, Allocator > &v2 )
{
	return( vector< T, Allocator >( v1 ) *= v2 );
}

#endif



#endif // _MIST_VECTOR_SUPPORT_

/// @}
//  xNgCuO[v̏I


// mistOԂ̏I
_MIST_END

#endif // __INCLUDE_MIST_VECTOR__
