/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.opinionbox;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBox;
import com.aimluck.eip.cayenne.om.portlet.OriTOpinionBoxMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipPost;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.opinionbox.util.OpinionBoxUtils;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.social.ALActivityService;
import com.aimluck.eip.services.social.model.ALActivityPutRequest;
import com.aimluck.eip.user.beans.UserGroupLiteBean;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 目安箱のフォームデータを管理するクラスです。 <BR>
 *
 */
public class OpinionBoxFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(OpinionBoxFormData.class.getName());

  /** 投稿者 */
  private ALNumberField tokumei_flg;

  /** タイトル */
  private ALStringField opinionbox_name;

  /** メモ */
  private ALStringField note;

  /** 通知先ユーザIDリスト */
  private ALStringField positions;

  /** 通知先一覧 */
  private List<ALEipUser> mapList;

  /** 役職者一覧 */
  // 2012/12/04 add morimoto
  private List<ALEipUser> superiorUserList;

  /** 親 報告書 ID */
  private ALNumberField parent_id;

  private String orgId;

  private int uid;

  /** <code>login_user</code> ログインユーザー */
  private ALEipUser login_user;

  /** アクセス権限の機能名 */
  private String aclPortletFeature = null;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    login_user = ALEipUtils.getALEipUser(rundata);

    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();
    // アクセス権

    int view_uid = OpinionBoxUtils.getViewId(rundata, context, uid);
    if (view_uid == uid) {
      aclPortletFeature = ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_SELF;
    } else {
      aclPortletFeature =
        ALAccessControlConstants.POERTLET_FEATURE_OPINIONBOX_OTHER;
    }

  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {

    // 匿名
	tokumei_flg = new ALNumberField();
	tokumei_flg.setFieldName("投稿者");

    // タイトル
    opinionbox_name = new ALStringField();
    opinionbox_name.setFieldName("タイトル");
    opinionbox_name.setTrim(true);
    opinionbox_name.setTrim(true);
    // メモ
    note = new ALStringField();
    note.setFieldName("内容");
    note.setTrim(false);
    // 通知先のリスト
    positions = new ALStringField();
    positions.setFieldName("通知先");
    positions.setTrim(true);
    // 通知先リスト
    mapList = new ArrayList<ALEipUser>();

    // 役職者一覧
    superiorUserList = new ArrayList<ALEipUser>();
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
      try {
        String userNames[] = rundata.getParameters().getStrings("positions");
        if (userNames != null && userNames.length > 0) {
          SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
          Expression exp =
            ExpressionFactory.inExp(TurbineUser.LOGIN_NAME_PROPERTY, userNames);
          query.setQualifier(exp);
          mapList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
        }
      } catch (Exception ex) {
        logger.error("Exception", ex);
      }


      // 所属する部署を取得 add 2012/12/07 morimoto
      List<UserGroupLiteBean> groupList = OpinionBoxUtils.getPostBeanList(ALEipUtils.getUserId(rundata));
      List<ALEipUser> userList = OpinionBoxUtils.getSuperiorUsers(groupList);
      superiorUserList.addAll(userList);

    }
    return res;
  }

  /**
   * リクエストの各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // 投稿者
    tokumei_flg.setNotNull(true);
    // 報告書名の文字数制限
    opinionbox_name.setNotNull(true);
    opinionbox_name.limitMaxLength(50);
    // メモの文字数制限
    note.setNotNull(true);
    note.limitMaxLength(1000);
  }

  /**
   * リクエストのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
	// 投稿者
	tokumei_flg.validate(msgList);
    // 報告書名
    opinionbox_name.validate(msgList);
    // メモ
    note.validate(msgList);

    // 通知先
    if (mapList == null || mapList.size() <= 0) {
      msgList.add("『 <span class='em'>通知先</span> 』を指定してください。");
    } else if (!(mapList.get(0) instanceof ALEipUser)) {
      msgList.add("通知先のユーザーが全て無効、もしくは削除されています。有効なユーザーが一人以上いる経路を選択してください。");
    }

    return (msgList.size() == 0);

  }

  /**
   * リクエストをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      OriTOpinionBox opinionbox = OpinionBoxUtils.getOriTOpinionBox(rundata, context);
      if (opinionbox == null) {
        return false;
      }

      // 匿名
      tokumei_flg.setValue(opinionbox.getTokumeiFlg());
      // 報告書名
      opinionbox_name.setValue(opinionbox.getOpinionBoxName());
      // メモ
      note.setValue(opinionbox.getNote());

      List<OriTOpinionBoxMap> maps = OpinionBoxUtils.getOriTOpinionBoxMap(opinionbox);
      OriTOpinionBoxMap map = null;
      int size = maps.size();
      for (int i = 0; i < size; i++) {
        map = maps.get(i);
        int user_id = map.getUserId().intValue();
        mapList.add(ALEipUtils.getALEipUser(user_id));
      }


      // 所属する部署を取得 add 2012/12/07 morimoto
      List<UserGroupLiteBean> groupList = OpinionBoxUtils.getPostBeanList(ALEipUtils.getUserId(rundata));
      List<ALEipUser> userList = OpinionBoxUtils.getSuperiorUsers(groupList);
      superiorUserList.addAll(userList);


    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * リクエストをデータベースから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      OriTOpinionBox opinionbox = OpinionBoxUtils.getOriTOpinionBox(rundata, context);

      // イベントログ用
      String reqname = opinionbox.getOpinionBoxName();

      // リクエストを削除
      Database.delete(opinionbox);

      if (opinionbox.getParentId() == 0) {
        List<OriTOpinionBox> opinionboxs =
          OpinionBoxUtils.getChildOpinionBoxs(opinionbox.getOpinionId());

        Database.deleteAll(opinionboxs);
      }

      Database.commit();

//      // イベントログに保存
//      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
//        opinionbox.getOpinionId(),
//        ALEventlogConstants.PORTLET_TYPE_OPINIONBOX,
//        reqname);

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * リクエストをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) throws ALDBErrorException {
    OriTOpinionBox opinionbox = null;
    try {
      Date nowDate = Calendar.getInstance().getTime();

      // 新規オブジェクトモデル
      opinionbox = Database.create(OriTOpinionBox.class);

      // 報告書名
      opinionbox.setOpinionBoxName(opinionbox_name.getValue());
      // ユーザーID
      opinionbox
        .setUserId(Integer.valueOf((int) login_user.getUserId().getValue()));
      // 匿名フラグ
      opinionbox.setTokumeiFlg((int) tokumei_flg.getValue());
      // メモ
      opinionbox.setNote(note.getValue());
      // 作成日
      opinionbox.setCreateDate(nowDate);
      // 更新日
      opinionbox.setUpdateDate(nowDate);
      // ユーザーID
      opinionbox.setTurbineUser(ALEipUtils.getTurbineUser(ALEipUtils
        .getUserId(rundata)));
      // 親レポート
      opinionbox.setParentId(0);

      // 通知先
      for (ALEipUser user : mapList) {
        OriTOpinionBoxMap map = Database.create(OriTOpinionBoxMap.class);
        int userid = (int) user.getUserId().getValue();

        map.setOriTOpinionBox(opinionbox);
        map.setUserId(Integer.valueOf(userid));
        // R: 未読 A: 既読
        map.setStatus(OpinionBoxUtils.DB_STATUS_UNREAD);
        map.setCreateDate(nowDate);
        map.setUpdateDate(nowDate);
      }

      // リクエストを登録
      Database.commit();

//      // イベントログに保存
//      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
//        opinionbox.getOpinionId(),
//        ALEventlogConstants.PORTLET_TYPE_OPINIONBOX,
//        opinionbox_name.getValue());


      List<String> recipients = new ArrayList<String>();
      for (ALEipUser user : mapList) {
    	  recipients.add(ALEipUtils.getTurbineUser((int)user.getUserId().getValue()).getLoginName().toString());
      }

      // 通知先に通知
      String userName = "匿名希望";
      String loginName = "";
      if ((int)tokumei_flg.getValue() == 0) {
    	  userName = ALEipUtils.getUserFullName(uid);
    	  loginName = login_user.getName().getValue();
      }

      ALActivityService.create(new ALActivityPutRequest().withAppId(
        "OpinionBox").withUserId(uid).withLoginName(loginName)
        .withPortletParams("?template=OpinionBoxDetailScreen&entityid="+opinionbox.getOpinionId())
        .withRecipients(recipients)
        .withTitle(userName + "さん より 目安箱 「"+ opinionbox_name.getValue()+"」 に投稿がありました。")
        .withPriority(1f)
        .withExternalId(String.valueOf(opinionbox.getOpinionId())));


    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
//    try {
//      // メール送信
//      int msgType = ALMailUtils.getSendDestType(ALMailUtils.KEY_MSGTYPE_OPINIONBOX);
//      if (msgType > 0) {
//        // パソコンへメールを送信
//        List<ALEipUserAddr> destMemberList =
//          ALMailUtils.getALEipUserAddrs(
//            mapList,
//            ALEipUtils.getUserId(rundata),
//            false);
//        String subject = "[" + ALOrgUtilsService.getAlias() + "]目安箱";
//        String orgId = Database.getDomainName();
//
//        List<ALAdminMailMessage> messageList =
//          new ArrayList<ALAdminMailMessage>();
//        for (ALEipUserAddr destMember : destMemberList) {
//          ALAdminMailMessage message = new ALAdminMailMessage(destMember);
//          message.setPcSubject(subject);
//          message.setCellularSubject(subject);
//          message.setPcBody(OpinionBoxUtils.createMsgForPc(
//            rundata,
//            opinionbox,
//            mapList,
//            true));
//          message.setCellularBody(OpinionBoxUtils.createMsgForCellPhone(
//            rundata,
//            opinionbox,
//            mapList,
//            true,
//            destMember.getUserId()));
//          messageList.add(message);
//        }
//
//        ALMailService.sendAdminMailAsync(new ALAdminMailContext(
//          orgId,
//          ALEipUtils.getUserId(rundata),
//          messageList,
//          ALMailUtils.getSendDestType(ALMailUtils.KEY_MSGTYPE_OPINIONBOX)));
//        // msgList.addAll(errors);
//
//      }
//    } catch (Exception ex) {
//      msgList.add("メールを送信できませんでした。");
//      logger.error("Exception", ex);
//      return false;
//    }
    return true;
  }

  /**
   * データベースに格納されているリクエストを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    OriTOpinionBox opinionbox = null;
    try {
      // オブジェクトモデルを取得
      opinionbox = OpinionBoxUtils.getOriTOpinionBox(rundata, context);
      if (opinionbox == null) {
        return false;
      }

      Date nowDate = Calendar.getInstance().getTime();

      // 匿名
      opinionbox.setTokumeiFlg((int) tokumei_flg.getValue());
      // 報告書名
      opinionbox.setOpinionBoxName(opinionbox_name.getValue());
      // ユーザーID
      opinionbox
        .setUserId(Integer.valueOf((int) login_user.getUserId().getValue()));
      // メモ
      opinionbox.setNote(note.getValue());
      // 更新日
      opinionbox.setUpdateDate(nowDate);
      // ユーザーID
      opinionbox.setTurbineUser(ALEipUtils.getTurbineUser(ALEipUtils
        .getUserId(rundata)));

      // 古いマップデータを削除
      List<OriTOpinionBoxMap> tmp_map = OpinionBoxUtils.getOriTOpinionBoxMap(opinionbox);
      Database.deleteAll(tmp_map);

      // 通知先
      for (ALEipUser user : mapList) {
        OriTOpinionBoxMap map = Database.create(OriTOpinionBoxMap.class);
        int userid = (int) user.getUserId().getValue();

        map.setOriTOpinionBox(opinionbox);
        map.setUserId(Integer.valueOf(userid));
        // R: 未読 A: 既読
        // 更新があった際には常に未読とする
        map.setStatus(OpinionBoxUtils.DB_STATUS_UNREAD);
        map.setCreateDate(nowDate);
        map.setUpdateDate(nowDate);
      }

      // リクエストを登録
      Database.commit();

//      // イベントログに保存
//      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
//        opinionbox.getOpinionId(),
//        ALEventlogConstants.PORTLET_TYPE_OPINIONBOX,
//        opinionbox_name.getValue());

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
//    try {
//      // メール送信
//      int msgType = ALMailUtils.getSendDestType(ALMailUtils.KEY_MSGTYPE_OPINIONBOX);
//      if (msgType > 0) {
//        // パソコンへメールを送信
//        List<ALEipUserAddr> destMemberList =
//          ALMailUtils.getALEipUserAddrs(
//            mapList,
//            ALEipUtils.getUserId(rundata),
//            false);
//        String subject = "[" + ALOrgUtilsService.getAlias() + "]目安箱";
//        String orgId = Database.getDomainName();
//
//        List<ALAdminMailMessage> messageList =
//          new ArrayList<ALAdminMailMessage>();
//        for (ALEipUserAddr destMember : destMemberList) {
//          ALAdminMailMessage message = new ALAdminMailMessage(destMember);
//          message.setPcSubject(subject);
//          message.setCellularSubject(subject);
//          message.setPcBody(OpinionBoxUtils.createMsgForPc(
//            rundata,
//            opinionbox,
//            mapList,
//            false));
//          message.setCellularBody(OpinionBoxUtils.createMsgForCellPhone(
//            rundata,
//            opinionbox,
//            mapList,
//            false,
//            destMember.getUserId()));
//          messageList.add(message);
//        }
//
//        ALMailService.sendAdminMailAsync(new ALAdminMailContext(
//          orgId,
//          ALEipUtils.getUserId(rundata),
//          messageList,
//          ALMailUtils.getSendDestType(ALMailUtils.KEY_MSGTYPE_OPINIONBOX)));
//        // msgList.addAll(errors);
//
//      }
//    } catch (Exception ex) {
//      msgList.add("メールを送信できませんでした。");
//      logger.error("Exception", ex);
//      return false;
//    }
    return true;
  }


  /**
   * メモを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  /**
   * メモのフィールドを設定します。 <BR>
   *
   * @param str
   * @return
   */
  public void setNote(String str) {
    note.setValue(str);
  }

  /**
   * 報告書名を取得します。 <BR>
   *
   * @return
   */
  public ALStringField getOpinionBoxName() {
    return opinionbox_name;
  }

  /**
   * 報告書名を格納します。 <BR>
   *
   * @param str
   * @return
   */

  public void setOpinionBoxName(String str) {
    opinionbox_name.setValue(str);
  }

  /**
   * 匿名フラグを取得します。 <BR>
   *
   * @return
   */
  public ALNumberField getTokumeiFlg() {
    return tokumei_flg;
  }

  /**
   * 匿名フラグのフィールドを設定します。 <BR>
   *
   * @param str
   * @return
   */
  public void setTokumeiFlg(Integer flg) {
    tokumei_flg.setValue(flg);
  }

  /**
   * グループメンバーを取得します。 <BR>
   *
   * @return
   */
  public List<ALEipUser> getMapList() {
    return mapList;
  }

  /**
   * グループメンバーを格納します。 <BR>
   *
   * @param str
   * @return
   */
  public void setMapList(ArrayList<ALEipUser> list) {
    mapList = list;
  }

  /**
   *
   * @param groupname
   * @return
   */
  public List<ALEipUser> getUsers(String groupname) {
    return ALEipUtils.getUsers(groupname);
  }

  /**
   *
   * @return
   */
  public Map<Integer, ALEipPost> getPostMap() {
    return ALEipManager.getInstance().getPostMap();
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

/**
 * @return superiorUserList
 */
public List<ALEipUser> getSuperiorUserList() {
	return superiorUserList;
}


}
