/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApply;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApplyMemberMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.improveope.util.ImproveopeApplyUtils;
import com.aimluck.eip.improveope.util.ImproveopeUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.modules.actions.improveope.ImproveopeAction;
import com.aimluck.eip.modules.screens.ImproveopeReceiveFormScreen;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 業務改善掲示板の評価フォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopeReceiveFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopeReceiveFormData.class.getName());

  /** ステータス */
  private ALStringField status;

  /** ポイント */
  private ALStringField point;

  private int uid;

  private String orgId;

  private String aclPortletFeature = null;

  /** 管理者権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclImproveopeAdmin;
  private boolean hasAclImproveopeUpdate;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclImproveopeAdmin =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_LIST);

    hasAclImproveopeUpdate =
    	      aclhandler.hasAuthority(
    	        ALEipUtils.getUserId(rundata),
    	        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
    	        ALAccessControlConstants.VALUE_ACL_UPDATE);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {

    // ステータス
    status = new ALStringField();
    status.setFieldName("ステータス");
    status.setTrim(true);
    status.setValue("");

  	point = new ALStringField();
  	point.setCharacterType(ALStringField.TYPE_NUMBER);
  	point.setFieldName("採用ポイント");
  	point.setValue("");

    aclPortletFeature =
      ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN;
  }

  /**
   * 申請の各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // ステータス
    status.setNotNull(true);
    point.setTrim(true);
  }

  /**
   * トピックのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // ステータス
    status.validate(msgList);
    // ポイント
    point.validate(msgList);

    if (status.getValue().equals(ImproveopeApplyUtils.APPLY_STATUS_ADOPT)) {
    	if (point.getValue() == null || point.getValue().equals("") ) {
   		 msgList.add("『 <span class='em'>採用ポイント</span> 』を入力してください。");
    	}
    }


    return (msgList.size() == 0);

  }

  /**
   * トピックをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // オブジェクトモデルを取得
      OriTImproveopeApply topic =
        ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);
      if (topic == null) {
        return false;
      }

      status.setValue(topic.getStatus());

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }


  /**
   * データベースに格納されているトピックを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 新規オブジェクトモデル
      OriTImproveopeApply apply =
        ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);
      // ステータス
      apply.setStatus(status.getValue());
      // 更新者
      apply.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      apply.setUpdateDate(Calendar.getInstance().getTime());

      Database.commit();

      if (status.getValue().equals(ImproveopeApplyUtils.APPLY_STATUS_ADOPT)) {
    	  // 採用ならポイント付与
    	  ImproveopeUtils.grantAdoptPoint(
    			  apply,
    			  Integer.valueOf(point.getValue())
    			  );
      } else if (status.getValue().equals(ImproveopeApplyUtils.APPLY_STATUS_REJECT)) {
    	  // 却下ならポイントを削除
        ImproveopeUtils.grantAdoptPointDelete(apply);
      }

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
	    apply.getApplyId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        apply.getApplyName());

      if (status.getValue().equals(ImproveopeApplyUtils.APPLY_STATUS_CHECK)) {
	      /* 社員評価実施のアクティビティー */
	      ALEipUser user = ALEipUtils.getALEipUser(uid);
	      ImproveopeApplyUtils.createCheckActivity(
	  		  apply,
	  		  user.getName().getValue(),
	  		  null);
      }
      if (status.getValue().equals(ImproveopeApplyUtils.APPLY_STATUS_REJECT)) {
        /* 却下のアクティビティー */
        List<OriTImproveopeApplyMemberMap> map = ImproveopeApplyUtils.getOriTImproveopeApplyMemberMap(apply);
        List<String> memberList = new ArrayList<String>();
        TurbineUser tuser = null;
        for (OriTImproveopeApplyMemberMap user : map) {
          try {
            tuser = ALEipUtils.getTurbineUser(Integer.valueOf(user.getUserId().toString()));
            memberList.add(tuser.getLoginName());
          } catch (ALDBErrorException e) {}
        }

        ALEipUser user = ALEipUtils.getALEipUser(uid);
        ImproveopeApplyUtils.createRejectActivity(
          apply,
          user.getName().getValue(),
          memberList);
      }
      if (status.getValue().equals(ImproveopeApplyUtils.APPLY_STATUS_ADOPT)) {
        /* 採用のアクティビティー */
        List<OriTImproveopeApplyMemberMap> map = ImproveopeApplyUtils.getOriTImproveopeApplyMemberMap(apply);
        List<String> memberList = new ArrayList<String>();
        TurbineUser tuser = null;
        for (OriTImproveopeApplyMemberMap user : map) {
          try {
            tuser = ALEipUtils.getTurbineUser(Integer.valueOf(user.getUserId().toString()));
            memberList.add(tuser.getLoginName());
          } catch (ALDBErrorException e) {}
        }

        ALEipUser user = ALEipUtils.getALEipUser(uid);
        ImproveopeApplyUtils.createAdoptActivity(
          apply,
          user.getName().getValue(),
          memberList);
      }

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 業務改善申請詳細表示ページにフォームを表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doViewForm(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);

      action.setMode("reply");
      List<String> msgList = new ArrayList<String>();
      boolean res = setFormData(rundata, context, msgList);
      if (action instanceof ImproveopeReceiveFormScreen) {
    	  ImproveopeReceiveFormScreen improveopeAction =
          (ImproveopeReceiveFormScreen) action;
        improveopeAction.setResultDataOnReceiveForm(this);
        improveopeAction.addErrorMessagesOnReceiveForm(msgList);
        improveopeAction.putDataOnReceiveForm(rundata, context);
      } else {
        ImproveopeAction improveopeAction = (ImproveopeAction) action;
        improveopeAction.setResultDataOnReceiveForm(this);
        improveopeAction.addErrorMessagesOnReceiveForm(msgList);
        improveopeAction.putDataOnReceiveForm(rundata, context);
      }
      return res;
      // } catch (ALPermissionException e) {
      // ALEipUtils.redirectPermissionError(rundata);
      // return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    // オブジェクトモデルを取得
    OriTImproveopeApply topic =
      ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);
    if (topic == null) {
      return res;
    }

    if (point.getValue().toString().equals("")) {
    	int now_point = ImproveopeUtils.getAdoptPoint(topic);
		if (now_point > 0) {
	      // ステータス
	      point.setValue(String.valueOf(now_point));
		} else {
		  point.setValue(ImproveopeUtils.getDefaultPoint(ImproveopeUtils.POINT_TYPE_P).toString());
		}
    }

    try {

    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    return res;
  }

  /**
   * ステータスを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getStatus() {
    return status;
  }

  /**
   * 申請中かどうか
   * @return
   */
  public boolean hasApply() {
	  if (status.toString().equals(ImproveopeApplyUtils.APPLY_STATUS_APPLY)) {
		  return true;
	  }
	  return false;
  }
  /**
   * @return point
   */
  public ALStringField getPoint() {
  	return point;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  public void setAclPortletFeature(String aclPortletFeature) {
    this.aclPortletFeature = aclPortletFeature;
  }

@Override
protected boolean insertFormData(RunData rundata, Context context,
		List<String> msgList) throws ALPageNotFoundException,
		ALDBErrorException {
	// TODO 自動生成されたメソッド・スタブ
	return false;
}

@Override
protected boolean deleteFormData(RunData rundata, Context context,
		List<String> msgList) throws ALPageNotFoundException,
		ALDBErrorException {
	// TODO 自動生成されたメソッド・スタブ
	return false;
}

/**
 * 他ユーザのトピックを編集する権限があるかどうかを返します。
 *
 * @return
 */
public boolean hasAclUpdateApply() {
  return hasAclImproveopeUpdate;
}

/**
 * 他ユーザの申請を見る権限があるかどうかを返します。
 *
 * @return
 */
public boolean hasAclAdminApply() {
  return hasAclImproveopeAdmin;
}


}
