/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriMPoint;
import com.aimluck.eip.cayenne.om.portlet.OriTPointHistory;


import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;
import com.aimluck.eip.fileupload.beans.FileuploadLiteBean;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.improveope.util.ImproveopeUtils;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.services.storage.ALStorageService;
import com.aimluck.eip.timeline.util.TimelineUtils;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 掲示板トピックのフォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopePointManagementFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopePointManagementFormData.class.getName());


  /** ポイントID */
  private ALNumberField point_id;

  /** ポイント名 */
  private ALStringField point_name;

  /** ポイント */
  private ALStringField point;

  /** ユーザーID */
  private int uid;

  /** デフォルトタイプ */
  private final String DEFAULT_TYPE = "O";

  /** 削除判定フラグ */
  private boolean deleteFlg = false;

  /** 閲覧権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclCategoryList;

  /** 他ユーザーの作成したトピックの編集権限 */
  private boolean hasAclUpdateTopicOthers;

  /** 他ユーザーの作成したトピックの削除権限 */
  private boolean hasAclDeleteTopicOthers;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    uid = ALEipUtils.getUserId(rundata);

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclCategoryList =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF,
        ALAccessControlConstants.VALUE_ACL_LIST);

    hasAclDeleteTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_DELETE);

    hasAclUpdateTopicOthers =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_UPDATE);

  }




  @Override
  public void initField() {

	point_id = new ALNumberField();

  	point_name = new ALStringField();
  	point_name.setFieldName("ポイント名");

  	point = new ALStringField();
  	point.setCharacterType(ALStringField.TYPE_NUMBER);
  	point.setFieldName("ポイント数");

  }


  @Override
  protected void setValidator() throws ALPageNotFoundException,
  		ALDBErrorException {

	  point_name.setNotNull(true);
	  point_name.limitMaxLength(60);
	  point.setNotNull(true);
	  point.setTrim(true);
  }



  @Override
  protected boolean validate(List<String> msgList)
  		throws ALPageNotFoundException, ALDBErrorException {

	 point_name.validate(msgList);
	 point.validate(msgList);

    return (msgList.size() == 0);
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);

    return res;
  }

  @Override
  protected boolean loadFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {
	  try {

		  OriMPoint pointMst = ImproveopeUtils.getOriMPoint(rundata, context);

		  if (point == null ) {
			  return false;
		  }

		  // ポイントID
		  point_id.setValue(pointMst.getPointId());
		  // ポイント名
		  point_name.setValue(pointMst.getPointName());
		  // ポイント数
		  point.setValue(pointMst.getPoint().toString());

		  // 削除判定フラグ
		  deleteFlg = getDelFlg(pointMst);

      } catch (Exception ex) {
	        logger.error("Exception", ex);
	        return false;
	  }
	  return true;
  }



  @Override
  protected boolean insertFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {

	  try {
		  // オブジェクトを作成
		  OriMPoint pointMst = Database.create(OriMPoint.class);

		  // ポイント名
		  pointMst.setPointName(point_name.getValue());
		  // ポイント数
		  pointMst.setPoint(Integer.valueOf(point.getValue()));
		  // タイプ
		  pointMst.setType(DEFAULT_TYPE);
		  // 作成者
		  pointMst.setCreateUserId(uid);
		  // 作成日時
		  Date now = Calendar.getInstance().getTime();
		  pointMst.setCreateDate(now);
		  // 更新者
		  pointMst.setUpdateUserId(uid);
		  // 更新日時
		  pointMst.setUpdateDate(now);

		  Database.commit();

	        // イベントログに保存
	        ALEventlogFactoryService.getInstance().getEventlogHandler().log(
	          pointMst.getPoint(),
	          ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
	          "ポイント管理 追加");

	      } catch (Exception ex) {
	        logger.error("Exception", ex);
	        return false;
	      }
	      return true;

  }



  @Override
  protected boolean updateFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {
  	try {

  		OriMPoint pointMst = ImproveopeUtils.getOriMPoint(rundata, context);

  		// ポイント名
  		pointMst.setPointName(point_name.getValue());
  		// ポイント数
  		pointMst.setPoint(Integer.valueOf(point.getValue()));
  		// 更新者
  		pointMst.setUpdateUserId(uid);
  		// 更新時間
  		pointMst.setUpdateDate(Calendar.getInstance().getTime());

  		Database.commit();

        // イベントログに保存
        ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          pointMst.getPoint(),
          ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
          "ポイント管理 更新");

  	} catch (Exception ex) {
        logger.error("Exception [ update ]", ex);
        return false;
  	}
  	return true;
  }



  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
  		List<String> msgList) throws ALPageNotFoundException,
  		ALDBErrorException {

	  try {

		  // オブジェクトを取得
		  OriMPoint pointMst = ImproveopeUtils.getOriMPoint(rundata, context);

		  if (pointMst == null) {
			  return false;
		  }

		  Database.delete(pointMst);
		  Database.commit();

        // イベントログに保存
        ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          pointMst.getPoint(),
          ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
          "ポイント管理 削除");

	  	} catch (Exception ex) {
	        logger.error("Exception [ delete ]", ex);
	        return false;
	  	}
	  return true;
  }


  private boolean getDelFlg(OriMPoint pointMst) {

	  try {

		  if(Arrays.asList(ImproveopeUtils.pointType).contains(pointMst.getType())){
			  return false;
		  }

		  SelectQuery<OriTPointHistory> query = Database.query(OriTPointHistory.class);
		  Expression exp =
				  ExpressionFactory.matchExp(OriTPointHistory.POINT_ID_PROPERTY, pointMst.getPointId());
		  query.setQualifier(exp);

		  if (query.fetchList().size() > 0) {
			  return false;
		  }

	  } catch (Exception ex) {
        logger.error("Exception [ delete ]", ex);
        return false;
	  }

	  return true;
  }

  /**
   * 他ユーザのトピックを編集する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclUpdateTopicOthers() {
    return hasAclUpdateTopicOthers;
  }

  /**
   * 他ユーザのトピックを削除する権限があるかどうかを返します。
   *
   * @return
   */
  public boolean hasAclDeleteTopicOthers() {
    return hasAclDeleteTopicOthers;
  }


/**
 * @return point_name
 */
public ALStringField getPointName() {
	return point_name;
}


/**
 * @return point
 */
public ALStringField getPoint() {
	return point;
}




/**
 * @return point_id
 */
public ALNumberField getPointId() {
	return point_id;
}


/**
 * @return deleteFlg
 */
public boolean getDeleteFlg() {
	return deleteFlg;
}

}
