/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApply;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApplyCheck;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.improveope.util.ImproveopeApplyUtils;
import com.aimluck.eip.improveope.util.ImproveopeUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.modules.actions.improveope.ImproveopeAction;
import com.aimluck.eip.modules.screens.ImproveopeCheckFormScreen;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 業務改善掲示板の受信フォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopeCheckFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopeCheckFormData.class.getName());

  /** 評価 */
  private ALStringField answer;

  /** メモ */
  private ALStringField note;

  /** ID */
  private ALStringField check_id;

  private int uid;

  private String orgId;

  private String aclPortletFeature = null;

  /** 閲覧権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclCategoryList;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();

    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclCategoryList =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF,
        ALAccessControlConstants.VALUE_ACL_LIST);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {

    // ステータス
	answer = new ALStringField();
	answer.setFieldName("評価");
	answer.setTrim(true);
	answer.setValue("");

    // メモ
    note = new ALStringField();
    note.setFieldName("コメント");
    note.setTrim(false);

    check_id = new ALStringField();

    aclPortletFeature =
      ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_SELF;
  }

  /**
   * 申請の各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // 評価
	answer.setNotNull(true);
    // メモ必須項目
    note.setNotNull(false);
    // メモの文字数制限
    note.limitMaxLength(10000);
  }

  /**
   * トピックのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // 評価
    answer.validate(msgList);
    // メモ
    note.validate(msgList);

    return (msgList.size() == 0);

  }

  /**
   * トピックをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // オブジェクトモデルを取得
      OriTImproveopeApplyCheck check =
        ImproveopeApplyUtils.getOriTImproveopeApplyCheck(rundata, context);
      if (check == null) {
        return false;
      }

      // 評価
      answer.setValue(check.getAnswer());
      // コメント
      note.setValue(check.getNote());
      // ID
      check_id.setValue(check.getId().toString());

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }


  /**
   * データベースに格納されているトピックを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 新規オブジェクトモデル
      OriTImproveopeApplyCheck apply =
        ImproveopeApplyUtils.getOriTImproveopeApplyCheck(rundata, context);
      // ステータス
      apply.setAnswer(answer.getValue());
      // メモ
      apply.setNote(note.getValue());
      // 更新者
      apply.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      apply.setUpdateDate(Calendar.getInstance().getTime());

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
	    apply.getId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        apply.getAnswer());

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 業務改善申請詳細表示ページにフォームを表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doViewForm(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);

	    // オブジェクトモデルを取得
	    OriTImproveopeApplyCheck check =
	      ImproveopeApplyUtils.getOriTImproveopeApplyCheck(rundata, context);
	    if (check != null) {
		    // 評価
		    answer.setValue(check.getAnswer());
		    // コメント
		    note.setValue(check.getNote());
		    // ID
		    check_id.setValue(check.getId().toString());
	    }

      action.setMode("reply");
      List<String> msgList = new ArrayList<String>();
      boolean res = setFormData(rundata, context, msgList);
      if (action instanceof ImproveopeCheckFormScreen) {
    	  ImproveopeCheckFormScreen improveopeAction =
          (ImproveopeCheckFormScreen) action;
        improveopeAction.setResultDataOnCheckForm(this);
        improveopeAction.addErrorMessagesOnCheckForm(msgList);
        improveopeAction.putDataOnCheckForm(rundata, context);
      } else {
        ImproveopeAction improveopeAction = (ImproveopeAction) action;
        improveopeAction.setResultDataOnCheckForm(this);
        improveopeAction.addErrorMessagesOnCheckForm(msgList);
        improveopeAction.putDataOnCheckForm(rundata, context);
      }
      return res;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    try {
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }

    return res;
  }

  /**
   * 評価を取得します。 <BR>
   *
   * @return
   */
  public ALStringField getAnswer() {
    return answer;
  }

  /**
   * メモを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  /**
   * IDを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getId() {
    return check_id;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  public void setAclPortletFeature(String aclPortletFeature) {
    this.aclPortletFeature = aclPortletFeature;
  }

  /**
   * トピックをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
        // 新規オブジェクトモデル
        OriTImproveopeApply apply =
          ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);

      // 新規オブジェクトモデル
      OriTImproveopeApplyCheck check = Database.create(OriTImproveopeApplyCheck.class);
      // 親ID
      check.setOriTImproveopeApply(apply);
      // トピック名
      check.setAnswer(answer.getValue());
      // ユーザーID
      check.setUserId(Integer.valueOf(uid));
      // メモ
      check.setNote(note.getValue());
      // 作成者
      check.setCreateUserId(Integer.valueOf(uid));
      // 更新者
      check.setUpdateUserId(Integer.valueOf(uid));
      // 作成日
      check.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      check.setUpdateDate(Calendar.getInstance().getTime());

      Database.commit();

      // 評価ポイント付与
      ImproveopeUtils.grantCheckPoint(check);

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
    		  check.getId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        check.getAnswer());
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }
@Override
protected boolean deleteFormData(RunData rundata, Context context,
		List<String> msgList) throws ALPageNotFoundException,
		ALDBErrorException {
	// TODO 自動生成されたメソッド・スタブ
	return false;
}


}
