/*
 * Copyright(C) 2012 - 2013 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.improveope;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApply;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApplyAction;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApplyActionFile;
import com.aimluck.eip.cayenne.om.portlet.OriTImproveopeApplyActionMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.fileupload.beans.FileuploadLiteBean;
import com.aimluck.eip.improveope.util.ImproveopeApplyUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.modules.actions.improveope.ImproveopeAction;
import com.aimluck.eip.modules.screens.ImproveopeActionFormScreen;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.services.storage.ALStorageService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 実施要項作成フォームデータを管理するクラスです。 <BR>
 *
 */
public class ImproveopeActionFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ImproveopeActionFormData.class.getName());

  /** 目標 */
  private ALStringField goal;

  /** 内容 */
  private ALStringField note;

  /** 実施開始日 */
  private ALDateTimeField start_date;

  /** 実施終了日 */
  private ALDateTimeField end_date;

  /** 添付ファイルリスト */
  private List<FileuploadLiteBean> fileuploadList = null;

  /** 添付フォルダ名 */
  private String folderName = null;

  /** 実施対象者ユーザIDリスト */
  private ALStringField members;

  /** 実施対象者一覧 */
  private List<ALEipUser> mapList;

  /** アクションID */
  private ALStringField action_id;

  private int uid;

  private String orgId;

  private String aclPortletFeature = null;

  /** 管理者権限の有無 */
  @SuppressWarnings("unused")
  private boolean hasAclImproveopeAdmin;
  @SuppressWarnings("unused")
  private boolean hasAclImproveopeUpdate;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    uid = ALEipUtils.getUserId(rundata);
    orgId = Database.getDomainName();

    folderName = rundata.getParameters().getString("folderName");


    ALAccessControlFactoryService aclservice =
      (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
        .getInstance()).getService(ALAccessControlFactoryService.SERVICE_NAME);
    ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

    hasAclImproveopeAdmin =
      aclhandler.hasAuthority(
        ALEipUtils.getUserId(rundata),
        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
        ALAccessControlConstants.VALUE_ACL_LIST);

    hasAclImproveopeUpdate =
    	      aclhandler.hasAuthority(
    	        ALEipUtils.getUserId(rundata),
    	        ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN,
    	        ALAccessControlConstants.VALUE_ACL_UPDATE);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {

    // 目標
	goal = new ALStringField();
	goal.setFieldName("目標");
	goal.setTrim(false);

    // 内容
    note = new ALStringField();
    note.setFieldName("実施内容");
    note.setTrim(false);

    // 実施開始日
    start_date = new ALDateTimeField("yyyy-MM-dd-HH-mm");
    start_date.setValue(new Date());

    // 実施終了日
    end_date = new ALDateTimeField("yyyy-MM-dd-HH-mm");
    end_date.setValue(new Date());

    // 実施対象者リスト
    members = new ALStringField();
    members.setFieldName("実施対象者");
    members.setTrim(true);
    // 実施対象者リスト
    mapList = new ArrayList<ALEipUser>();

    fileuploadList = new ArrayList<FileuploadLiteBean>();

    action_id = new ALStringField();

    aclPortletFeature =
      ALAccessControlConstants.POERTLET_FEATURE_IMPROVEOPE_ADMIN;
  }

  /**
   * 申請の各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // 目標必須項目
	goal.setNotNull(true);
    // 目標の文字数制限
	goal.limitMaxLength(10000);
    // 内容必須項目
    note.setNotNull(true);
    // 内容の文字数制限
    note.limitMaxLength(10000);
  }

  /**
   * トピックのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // 目標
    goal.validate(msgList);
    // 内容
    note.validate(msgList);

    // 実施対象者
    if (mapList == null || mapList.size() <= 0) {
      msgList.add("『 <span class='em'>実施対象者</span> 』を指定してください。");
    } else if (!(mapList.get(0) instanceof ALEipUser)) {
      msgList.add("実施対象者のユーザーが全て無効、もしくは削除されています。有効なユーザーを一人以上選択してください。");
    }

    return (msgList.size() == 0);

  }

  /**
   * トピックをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

        // オブジェクトモデルを取得
        OriTImproveopeApplyAction action =
          ImproveopeApplyUtils.getOriTImproveopeApplyAction(rundata, context);
        if (action == null) {
          return false;
        }

        // ID
        action_id.setValue(action.getActionId().toString());

        // 目標
        goal.setValue(action.getGoal());

        // 内容
        note.setValue(action.getNote());

        Date now = Calendar.getInstance().getTime();// 現在時刻取得
        // 実施開始日を取得
        if (action.getStartDate() != null) {
      	    start_date.setValue(action.getStartDate());
        } else {
        	start_date.setValue(now);
        }

        // 実施終了日を取得
        if (action.getEndDate() != null) {
      	    end_date.setValue(action.getEndDate());
        } else {
        	end_date.setValue(now);
        }

        // ファイル
        SelectQuery<OriTImproveopeApplyActionFile> query =
          Database.query(OriTImproveopeApplyActionFile.class);
        query.andQualifier(ExpressionFactory.matchDbExp(
        		OriTImproveopeApplyActionFile.ORI_TIMPROVEOPE_APPLY_ACTION_PROPERTY,
        		action.getOriTImproveopeApply().getApplyId()));
        List<OriTImproveopeApplyActionFile> improveopeFileList = query.fetchList();
        for (OriTImproveopeApplyActionFile file : improveopeFileList) {
          FileuploadLiteBean fbean = new FileuploadLiteBean();
          fbean.initField();
          fbean.setFileId(file.getFileId());
          fbean.setFileName(file.getFileName());
          fileuploadList.add(fbean);
        }

        // 実施対象者
        List<OriTImproveopeApplyActionMap> maps = ImproveopeApplyUtils.getOriTImproveopeApplyActionMap(action);
        OriTImproveopeApplyActionMap map = null;
        int size = maps.size();
        for (int i = 0; i < size; i++) {
          map = maps.get(i);
          int user_id = map.getUserId().intValue();
          mapList.add(ALEipUtils.getALEipUser(user_id));
        }

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }


  /**
   * データベースに格納されているトピックを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {

      // 親オブジェクト
      OriTImproveopeApply apply =
        ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);
      // オブジェクトモデル
      OriTImproveopeApplyAction action =
        ImproveopeApplyUtils.getOriTImproveopeApplyAction(rundata, context);
      // 目標
      action.setGoal(goal.getValue());
      // 内容
      action.setNote(note.getValue());
      // 更新者
      action.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      action.setUpdateDate(Calendar.getInstance().getTime());
      // 実施開始日
      SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
      Date date1 = format.parse(rundata.getParameters().getString("start_date").replace("/", "-") + " 00:00:00");
      action.setStartDate(date1);
      // 実施終了日
      Date date2 = format.parse(rundata.getParameters().getString("end_date").replace("/", "-") + " 23:59:59");
      action.setEndDate(date2);
      // ファイルをデータベースに登録する．
      if (!ImproveopeApplyUtils.insertFileDataDelegateAction(
        rundata,
        context,
        action,
        fileuploadList,
        folderName,
        msgList)) {
        return false;
      }
      // 古いマップデータを削除
      List<OriTImproveopeApplyActionMap> tmp_map = ImproveopeApplyUtils.getOriTImproveopeApplyActionMap(action);
      Database.deleteAll(tmp_map);
      // 実施対象者
      for (ALEipUser user : mapList) {
    	  OriTImproveopeApplyActionMap map = Database.create(OriTImproveopeApplyActionMap.class);
        int userid = (int) user.getUserId().getValue();

        map.setOriTImproveopeApplyAction(action);
        map.setUserId(Integer.valueOf(userid));
      }


      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
    	action.getActionId(),
        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
        apply.getApplyName());


      /* 実施要項のアクティビティー */
      List<String> memberList = new ArrayList<String>();
      TurbineUser tuser = null;
      for (ALEipUser user : mapList) {
          try {
			tuser = ALEipUtils.getTurbineUser(Integer.valueOf(user.getUserId().toString()));
			memberList.add(tuser.getLoginName());
          } catch (ALDBErrorException e) {}
      }
      ALEipUser user = ALEipUtils.getALEipUser(uid);
      ImproveopeApplyUtils.createActionActivity(
	      apply,
		  user.getName().getValue(),
		  memberList,
		  false);

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 業務改善申請詳細表示ページにフォームを表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doViewForm(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);

	    // オブジェクトモデルを取得
	    OriTImproveopeApplyAction check =
	      ImproveopeApplyUtils.getOriTImproveopeApplyAction(rundata, context);
	    if (check != null) {
		    // 目標
		    goal.setValue(check.getGoal());
		    // 内容
		    note.setValue(check.getNote());
		    // ID
		    action_id.setValue(check.getActionId().toString());
	        // 実施開始日を取得
      	    start_date.setValue(check.getStartDate());
	        // 実施終了日を取得
      	    end_date.setValue(check.getEndDate());
	        // ファイル
	        SelectQuery<OriTImproveopeApplyActionFile> query =
	          Database.query(OriTImproveopeApplyActionFile.class);
	        query.andQualifier(ExpressionFactory.matchDbExp(
	        		OriTImproveopeApplyActionFile.ORI_TIMPROVEOPE_APPLY_ACTION_PROPERTY,
	        		check.getActionId()));
	        List<OriTImproveopeApplyActionFile> improveopeFileList = query.fetchList();
	        for (OriTImproveopeApplyActionFile file : improveopeFileList) {
	          FileuploadLiteBean fbean = new FileuploadLiteBean();
	          fbean.initField();
	          fbean.setFileId(file.getFileId());
	          fbean.setFileName(file.getFileName());
	          fileuploadList.add(fbean);
	        }

	        // 実施対象者
	        List<OriTImproveopeApplyActionMap> maps = ImproveopeApplyUtils.getOriTImproveopeApplyActionMap(check);
	        OriTImproveopeApplyActionMap map = null;
	        int size = maps.size();
	        for (int i = 0; i < size; i++) {
	          map = maps.get(i);
	          int user_id = map.getUserId().intValue();
	          mapList.add(ALEipUtils.getALEipUser(user_id));
	        }
	    }


      action.setMode("reply");
      List<String> msgList = new ArrayList<String>();
      boolean res = setFormData(rundata, context, msgList);
      if (action instanceof ImproveopeActionFormScreen) {
    	  ImproveopeActionFormScreen improveopeAction =
          (ImproveopeActionFormScreen) action;
        improveopeAction.setResultDataOnActionForm(this);
        improveopeAction.addErrorMessagesOnActionForm(msgList);
        improveopeAction.putDataOnActionForm(rundata, context);
      } else {
        ImproveopeAction improveopeAction = (ImproveopeAction) action;
        improveopeAction.setResultDataOnActionForm(this);
        improveopeAction.addErrorMessagesOnActionForm(msgList);
        improveopeAction.putDataOnActionForm(rundata, context);
      }
      return res;
      // } catch (ALPermissionException e) {
      // ALEipUtils.redirectPermissionError(rundata);
      // return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }
  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    try {

    	if (fileuploadList.size() == 0 && mapList.size() == 0) {
	        fileuploadList = ImproveopeApplyUtils.getActionFileuploadList(rundata);

	        String userNames[] = rundata.getParameters().getStrings("members");
	        if (userNames != null && userNames.length > 0) {
	          SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
	          Expression exp =
	            ExpressionFactory.inExp(TurbineUser.LOGIN_NAME_PROPERTY, userNames);
	          query.setQualifier(exp);
	          mapList.addAll(ALEipUtils.getUsersFromSelectQuery(query));
	        }
    	}

      } catch (Exception ex) {
        logger.error("Exception", ex);
      }

    return res;
  }


  /**
   * 目標を取得します。 <BR>
   *
   * @return
   */
  public ALStringField getGoal() {
    return goal;
  }

  /**
   * メモを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  /**
   * IDを取得します。 <BR>
   *
   * @return
   */
  public ALStringField getId() {
    return action_id;
  }

  /**
   * 実施開始日を取得します。 <BR>
   *
   * @return
   */
  public ALDateTimeField getStartDate() {
    return start_date;
  }

  /**
   * 実施終了日を取得します。 <BR>
   *
   * @return
   */
  public ALDateTimeField getEndDate() {
    return end_date;
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return aclPortletFeature;
  }

  public void setAclPortletFeature(String aclPortletFeature) {
    this.aclPortletFeature = aclPortletFeature;
  }

  /**
   * 実施要項をデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
	@Override
	protected boolean insertFormData(RunData rundata, Context context,
	    List<String> msgList) {
	  try {
	      // 新規オブジェクトモデル
	      OriTImproveopeApply apply =
	        ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);

		  // 新規オブジェクトモデル
		  OriTImproveopeApplyAction action = Database.create(OriTImproveopeApplyAction.class);
		  // 親ID
		  action.setOriTImproveopeApply(apply);
	      // 目標
	      action.setGoal(goal.getValue());
	      // 内容
	      action.setNote(note.getValue());
	      // 作成者
	      action.setCreateUserId(Integer.valueOf(uid));
	      // 更新者
	      action.setUpdateUserId(Integer.valueOf(uid));
	      // 作成日
	      action.setCreateDate(Calendar.getInstance().getTime());
	      // 更新日
	      action.setUpdateDate(Calendar.getInstance().getTime());
	      // 実施開始日
	      SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	      Date date1 = format.parse(rundata.getParameters().getString("start_date").replace("/", "-") + " 00:00:00");
	      action.setStartDate(date1);
	      // 実施終了日
	      Date date2 = format.parse(rundata.getParameters().getString("end_date").replace("/", "-") + " 23:59:59");
	      action.setEndDate(date2);
	      // ファイルをデータベースに登録する．
	      if (!ImproveopeApplyUtils.insertFileDataDelegateAction(
	        rundata,
	        context,
	        action,
	        fileuploadList,
	        folderName,
	        msgList)) {
	        return false;
	      }
	      // 実施対象者
	      for (ALEipUser user : mapList) {
	    	  OriTImproveopeApplyActionMap map = Database.create(OriTImproveopeApplyActionMap.class);
	        int userid = (int) user.getUserId().getValue();

	        map.setOriTImproveopeApplyAction(action);
	        map.setUserId(Integer.valueOf(userid));
	      }

	    Database.commit();

	    // イベントログに保存
	    ALEventlogFactoryService.getInstance().getEventlogHandler().log(
	      action.getActionId(),
	      ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
	      apply.getApplyName());

	      /* 実施要項のアクティビティー */
	      List<String> memberList = new ArrayList<String>();
	      TurbineUser tuser = null;
	      for (ALEipUser user : mapList) {
	          try {
				tuser = ALEipUtils.getTurbineUser(Integer.valueOf(user.getUserId().toString()));
				memberList.add(tuser.getLoginName());
	          } catch (ALDBErrorException e) {}
	      }
	      ALEipUser user = ALEipUtils.getALEipUser(uid);
	      ImproveopeApplyUtils.createActionActivity(
		      apply,
			  user.getName().getValue(),
			  memberList,
			  true);

	  } catch (Exception ex) {
	    logger.error("Exception", ex);
	    return false;
	  }
	  return true;
	}

	  /**
	   * 実施要項をデータベースから削除します。 <BR>
	   *
	   * @param rundata
	   * @param context
	   * @param msgList
	   * @return TRUE 成功 FALSE 失敗
	   */
	  @Override
	  protected boolean deleteFormData(RunData rundata, Context context,
	      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
	    try {

	      // 新規オブジェクトモデル
	      OriTImproveopeApply apply =
	        ImproveopeApplyUtils.getOriTImproveopeApply(rundata, context);

	      // オブジェクトモデル
	      OriTImproveopeApplyAction action =
	        ImproveopeApplyUtils.getOriTImproveopeApplyAction(rundata, context);


	      List<String> fpaths = new ArrayList<String>();
          List<?> files = action.getOriTImproveopeApplyActionFile();
          if (files != null && files.size() > 0) {
            int fsize = files.size();
            for (int j = 0; j < fsize; j++) {
              fpaths.add(((OriTImproveopeApplyActionFile) files.get(j)).getFilePath());
            }
          }

	      Database.delete(action);
	      Database.commit();

	      if (fpaths.size() > 0) {
	        // ローカルファイルに保存されているファイルを削除する．
	        int fsize = fpaths.size();
	        for (int i = 0; i < fsize; i++) {
	          ALStorageService.deleteFile(ImproveopeApplyUtils.getSaveDirPath(orgId, uid)
	            + fpaths.get(i));
	        }
	      }

	      // イベントログに保存
	      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
	        action.getActionId(),
	        ALEventlogConstants.PORTLET_TYPE_IMPROVEOPE,
	        apply.getApplyName());

	    } catch (Exception e) {
	      Database.rollback();
	      logger.error("[ImproveopeApplyFormData]", e);
	      throw new ALDBErrorException();
	    }
	    return true;
	  }
/**
 * 他ユーザのトピックを編集する権限があるかどうかを返します。
 *
 * @return
 */
public boolean hasAclUpdateApply() {
  return hasAclImproveopeUpdate;
}

public List<FileuploadLiteBean> getAttachmentFileNameList() {
    return fileuploadList;
  }

  public String getFolderName() {
    return folderName;
  }

  /**
   * グループメンバーを取得します。 <BR>
   *
   * @return
   */
  public List<ALEipUser> getMapList() {
    return mapList;
  }

  /**
   * グループメンバーを格納します。 <BR>
   *
   * @param str
   * @return
   */
  public void setMapList(ArrayList<ALEipUser> list) {
    mapList = list;
  }


}
