/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.exttimecard;

import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALDateField;
import com.aimluck.commons.field.ALDateTimeField;
import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.account.util.AccountUtils;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecard;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecardSystem;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.common.ALPermissionException;
import com.aimluck.eip.exttimecard.util.ExtTimecardUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.services.social.ALActivityService;
import com.aimluck.eip.services.social.model.ALActivityPutRequest;
import com.aimluck.eip.util.ALEipUtils;

/**
 * タイムカードのフォームデータを管理するクラスです。 <BR>
 *
 */
public class ExtTimecardFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(ExtTimecardFormData.class.getName());

  private ALNumberField timecard_id;

  private ALNumberField user_id;

  private ALDateTimeField punch_date;

  private ALStringField type;

  private ALDateTimeField clock_in_time;

  private ALDateTimeField clock_out_time;

  private ALStringField reason;

  private ALDateTimeField outgoing_time1;

  private ALDateTimeField outgoing_time2;

  private ALDateTimeField outgoing_time3;

  private ALDateTimeField outgoing_time4;

  private ALDateTimeField outgoing_time5;

  private ALDateTimeField comeback_time1;

  private ALDateTimeField comeback_time2;

  private ALDateTimeField comeback_time3;

  private ALDateTimeField comeback_time4;

  private ALDateTimeField comeback_time5;

  private ALDateTimeField outgoing_comeback;

  private ALStringField remarks;

  private ALDateField create_date;

  private ALDateField update_date;

  private int entity_id;

  private int login_uid;

  private String edit_mode;

  private String alt_mode;

  private String old_clock_in_time_hour;

  private String old_clock_in_time_minute;

  private String old_clock_out_time_hour;

  private String old_clock_out_time_minute;

  private ALStringField rest_num;

  /** タイムカードの設定 */
  private EipTExtTimecardSystem timecard_system;

  private int current_clock_out_time_hour;

  private int current_clock_out_time_minute;

  /** 追加項目 */
  // 当直代行時間
  private ALStringField waiting_time;

  // 時間有休
  private ALStringField used_time;

  // 登録前の時間有休
  private String old_used_time;

  // 当直フラグ
  private ALStringField watch_flag;

  // 一人応援フラグ
  private ALStringField support_flag;

  // 移動手当フラグ
  private ALStringField move_flag;

  // 車出手当フラグ
  private ALStringField car_flag;

  // 遠隔手当1フラグ
  private ALStringField help1_flag;

  // 遠隔手当2フラグ
  private ALStringField help2_flag;

  // ステータス
  private ALStringField status;

  // 休憩時間
  private ALStringField break_time;

  // 出退勤時刻編集フラグ
  private ALStringField edit_flag;

  // 承認者ID
  private ALNumberField approval_user_id;

  // 休日加算フラグ
  private ALStringField addholiday_flag;

  // 休消手当フラグ
  private ALStringField subholiday_flag;

  /**
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    login_uid = ALEipUtils.getUserId(rundata);

    // 出勤・退勤時間
    old_clock_in_time_hour =
      rundata.getParameters().get("old_clock_in_time_hour");
    old_clock_in_time_minute =
      rundata.getParameters().get("old_clock_in_time_minute");
    old_clock_out_time_hour =
      rundata.getParameters().get("old_clock_out_time_hour");
    old_clock_out_time_minute =
      rundata.getParameters().get("old_clock_out_time_minute");

    timecard_system =
      ExtTimecardUtils.getEipTExtTimecardSystemCurrentUserId(rundata, context);

    // 有休利用時間
    old_used_time = rundata.getParameters().get("old_used_time");
    if ("".equals(old_used_time)) {
    	old_used_time = "0";
    }
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {
    timecard_id = new ALNumberField();
    user_id = new ALNumberField();

    type = new ALStringField();
    type.setFieldName("種類");
    type.setValue("");

    punch_date = new ALDateTimeField();
    punch_date.setFieldName("日付");

    clock_in_time = new ALDateTimeField();
    clock_in_time.setFieldName("勤怠時間");
    clock_out_time = new ALDateTimeField();

    outgoing_comeback = new ALDateTimeField();
    outgoing_comeback.setFieldName("外出時間");

    reason = new ALStringField();
    reason.setFieldName("却下理由");
    reason.setValue("");

    remarks = new ALStringField();
    remarks.setFieldName("連絡事項");
    remarks.setValue("");

    create_date = new ALDateField();
    create_date.setValue(new Date());

    update_date = new ALDateField();
    update_date.setValue(new Date());

    waiting_time = new ALStringField();
    waiting_time.setFieldName("当直代行時間");

    used_time = new ALStringField();
    used_time.setFieldName("有休時間");

    watch_flag = new ALStringField();
    watch_flag.setFieldName("当直");

    support_flag = new ALStringField();
    support_flag.setFieldName("一人応援");

    move_flag = new ALStringField();
    move_flag.setFieldName("移動手当");

    car_flag = new ALStringField();
    car_flag.setFieldName("車出手当");

    help1_flag = new ALStringField();
    help1_flag.setFieldName("遠隔手当(1)");

    help2_flag = new ALStringField();
    help2_flag.setFieldName("遠隔手当(2)");

    status = new ALStringField();
    status.setFieldName("ステータス");

    break_time = new ALStringField();
    break_time.setFieldName("休憩時間");

    addholiday_flag = new ALStringField();
    addholiday_flag.setFieldName("休日加算");

    subholiday_flag = new ALStringField();
    subholiday_flag.setFieldName("休消手当");

    edit_flag = new ALStringField();

    approval_user_id = new ALNumberField();

    edit_mode = "";

    alt_mode = "";

    rest_num = new ALStringField();
    rest_num.setValue("0");

    try {
      Field field;
      for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
        field = this.getClass().getDeclaredField("outgoing_time" + i);
        field.set(this, new ALDateTimeField());
        field = this.getClass().getDeclaredField("comeback_time" + i);
        field.set(this, new ALDateTimeField());
      }
    } catch (Exception e) {
      return;
    }
  }

  /**
   * フォームを表示します。
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  public boolean doViewForm(ALAction action, RunData rundata, Context context) {
    try {
      init(action, rundata, context);
      boolean isedit =
        (ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID) != null);

      action.setMode(isedit
        ? ALEipConstants.MODE_EDIT_FORM
        : ALEipConstants.MODE_NEW_FORM);
      setMode(action.getMode());

      List<String> msgList = new ArrayList<String>();
      boolean res =
        (isedit) ? loadFormData(rundata, context, msgList) : setFormData(
          rundata,
          context,
          msgList);

      int aclType = ALAccessControlConstants.VALUE_ACL_INSERT;
      if (isedit || (getIsPast() || getIsToday())) {
        aclType = ALAccessControlConstants.VALUE_ACL_UPDATE;
      }
      doCheckAclPermission(rundata, context, aclType);

      action.setResultData(this);
      if (!msgList.isEmpty()) {
        action.addErrorMessages(msgList);
      }
      action.putData(rundata, context);
      return res;
    } catch (ALPermissionException e) {
      ALEipUtils.redirectPermissionError(rundata);
      return false;
    } catch (ALPageNotFoundException e) {
      ALEipUtils.redirectPageNotFound(rundata);
      return false;
    } catch (ALDBErrorException e) {
      ALEipUtils.redirectDBError(rundata);
      return false;
    }

  }

  /**
   * タイムカードの各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {
    // reason.setNotNull(true);
    // reason.limitMaxLength(1000);

    // work_date.setNotNull(true);

  }

  /**
   * タイムカードのフォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
    try {
      if (type.getValue().equals("")) {
        msgList.add("『 <span class='em'>種類</span> 』を選択してください。");
      }

      if (!"punchin".equals(edit_mode)
        && !"punchout".equals(edit_mode)
        && !"outgoing".equals(edit_mode)
        && !"comeback".equals(edit_mode)
        && !"approval".equals(edit_mode)) {
        if ((old_clock_in_time_hour != clock_in_time.getHour())
          || (old_clock_in_time_minute != clock_in_time.getMinute())
          || (old_clock_out_time_hour != clock_out_time.getHour())
          || (old_clock_out_time_minute != clock_out_time.getMinute())) {
          if (Calendar.getInstance().getTime().after(punch_date.getValue())) {
            // 修正理由は入力しない
            // reason.setNotNull(true);
            // reason.validate(msgList);
          }
          remarks.validate(msgList);
        }
      }

      if (getMode() == ALEipConstants.MODE_INSERT) {
        SelectQuery<EipTExtTimecard> workflg_query =
          Database.query(EipTExtTimecard.class);
        Expression workflg_exp =
          ExpressionFactory.matchExp(EipTExtTimecard.USER_ID_PROPERTY, Integer
            .valueOf(login_uid));
        workflg_query.setQualifier(workflg_exp);
        workflg_query.orderDesending(EipTExtTimecard.PUNCH_DATE_PROPERTY);
        List<EipTExtTimecard> workflg_list = workflg_query.fetchList();
        if (workflg_list != null && workflg_list.size() > 0) {
        } else {
        }
      }

      /** 有休利用 */
      if (edit_mode.equals("") && ("H".equals(type.getValue()))) {
    	  Integer paid = AccountUtils.getPaidRemaining(login_uid);
    	  paid = paid + Integer.valueOf(old_used_time);
    	  if (paid < 8) {
	            msgList
	              .add("残有休が少ないため『 <span class='em'>有休</span> 』を申請できません。");
    	  }
      }
      /** 時間有休 */
      if (edit_mode.equals("") && !"0".equals(used_time.getValue()) && (
    		  "P".equals(type.getValue()) || "P1".equals(type.getValue()) || "P2".equals(type.getValue()))) {
    	  Integer paid = AccountUtils.getPaidRemaining(login_uid);
    	  paid = paid + Integer.valueOf(old_used_time);
    	  if (paid < Integer.valueOf(used_time.getValue().toString())) {
	            msgList
	              .add("残有休が少ないため『 <span class='em'>時間有休</span> 』を申請できません。");
    	  }
      }

      /** 更新・挿入時 */
      if (edit_mode.equals("")
        && ("P".equals(type.getValue()) || "P1".equals(type.getValue()) || "P2"
          .equals(type.getValue())) || "N".equals(type.getValue())) {
        /** 日付を punch_date に合わせる */
//        if (ajustDate(clock_in_time, punch_date)
//          && ajustDate(clock_out_time, punch_date)) {
    	  if (clock_out_time != null && !clock_out_time.isNullHour() && !clock_out_time.isNullMinute()) {
	          if (clock_in_time.getValue().getTime() > clock_out_time
	            .getValue()
	            .getTime()) {
	            msgList
	              .add("『 <span class='em'>退勤時刻</span> 』は『 <span class='em'>出勤時刻</span> 』以降の時刻を指定してください。");
	          }
	          if (!clock_in_time.getDay().equals(clock_out_time.getDay())
	              && ExtTimecardUtils
	              .getEipTExtTimecardSystemByUserId(login_uid)
	              .getChangeHour() < 9
        		  && clock_out_time.getHour().equals("9")
        		  && !clock_out_time.getMinute().equals("0")) {
		            msgList
		              .add("『 <span class='em'>退勤時刻</span> 』は9時までの時刻を指定してください。");
	          }
          }

        /** 勤怠時間は必須項目 */
        boolean out_flag = false;
        if (!(old_clock_out_time_hour.isEmpty())
          || !(old_clock_out_time_minute.isEmpty())) {
          out_flag = true;
        } else {
          out_flag = false;
        }

        if (getIsPast()){

          if (!clock_in_time.isNotNullValue()
            || (!clock_out_time.isNotNullValue() && out_flag)
            || current_clock_out_time_hour == -1
            && current_clock_out_time_minute != -1
            || current_clock_out_time_hour != -1
            && current_clock_out_time_minute == -1) {
            msgList.add("『 <span class='em'>"
              + clock_in_time.getFieldName()
              + "</span> 』を入力してください。");
          }
        }

        if (getIsPast() && ("P".equals(type.getValue()) || "P1".equals(type.getValue()) || "P2".equals(type.getValue()))) {

          // if ("T".equals(type.getValue()) &&
          // "0".equals(used_time.getValue())) {
          // msgList.add("『 <span class='em'>"
          // + used_time.getFieldName()
          // + "</span> 』を選択してください。");
          // }

          if ("1".equals(help1_flag.getValue()) && "1".equals(help2_flag.getValue())) {
        	  msgList.add("『"+help1_flag.getFieldName()+"』と『"
        			  + help2_flag.getFieldName() + "』は同時に申請できません。");
          }

          if ("1".equals(addholiday_flag.getValue()) && "1".equals(subholiday_flag.getValue())) {
        	  msgList.add("『"+addholiday_flag.getFieldName()+"』と『"
        			  + subholiday_flag.getFieldName() + "』は同時に申請できません。");
          }

          /** 外出復帰時間が適切に入力されているかチェック */
          Field field_out, field_come;
          for (int i = 1; i <= Integer.parseInt(rest_num.getValue()); i++) {
            field_out = this.getClass().getDeclaredField("outgoing_time" + i);
            field_come = this.getClass().getDeclaredField("comeback_time" + i);
            ALDateTimeField outgoing = (ALDateTimeField) field_out.get(this);
            ALDateTimeField comeback = (ALDateTimeField) field_come.get(this);
            if (!outgoing.isNotNullValue() || !comeback.isNotNullValue()) {
              msgList.add("『 <span class='em'>"
                + outgoing_comeback.getFieldName()
                + "</span> 』を入力してください。"
                + "（"
                + i
                + "行目）");
            }
          }

          /** 外出／復帰時間をリストに代入 */
          List<Map<String, Long>> list_from_to =
            new ArrayList<Map<String, Long>>();
          for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
            field_out = this.getClass().getDeclaredField("outgoing_time" + i);
            field_come = this.getClass().getDeclaredField("comeback_time" + i);
            ALDateTimeField outgoing = (ALDateTimeField) field_out.get(this);
            ALDateTimeField comeback = (ALDateTimeField) field_come.get(this);
            if (ajustDate(outgoing, punch_date)
              && ajustDate(comeback, punch_date)) {
              long from = outgoing.getValue().getTime();
              long to = comeback.getValue().getTime();
              if (from <= to) {
                if (clock_in_time.isNotNullValue()
                  && from < clock_in_time.getValue().getTime()) {
                  msgList
                    .add("『 <span class='em'>外出時刻</span> 』は『 <span class='em'>出勤時刻</span> 』以降の時刻を指定してください。（"
                      + i
                      + "行目）");
                }
                if (clock_out_time.isNotNullValue()
                  && to > clock_out_time.getValue().getTime()) {
                  msgList
                    .add("『 <span class='em'>復帰時刻</span> 』は『 <span class='em'>退勤時刻</span> 』以前の時刻を指定してください。（"
                      + i
                      + "行目）");
                }
                HashMap<String, Long> from_to = new HashMap<String, Long>();
                from_to.put("from", outgoing.getValue().getTime());
                from_to.put("to", comeback.getValue().getTime());
                list_from_to.add(from_to);
              } else {
                msgList
                  .add("『 <span class='em'>復帰時刻</span> 』は『 <span class='em'>外出時刻</span> 』以降の時刻を指定してください。（"
                    + i
                    + "行目）");
              }
            } else if (ajustDate(outgoing, punch_date)
              && !ajustDate(comeback, punch_date)
              && i == 1) {
              HashMap<String, Long> from_to = new HashMap<String, Long>();
              from_to.put("from", outgoing.getValue().getTime());
              // from_to.put("to", comeback.getValue().getTime());
              list_from_to.add(from_to);
              return (msgList.size() == 0);
            }
          }

          /** 外出時間の重複をチェックする */
          int i = 1;
          if (list_from_to.size() > 0) {
            List<Map<String, Long>> empty_from_to =
              new ArrayList<Map<String, Long>>();
            long min_from = list_from_to.get(0).get("from");
            long max_to = list_from_to.get(0).get("to");
            list_from_to.remove(0);
            for (Map<String, Long> map : list_from_to) {
              long new_from = map.get("from");
              long new_to = map.get("to");
              if (new_to <= min_from) {
                Map<String, Long> empty = new HashMap<String, Long>();
                empty.put("from", new_to);
                empty.put("to", min_from);
                empty_from_to.add(empty);
                min_from = new_from;
              } else if (new_from >= max_to) {
                Map<String, Long> empty = new HashMap<String, Long>();
                empty.put("from", max_to);
                empty.put("to", new_from);
                empty_from_to.add(empty);
                max_to = new_to;
              } else {
                /** empty_from_toのリストから入れる場所を探す */
                boolean duplicate_flag = true;
                for (Map<String, Long> empty_map : empty_from_to) {
                  if (empty_map.get("from") <= new_from
                    && empty_map.get("to") >= new_to) {
                    /** 区間を分割し、もとあった空白を削除 */
                    HashMap<String, Long> empty_left =
                      new HashMap<String, Long>();
                    empty_left.put("from", empty_map.get("from"));
                    empty_left.put("to", new_from);
                    empty_from_to.add(empty_left);

                    HashMap<String, Long> empty_right =
                      new HashMap<String, Long>();
                    empty_right.put("from", new_to);
                    empty_right.put("to", empty_map.get("to"));
                    empty_from_to.add(empty_right);

                    empty_from_to.remove(empty_map);

                    duplicate_flag = false;
                    break;
                  }
                }
                if (duplicate_flag) {
                  msgList.add("外出時間が重複しています。");
                  return false;
                }
              }
            }

            /** 並べ替える */
            Collections.sort(
              empty_from_to,
              new Comparator<Map<String, Long>>() {
                @Override
                public int compare(Map<String, Long> o1, Map<String, Long> o2) {
                  Map<String, Long> hash1 = o1;
                  Map<String, Long> hash2 = o2;
                  long from1 = hash1.get("from");
                  long from2 = hash2.get("from");
                  if (from1 == from2) {
                    long to1 = hash1.get("to");
                    long to2 = hash2.get("to");
                    return (int) (to1 - to2);
                  } else {
                    return (int) (from1 - from2);
                  }
                }

                @Override
                public boolean equals(Object obj) {
                  return super.equals(obj);
                }
              });

            long from = min_from;
            long to;
            for (Map<String, Long> empty : empty_from_to) {
              to = empty.get("from");
              field_out = this.getClass().getDeclaredField("outgoing_time" + i);
              field_come =
                this.getClass().getDeclaredField("comeback_time" + i);
              ALDateTimeField outgoing = (ALDateTimeField) field_out.get(this);
              ALDateTimeField comeback = (ALDateTimeField) field_come.get(this);
              outgoing.setValue(new Date(from));
              comeback.setValue(new Date(to));
              i++;
              from = empty.get("to");
            }
            to = max_to;
            field_out = this.getClass().getDeclaredField("outgoing_time" + i);
            field_come = this.getClass().getDeclaredField("comeback_time" + i);
            ALDateTimeField outgoing = (ALDateTimeField) field_out.get(this);
            ALDateTimeField comeback = (ALDateTimeField) field_come.get(this);
            outgoing.setValue(new Date(from));
            comeback.setValue(new Date(to));
            i++;
          }

          /** 余った場所には空のALDateTimeFieldを追加する */
          for (; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
            field_out = this.getClass().getDeclaredField("outgoing_time" + i);
            field_come = this.getClass().getDeclaredField("comeback_time" + i);
            field_out.set(this, new ALDateTimeField());
            field_come.set(this, new ALDateTimeField());
          }
        }
      }

      // 承認／却下
      if (edit_mode.equals("approval")) {
        if (!"1".equals(status.getValue()) && !"0".equals(status.getValue())) {
          msgList.add("『承認』または『却下』を選択してください。");
        }
        if ("0".equals(status.getValue()) && "".equals(reason.getValue())) {
          msgList.add("却下する場合は『却下理由』を入力してください。");
        }
      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return (msgList.size() == 0);
  }

  @SuppressWarnings("unused")
  private EipTExtTimecard getNearlyAboveRecord(List<EipTExtTimecard> list,
      int timecard_id) {
    EipTExtTimecard result = null;
    EipTExtTimecard record = null;
    int size = list.size();
    for (int i = 0; i < size; i++) {
      record = list.get(i);
      if (record.getExtTimecardId().intValue() >= timecard_id) {
        return result;
      } else {
        result = record;
      }
    }

    return null;
  }

  @SuppressWarnings("unused")
  private EipTExtTimecard getNearlyBelowRecord(List<EipTExtTimecard> list,
      int timecard_id) {
    EipTExtTimecard record = null;
    int size = list.size();
    for (int i = 0; i < size; i++) {
      record = list.get(i);
      if (record.getExtTimecardId().intValue() > timecard_id) {
        return record;
      }
    }

    return null;
  }

  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    setClockOutTime(rundata);
    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
      if (ALEipConstants.MODE_UPDATE.equals(this.getMode())) {
        try {
          if (!(this.entity_id > 0)) {
            entity_id =
              Integer.parseInt(ALEipUtils.getTemp(
                rundata,
                context,
                ALEipConstants.ENTITY_ID));
          }
          if ("".equals(this.type.getValue())) {
            String type = rundata.getParameters().get("type");
            this.type.setValue(type);
          }

          // this.waiting_time.setValue("5");

          String punch_date_year =
            rundata.getParameters().get("punch_date_year");
          String punch_date_month =
            rundata.getParameters().get("punch_date_month");
          String punch_date_day = rundata.getParameters().get("punch_date_day");
          StringBuffer buffer = new StringBuffer(8);
          buffer
            .append(punch_date_year)
            .append('/')
            .append(punch_date_month)
            .append('/')
            .append(punch_date_day);

          this.punch_date.setValue(buffer.toString());

        } catch (Exception e) {
          logger.error("Exception", e);
        }
      } else if (ALEipConstants.MODE_NEW_FORM.equals(this.getMode())) {
        String session_date = rundata.getParameters().get("date");
        if (session_date != null && !"".equals(session_date)) {
          this.punch_date.setValue(session_date);

          // 種類の初期値
          if (timecard_system.getNikyuFlag().equals("1")) {
        	  // 初期値が日給ならNをセット
        	  this.type.setValue("N");
          } else {
        	  this.type.setValue("P");
          }
        }
      } else if ("alt_insert".equals(this.alt_mode)) {

        StringBuffer buffer = new StringBuffer(8);
        String type = rundata.getParameters().get("type");
        String punch_date_year = rundata.getParameters().get("punch_date_year");
        String punch_date_month =
          rundata.getParameters().get("punch_date_month");
        String punch_date_day = rundata.getParameters().get("punch_date_day");

        buffer
          .append(punch_date_year)
          .append('/')
          .append(punch_date_month)
          .append('/')
          .append(punch_date_day);

        String reason = rundata.getParameters().get("reason");
        String remarks = rundata.getParameters().get("remarks");

        this.type.setValue(type);
        this.punch_date.setValue(buffer.toString());
        this.reason.setValue(reason);
        this.remarks.setValue(remarks);

      } else {
        ALDateTimeField current_date = new ALDateTimeField();
        current_date.setValue(new Date());

        // 種類＝出勤を入れる
        this.type.setValue("P");
        this.punch_date.setValue(current_date.toString());
      }

      // 日時をセッションに保存
      ALEipUtils.setTemp(rundata, context, "punch_date", punch_date.toString());

    }

    return res;
  }

  /**
   * タイムカードをデータベースから読み出します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      EipTExtTimecard timecard =
        ExtTimecardUtils.getEipTExtTimecard(rundata, context);
      if (timecard == null) {
        return false;
      }
      //
      timecard_id.setValue(timecard.getExtTimecardId().longValue());
      user_id.setValue(timecard.getUserId().intValue());
      punch_date.setValue(timecard.getPunchDate());
      type.setValue(timecard.getType());
      used_time.setValue(timecard.getUsedTime());

      watch_flag.setValue(timecard.getWatchFlag());
      support_flag.setValue(timecard.getSupportFlag());
      move_flag.setValue(timecard.getMoveFlag());
      car_flag.setValue(timecard.getCarFlag());
      help1_flag.setValue(timecard.getHelp1Flag());
      help2_flag.setValue(timecard.getHelp2Flag());
      addholiday_flag.setValue(timecard.getAddholidayFlag());
      subholiday_flag.setValue(timecard.getSubholidayFlag());

      clock_in_time.setValue(timecard.getClockInTime());
      clock_out_time.setValue(timecard.getClockOutTime());

      // 外出・復帰時間
      outgoing_time1.setValue(timecard.getOutgoingTime1());
      outgoing_time2.setValue(timecard.getOutgoingTime2());
      outgoing_time3.setValue(timecard.getOutgoingTime3());
      outgoing_time4.setValue(timecard.getOutgoingTime4());
      outgoing_time5.setValue(timecard.getOutgoingTime5());

      comeback_time1.setValue(timecard.getComebackTime1());
      comeback_time2.setValue(timecard.getComebackTime2());
      comeback_time3.setValue(timecard.getComebackTime3());
      comeback_time4.setValue(timecard.getComebackTime4());
      comeback_time5.setValue(timecard.getComebackTime5());

      reason.setValue(timecard.getReason());
      remarks.setValue(timecard.getRemarks());
      create_date.setValue(timecard.getCreateDate());
      update_date.setValue(timecard.getUpdateDate());

      // 当直代行時間
      waiting_time.setValue(timecard.getWaitingTime());

      // 休憩時間
      break_time.setValue(timecard.getBreakTime());

      // 時刻編集フラグ
      edit_flag.setValue(timecard.getEditFlag());

      int rest_num_tmp = 0;
      for (int i = 1; i <= 5; i++) {
        if (this.getOutgoingTime(i).isNotNullValue()) {
          rest_num_tmp++;
        } else if (this.getComebackTime(i).isNotNullValue()) {
          rest_num_tmp++;
        }
      }
      rest_num.setValue(Integer.toString(rest_num_tmp));
      status.setValue(timecard.getStatus());
      approval_user_id.setValue(timecard.getApprovalUserId());


      // 日時をセッションに保存
      ALEipUtils.setTemp(rundata, context, "punch_date", punch_date
        .getValue()
        .toString());

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * タイムカードをデータベースから削除します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      EipTExtTimecard timecard =
        ExtTimecardUtils.getEipTExtTimecard(rundata, context);
      if (timecard == null) {
        return false;
      }

      // タイムカードを削除
      Database.delete(timecard);
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        timecard.getExtTimecardId(),
        ALEventlogConstants.PORTLET_TYPE_EXTTIMECARD,
        timecard.getPunchDate().toString());
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * タイムカードをデータベースに格納します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      if (ExtTimecardUtils.hasDayExtTimecard(Integer.valueOf(ALEipUtils.getUserId(rundata)), punch_date.getValue())) {
    	  // 既にタイムカードが登録されている場合
    	  return true;
      }


      // 新規オブジェクトモデル
      EipTExtTimecard timecard = Database.create(EipTExtTimecard.class);

      // ユーザーID
      timecard.setUserId(Integer.valueOf(ALEipUtils.getUserId(rundata)));

      Calendar cal = Calendar.getInstance();
      if (timecard_system.getChangeHour() > cal.get(Calendar.HOUR_OF_DAY)) {
        cal.add(Calendar.DATE, -1);
      }

      // 日付
      timecard.setPunchDate(cal.getTime());

      // タイプ
      timecard.setType(type.getValue().substring(0, 1));
      if ("D".equals(type.getValue())) {
        timecard.setType("D");
      }

      // 初期化
      timecard.setSupportFlag("0");
      timecard.setMoveFlag("0");
      timecard.setCarFlag("0");
      timecard.setUsedTime("0");
      timecard.setWaitingTime("0");
      timecard.setHelp1Flag("0");
      timecard.setHelp2Flag("0");
      timecard.setWatchFlag(timecard_system.getWatchFlag());
      timecard.setApprovalUserId(0);
      timecard.setAddholidayFlag("0");
      timecard.setSubholidayFlag("0");

      // 休憩時間（初期化）
      if ("1".equals(timecard_system.getWatchFlag())) {
        timecard.setBreakTime("120");
      } else if ("2".equals(timecard_system.getWatchFlag())) {
        timecard.setBreakTime("180");
      } else {
        timecard.setBreakTime("60");
      }
      // 日付編集フラグ（初期値はOFF）
      timecard.setEditFlag("0");

      if (edit_mode.equals("punchin")) {
        // 出勤
        timecard.setClockInTime(cal.getTime());
        if ("1".equals(timecard_system.getNikyuFlag().toString())) {
        	timecard.setType("N");
        }
      } else if (edit_mode.equals("punchout")) {
        // 退勤
        timecard.setClockOutTime(cal.getTime());
      } else if (edit_mode.equals("outgoing")) {
        // 外出
        timecard.setOutgoingTime1(cal.getTime());
      } else if (edit_mode.equals("comeback")) {
        // 復帰
        timecard.setComebackTime1(cal.getTime());
      } else {
        // 修正時

        timecard.setPunchDate(punch_date.getValue());

        /** 未来時刻への打刻は不可 */

        if (cal.getTime().after(punch_date.getValue())) {

          // 出勤または日給以外の場合は、働いていないので出退勤時間を削除
          if (!"P".equals(type.getValue())
            && !"P1".equals(type.getValue())
            && !"P2".equals(type.getValue())) {

        	  if ("N".equals(type.getValue())) {
                // 日給の場合は、出退勤時刻のみ打刻する
                timecard.setClockInTime(clock_in_time.getValue());
                if (!clock_out_time.getDay().isEmpty()) {
                  // 退勤時間が入っていた時
                  timecard.setClockOutTime(clock_out_time.getValue());
                }
        	  } else {
  	            // 出退勤時間
  	            timecard.setClockInTime(null);
  	            timecard.setClockOutTime(null);
        	  }
              // 外出・復帰時間
              for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
                timecard.setOutgoingTime(null, i);
                timecard.setComebackTime(null, i);
              }
              // 全休の場合は有休を8時間使用したことにする
              if ("H".equals(type.getValue())) {
            	  timecard.setUsedTime("8");
              }

          } else {
            // 出退勤時間
            timecard.setClockInTime(clock_in_time.getValue());
            if (!clock_out_time.getDay().isEmpty()) {
              // 退勤時間が入っていた時
              timecard.setClockOutTime(clock_out_time.getValue());
            }

            // 外出・復帰時間
            Field field_out, field_come;
            for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
              field_out = this.getClass().getDeclaredField("outgoing_time" + i);
              field_come =
                this.getClass().getDeclaredField("comeback_time" + i);
              ALDateTimeField outgoing = (ALDateTimeField) field_out.get(this);
              ALDateTimeField comeback = (ALDateTimeField) field_come.get(this);
              if (!outgoing.isNullHour() && !outgoing.isNullMinute()) {
                timecard.setOutgoingTime(outgoing.getValue(), i);
              }
              if (!comeback.isNullHour() && !comeback.isNullMinute()) {
                timecard.setComebackTime(comeback.getValue(), i);
              }
            }
            // 休憩時間
            timecard.setBreakTime(break_time.getValue());
            // 当直代行時間
            timecard.setWaitingTime(waiting_time.getValue());
            // 当直
            if ("P".equals(type.getValue())) {
                timecard.setWatchFlag("0"); // なし
            } else if ("P1".equals(type.getValue())) {
              timecard.setWatchFlag("1"); // 当直
            } else if ("P2".equals(type.getValue())) {
              timecard.setWatchFlag("2"); // 日勤当直
            }

            // 一人応援
            if ("1".equals(support_flag.getValue())) {
              timecard.setSupportFlag(support_flag.getValue());
            }
            // 移動手当
            if ("1".equals(move_flag.getValue())) {
              timecard.setMoveFlag(move_flag.getValue());
            }
            // 車出手当
            if (!"0".equals(car_flag.getValue())) {
              timecard.setCarFlag(car_flag.getValue());
            }
            // 遠隔手当て(1)
            if ("1".equals(help1_flag.getValue())) {
              timecard.setHelp1Flag(help1_flag.getValue());
            }
            // 遠隔手当て(2)
            if ("1".equals(help2_flag.getValue())) {
              timecard.setHelp2Flag(help2_flag.getValue());
            }
            // 休日加算
            if ("1".equals(addholiday_flag.getValue())) {
              timecard.setAddholidayFlag(addholiday_flag.getValue());
            }
            // 休消手当
            if ("1".equals(subholiday_flag.getValue())) {
              timecard.setSubholidayFlag(subholiday_flag.getValue());
            }
            // 時間有休
            if ("P".equals(type.getValue().substring(0, 1))) {
              timecard.setUsedTime(used_time.getValue());
            }

            // 時刻の編集あり
            timecard.setEditFlag("1");
          }

          // 修正理由
          // timecard.setReason(reason.getValue());
        }

        // 備考
        timecard.setRemarks(remarks.getValue());
      }

      // 作成日
      timecard.setCreateDate(Calendar.getInstance().getTime());

      // 更新日
      timecard.setUpdateDate(Calendar.getInstance().getTime());

      // ステータス
      if ("D".equals(type.getValue())) {
        // 未選択の場合は、承認済みへ
        timecard.setStatus(ExtTimecardUtils.APPROVAL);
      } else {
        // 申請中
        timecard.setStatus(ExtTimecardUtils.PENDING);
      }

      // タイムカードを登録
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        timecard.getExtTimecardId(),
        ALEventlogConstants.PORTLET_TYPE_EXTTIMECARD,
        null);

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * データベースに格納されているタイムカードを更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      if (!edit_mode.equals("")) {
        ALEipUtils.setTemp(rundata, context, ALEipConstants.ENTITY_ID, String
          .valueOf(entity_id));
      }
      EipTExtTimecard timecard =
        ExtTimecardUtils.getEipTExtTimecard(rundata, context);
      if (timecard == null) {
        return false;
      }

      Calendar cal = Calendar.getInstance();
      if (edit_mode.equals("punchin")) {
        // 出勤
        timecard.setClockInTime(cal.getTime());
        timecard.setType("P");
      } else if (edit_mode.equals("punchout")) {
        // 退勤
        timecard.setClockOutTime(cal.getTime());
      } else if (edit_mode.equals("outgoing")) {
        // 外出
        timecard.setNewOutgoingTime(cal.getTime());
      } else if (edit_mode.equals("comeback")) {
        // 復帰
        timecard.setNewComebackTime(cal.getTime());
      } else if (edit_mode.equals("approval")) {
        timecard.setStatus(status.getValue());
        if ("0".equals(status.getValue())) {
          timecard.setReason(reason.getValue());
        } else {
          timecard.setReason("");
        }
        // 承認・却下者のIDを保存
        timecard.setApprovalUserId(login_uid);
      } else {
        // 修正時

        // タイプ
        timecard.setType(type.getValue().substring(0, 1));
        if ("D".equals(type.getValue())) {
          timecard.setType("D");
        }
        // 修正理由
        // timecard.setReason(reason.getValue());
        // 備考
        timecard.setRemarks(remarks.getValue());

        /** 未来時刻への打刻は不可 */
        // Calendar cal = Calendar.getInstance();
        if (cal.getTime().after(punch_date.getValue())) {

          // 出勤または日給以外の場合は、働いていないので出退勤時間を削除
          if (!"P".equals(type.getValue())
            && !"P1".equals(type.getValue())
            && !"P2".equals(type.getValue())) {

        	if ("N".equals(type.getValue())) {
        		// 日給の場合は出退勤時刻のみ打刻
                timecard.setClockInTime(clock_in_time.getValue());
                if (!clock_out_time.getHour().equals("")
                  && !clock_out_time.getMinute().equals("")) {
                  timecard.setClockOutTime(clock_out_time.getValue());
                } else {
                  // 初期化
                  timecard.setClockOutTime(null);
                }
                // 出退勤時間の手動変更があったかどうか
                if (   !old_clock_in_time_hour.equals(clock_in_time.getHour())
                    || !old_clock_in_time_minute.equals(clock_in_time.getMinute())
                    || !old_clock_out_time_hour.equals(clock_out_time.getHour())
                    || !old_clock_out_time_minute.equals(clock_out_time.getMinute())) {
                	timecard.setEditFlag("1");
                }
        	} else {
	            // 出退勤時間
	            timecard.setClockInTime(null);
	            timecard.setClockOutTime(null);
	            // 外出・復帰時間
	            for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
	              timecard.setOutgoingTime(null, i);
	              timecard.setComebackTime(null, i);
	            }
        	}
            // 初期化
        	timecard.setBreakTime("0");
            timecard.setWatchFlag("0");
            timecard.setSupportFlag("0");
            timecard.setMoveFlag("0");
            timecard.setCarFlag("0");
            timecard.setUsedTime("0");
            timecard.setWaitingTime("0");
            timecard.setHelp1Flag("0");
            timecard.setHelp2Flag("0");
            // 全休の場合は有休を8時間使用したことにする
            if ("H".equals(type.getValue())) {
          	  timecard.setUsedTime("8");
            }
          } else {
            // 出退勤時間
            timecard.setClockInTime(clock_in_time.getValue());
            if (!clock_out_time.getHour().equals("")
              && !clock_out_time.getMinute().equals("")) {
              timecard.setClockOutTime(clock_out_time.getValue());
            } else {
              // 初期化
              timecard.setClockOutTime(null);
            }

            // 出退勤時間の手動変更があったかどうか
            if (   !old_clock_in_time_hour.equals(clock_in_time.getHour())
                || !old_clock_in_time_minute.equals(clock_in_time.getMinute())
                || !old_clock_out_time_hour.equals(clock_out_time.getHour())
                || !old_clock_out_time_minute.equals(clock_out_time.getMinute())) {
            	timecard.setEditFlag("1");
            }

            // 外出・復帰時間
            Field field_out, field_come;
            for (int i = 1; i <= EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY; i++) {
              field_out = this.getClass().getDeclaredField("outgoing_time" + i);
              field_come =
                this.getClass().getDeclaredField("comeback_time" + i);
              timecard.setOutgoingTime(null, i);
              timecard.setComebackTime(null, i);

              ALDateTimeField outgoing = (ALDateTimeField) field_out.get(this);
              ALDateTimeField comeback = (ALDateTimeField) field_come.get(this);
              if (!outgoing.isNullHour() && !outgoing.isNullMinute()) {
                timecard.setOutgoingTime(outgoing.getValue(), i);
              }
              if (!comeback.isNullHour() && !comeback.isNullMinute()) {
                timecard.setComebackTime(comeback.getValue(), i);
              }
            }
            // 休憩時間
            timecard.setBreakTime(break_time.getValue());
            // 当直代行時間
            timecard.setWaitingTime(waiting_time.getValue());
            // 当直
            if ("P1".equals(type.getValue())) {
              timecard.setWatchFlag("1"); // 当直
            } else if ("P2".equals(type.getValue())) {
              timecard.setWatchFlag("2"); // 日勤当直
            } else {
              timecard.setWatchFlag("0"); // 日勤
            }
            // 一人応援
            if ("1".equals(support_flag.getValue())) {
              timecard.setSupportFlag(support_flag.getValue());
            } else {
              timecard.setSupportFlag("0");
            }
            // 移動手当
            if ("1".equals(move_flag.getValue())) {
              timecard.setMoveFlag(move_flag.getValue());
            } else {
              timecard.setMoveFlag("0");
            }
            // 車出手当
            if (!"0".equals(car_flag.getValue())) {
              timecard.setCarFlag(car_flag.getValue());
            } else {
              timecard.setCarFlag("0");
            }
            // 遠隔手当(1)
            if ("1".equals(help1_flag.getValue())) {
              timecard.setHelp1Flag(help1_flag.getValue());
            } else {
              timecard.setHelp1Flag("0");
            }
            // 遠隔手当(2)
            if ("1".equals(help2_flag.getValue())) {
              timecard.setHelp2Flag(help2_flag.getValue());
            } else {
              timecard.setHelp2Flag("0");
            }
            // 休日加算
            if ("1".equals(addholiday_flag.getValue())) {
              timecard.setAddholidayFlag(addholiday_flag.getValue());
            } else {
              timecard.setAddholidayFlag("0");
            }
            // 休消手当
            if ("1".equals(subholiday_flag.getValue())) {
              timecard.setSubholidayFlag(subholiday_flag.getValue());
            } else {
              timecard.setSubholidayFlag("0");
            }
            // 時間有休
            if ("P".equals(type.getValue().substring(0, 1))) {
              timecard.setUsedTime(used_time.getValue());
            } else {
              timecard.setUsedTime("0");
            }

          }

        }
      }

      // 更新日
      timecard.setUpdateDate(Calendar.getInstance().getTime());

      // morimoto-junichi add 2012/10/09
      if (!edit_mode.equals("approval")) {
        // ステータス
        if ("D".equals(type.getValue())) {
          // 未選択の場合は、承認済みへ
          timecard.setStatus(ExtTimecardUtils.APPROVAL);
        } else {
          // 申請中
          timecard.setStatus(ExtTimecardUtils.PENDING);
        }
      }

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        timecard.getExtTimecardId(),
        ALEventlogConstants.PORTLET_TYPE_EXTTIMECARD,
        null);

      // 却下された場合、対象者にお知らせ通知する
      if (edit_mode.equals("approval") && "0".equals(status.getValue())) {

        ALEipUser user = ALEipUtils.getALEipUser(rundata);
        List<String> recipients = new ArrayList<String>();
        recipients.add(ALEipUtils
          .getTurbineUser(timecard.getUserId())
          .getLoginName()
          .toString());

        SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy年MM月dd日（EE）");

        ALActivityService.create(new ALActivityPutRequest().withAppId(
          "ExtTimecard").withUserId(login_uid).withLoginName(
          user.getName().getValue())
          .withPortletParams("?template=ExtTimecardFormScreen&entityid="+entity_id)
          .withRecipients(recipients)
          .withTitle(sdf1.format(timecard.getPunchDate()) + "のタイムカードが却下されました。")
          .withPriority(1f)
          .withExternalId(String.valueOf(entity_id)));
      }

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }

    return true;
  }

  /**
   * 各ボタンを押したときの動作 <BR>
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  public boolean doPunch(ALAction action, RunData rundata, Context context,
      String mode) {
    try {
      edit_mode = mode;
      EipTExtTimecard timecard =
	        ExtTimecardUtils.getUpdateEipTExtTimecard(rundata, context, edit_mode);
      this.type.setValue("P");

      if (timecard != null) {
        // 更新すべきidを記憶
        entity_id = timecard.getExtTimecardId();

        // 当日内の更新なのでupdate文を発行
        super.doUpdate(action, rundata, context);
      } else {
        // 日をまたいでの更新なのでinsert文を発行
        super.doInsert(action, rundata, context);
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * 承認・却下を押したときの動作 <BR>
   *
   * @param action
   * @param rundata
   * @param context
   * @return TRUE 成功 FALSE 失敗
   */
  public boolean doApproval(ALAction action, RunData rundata, Context context,
      String mode) {
    try {
      edit_mode = mode;
//      EipTExtTimecard timecard =
//        ExtTimecardUtils.getUpdateEipTExtTimecard(rundata, context);
      EipTExtTimecard timecard = ExtTimecardUtils.getEipTExtTimecard(rundata, context);
      this.type.setValue(timecard.getType());
      this.user_id.setValue(timecard.getUserId());

      return super.doUpdate(action, rundata, context);
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
  }

  /**
   * アクセス権限をチェックします。
   *
   * @return
   */
  @Override
  protected boolean doCheckAclPermission(RunData rundata, Context context,
      int defineAclType) throws ALPermissionException {

    // 当日中のタイムカード打刻は、更新ではなく追加扱いにする
    if (defineAclType == ALAccessControlConstants.VALUE_ACL_UPDATE) {
      if ("punchin".equals(edit_mode)
        || "punchout".equals(edit_mode)
        || "outgoing".equals(edit_mode)
        || "comeback".equals(edit_mode)) {
        defineAclType = ALAccessControlConstants.VALUE_ACL_INSERT;
      }
    }
    return super.doCheckAclPermission(rundata, context, defineAclType);
  }

  /**
   * 日付
   *
   * @return
   */
  public ALDateTimeField getPunchDate() {
    return punch_date;
  }

  /**
   * 日付を取得します。
   *
   * @return
   */
  public String getDateStr() {
    try {
      SimpleDateFormat sdf = new SimpleDateFormat("yyyy年M月d日（EE）");
      return sdf.format(punch_date.getValue());
    } catch (Exception e) {
      return "";
    }
  }

  /**
   * 日付が過去かどうか
   *
   * @return
   */
  public boolean getIsPast() {
    Date now = Calendar.getInstance().getTime();
    Date date = punch_date.getValue();
    return date.before(now);
  }

  /**
   * 日付が現在かどうか
   *
   * @return
   */
  public boolean getIsToday() {
    int change_hour =
      ExtTimecardUtils
        .getEipTExtTimecardSystemByUserId(login_uid)
        .getChangeHour();
    Calendar cal = Calendar.getInstance();
    Date date = punch_date.getValue();
    boolean is_today = false;
    if ((Calendar.getInstance().get(Calendar.HOUR_OF_DAY)) < change_hour) {
      Calendar tmp_cal = Calendar.getInstance();
      tmp_cal.set(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH), cal
        .get(Calendar.DATE));
      is_today = ExtTimecardUtils.sameDay(date, tmp_cal.getTime());
    } else {
      is_today = ExtTimecardUtils.sameDay(date, cal.getTime());
    }
    return is_today;
  }

  /**
   * 種類
   *
   * @return
   */
  public ALStringField getType() {
    return type;
  }

  /**
   * 出勤時間
   *
   * @return
   */
  public ALDateTimeField getClockInTime() {
    return clock_in_time;
  }

  /**
   * 退勤時間
   *
   * @return
   */
  public ALDateTimeField getClockOutTime() {
    return clock_out_time;
  }

  /**
   * 外出時間
   *
   * @return
   */
  public ALDateTimeField getOutgoingTime(int n) {
    if (n > EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY) {
      return null;
    }
    switch (n) {
      case 1:
        return outgoing_time1;
      case 2:
        return outgoing_time2;
      case 3:
        return outgoing_time3;
      case 4:
        return outgoing_time4;
      case 5:
        return outgoing_time5;
    }
    return null;
  }

  /**
   * 復帰時間
   *
   * @return
   */
  public ALDateTimeField getComebackTime(int n) {
    if (n > EipTExtTimecard.OUTGOING_COMEBACK_PER_DAY) {
      return null;
    }
    switch (n) {
      case 1:
        return comeback_time1;
      case 2:
        return comeback_time2;
      case 3:
        return comeback_time3;
      case 4:
        return comeback_time4;
      case 5:
        return comeback_time5;
    }
    return null;
  }

  public ALDateTimeField getOutgoingTime2() {
    return outgoing_time2;
  }

  /**
   * 修正理由
   *
   * @return
   */
  public ALStringField getReason() {
    return reason;
  }

  /**
   * 備考
   *
   * @return
   */
  public ALStringField getRemarks() {
    return remarks;
  }

  /**
   * 備考表示用
   */
  public String getRemarksHtml() {
    return ALEipUtils.getMessageList(remarks.toString());
  }

  /**
   * 休憩時間
   *
   * @return
   */
  public ALStringField getBreakTime() {
    return break_time;
  }

  /**
   * 当直代行時間
   *
   * @return
   */
  public ALStringField getWaitingTime() {
    return waiting_time;
  }

  /**
   * 時間有休
   *
   * @return
   */
  public ALStringField getUsedTime() {
    return used_time;
  }

  /**
   * 当直フラグ
   *
   * @return
   */
  public ALStringField getWatchFlag() {
    return watch_flag;
  }

  /**
   * 一人応援フラグ
   *
   * @return
   */
  public ALStringField getSupportFlag() {
    return support_flag;
  }

  /**
   * 移動手当フラグ
   *
   * @return
   */
  public ALStringField getMoveFlag() {
    return move_flag;
  }

  /**
   * 車出手当フラグ
   *
   * @return
   */
  public ALStringField getCarFlag() {
    return car_flag;
  }

  /**
   * 遠隔手当(1)フラグ
   *
   * @return
   */
  public ALStringField getHelp1Flag() {
    return help1_flag;
  }

  /**
   * 遠隔手当(2)フラグ
   *
   * @return
   */
  public ALStringField getHelp2Flag() {
    return help2_flag;
  }

  /**
   * 休日加算
   *
   * @return
   */
  public ALStringField getAddholidayFlag() {
	return addholiday_flag;
  }

  /**
   * 休消手当
   *
   * @return
   */
  public ALStringField getSubholidayFlag() {
	return subholiday_flag;
  }

  /**
   * ステータス
   */
  public ALStringField getStatus() {
    return status;
  }

  /**
   * ID
   */
  public ALNumberField getTimecardId() {
    return timecard_id;
  }

  /**
   * 出退勤時刻編集フラグ
   *
   * @return
   */
  public ALStringField getEditFlag() {
    return edit_flag;
  }

  /**
   * タイムカード設定
   *
   * @return
   */
  public EipTExtTimecardSystem getTimecardSystem() {
    return timecard_system;
  }

  /**
   * 指定した2つの日付を比較する．
   *
   * @param date1
   * @param date2
   * @param checkTime
   *          時間まで比較する場合，true．
   * @return 等しい場合，0. date1>date2の場合, 1. date1 <date2の場合, 2.
   */
  @SuppressWarnings("unused")
  private int compareToDate(Date date1, Date date2) {
    Calendar cal1 = Calendar.getInstance();
    Calendar cal2 = Calendar.getInstance();
    cal1.setTime(date1);
    cal2.setTime(date2);

    int date1Year = cal1.get(Calendar.YEAR);
    int date1Month = cal1.get(Calendar.MONTH) + 1;
    int date1Day = cal1.get(Calendar.DATE);
    int date1Hour = cal1.get(Calendar.HOUR);
    int date1Minute = cal1.get(Calendar.MINUTE);
    int date1Second = cal1.get(Calendar.SECOND);
    int date2Year = cal2.get(Calendar.YEAR);
    int date2Month = cal2.get(Calendar.MONTH) + 1;
    int date2Day = cal2.get(Calendar.DATE);
    int date2Hour = cal2.get(Calendar.HOUR);
    int date2Minute = cal2.get(Calendar.MINUTE);
    int date2Second = cal2.get(Calendar.SECOND);

    if (date1Year == date2Year
      && date1Month == date2Month
      && date1Day == date2Day
      && date1Hour == date2Hour
      && date1Minute == date2Minute
      && date1Second == date2Second) {
      return 0;
    }
    if (cal1.after(cal2)) {
      return 2;
    } else {
      return 1;
    }
  }

  /**
   * 日付がずれていたら、強制的に直します。
   *
   * @param datetime
   * @param ajustto
   * @return
   */
  private boolean ajustDate(ALDateTimeField datetime, ALDateTimeField ajustto) {
    if (datetime != null && !datetime.isNullHour() && !datetime.isNullMinute()) {
      Date punch = ajustto.getValue();
      Calendar cal = Calendar.getInstance();
      cal.setTime(punch);
      cal.set(Calendar.HOUR, Integer.parseInt(datetime.getHour()));
      cal.set(Calendar.MINUTE, Integer.parseInt(datetime.getMinute()));
      cal.set(Calendar.SECOND, 0);

      /** CHANGE_HOUR以下だったら、次の日とみなす */
      if (Integer.parseInt(datetime.getHour()) < timecard_system
        .getChangeHour()) {
        cal.add(Calendar.DAY_OF_MONTH, 1);
      }
      datetime.setValue(cal.getTime());
      return true;
    }
    return false;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   *
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return ALAccessControlConstants.POERTLET_FEATURE_TIMECARD_TIMECARD_SELF;
  }

  /**
   * alt_modeをセットする
   */
  public void setAltMode(String alt_mode) {
    this.alt_mode = alt_mode;
  }

  /**
   * alt_modeをゲットする
   */
  public String getAltMode() {
    return this.alt_mode;
  }

  /**
   * 外出数をゲットする
   */
  public int getRestNum() {
    return Integer.parseInt(rest_num.getValue());
    // if (this.rest_num == 0) {
    // return 1;
    // } else {
    // return this.rest_num;
    // }

  }

  /**
   * 編集のとき、フォームに入力されている退勤時間を取得します。
   * */
  public void setClockOutTime(RunData rundata) {
    try {
      Field[] fields = this.getClass().getDeclaredFields();
      int length = fields.length;
      for (int i = 0; i < length; i++) {
        fields[i].setAccessible(true);
        String name = fields[i].getName();
        Object obj = fields[i].get(this);
        if (name.equals("clock_out_time")) {
          // フィールドが ALDateTimeField の場合
          if (obj instanceof ALDateTimeField) {
            String hourString =
              new StringBuffer().append(name).append(
                ALEipConstants.POST_DATE_HOUR).toString();
            String minitusString =
              new StringBuffer().append(name).append(
                ALEipConstants.POST_DATE_MINUTE).toString();
            current_clock_out_time_hour =
              rundata.getParameters().getInt(hourString);
            current_clock_out_time_minute =
              rundata.getParameters().getInt(minitusString);
          }
        }
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }
  }

  /**
   * 現時点の残有休を返す
   * @return
   */
  public String getPaidRemaining() {
	  Integer paid = AccountUtils.getPaidRemaining(login_uid);
	  return AccountUtils.getPaidFormat(paid);
  }

  /**
   * 締め処理が行われているか
   */
  public boolean getExtTimecardAggregate() {

	  SimpleDateFormat sf = new SimpleDateFormat("yyyy-MM");
	  String target_ym = sf.format(punch_date.getValue());

	  int id = 0;
	  if (user_id.getValueAsString() == null || "".equals(user_id.getValueAsString())) {
		  /** データがない場合はログインユーザーID */
		  id = login_uid;
	  } else {
		  id = Integer.valueOf(user_id.toString());
	  }

	  if (ExtTimecardUtils.getExtTimecardAggregate(target_ym)
			  || ExtTimecardUtils.getExtTimecardAggregateUser(id, target_ym)) {
		  return true;
	  }
	  return false;
  }


}
