/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.account.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.om.security.Group;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.security.JetspeedSecurityException;
import org.apache.jetspeed.services.security.UnknownUserException;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.utils.ALStringUtil;
import com.aimluck.eip.cayenne.om.account.EipMCompany;
import com.aimluck.eip.cayenne.om.account.EipMPosition;
import com.aimluck.eip.cayenne.om.account.EipMPost;
import com.aimluck.eip.cayenne.om.portlet.EipTExtTimecard;
import com.aimluck.eip.cayenne.om.portlet.EipTPostEmployment;
import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowRequestMap;
import com.aimluck.eip.cayenne.om.portlet.OriTPaid;
import com.aimluck.eip.cayenne.om.portlet.OriTPayrollSystem;
import com.aimluck.eip.cayenne.om.security.TurbineGroup;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.cayenne.om.security.TurbineUserGroupRole;
import com.aimluck.eip.common.ALBaseUser;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.Operations;
import com.aimluck.eip.orm.query.SQLTemplate;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.config.ALConfigHandler.Property;
import com.aimluck.eip.services.config.ALConfigService;
import com.aimluck.eip.user.beans.UserGroupLiteBean;
import com.aimluck.eip.util.ALEipUtils;

/**
 * ユーザーアカウントのユーティリティクラスです
 */
public class AccountUtils {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(AccountUtils.class.getName());

  /** CSVファイル名 */
  public static final String FOLDER_TMP_FOR_USERINFO_CSV_FILENAME =
    "user_info.csv";

  public static final int CSV_FILE_COL_COUNT = 11;

  /** ユーザー名として使用可能な記号群 */
  public static final String[] USER_NAME_SYMBOLS = { ".", "-", "_" };

  public static final String ACCOUNT_PORTLET_NAME = "Account";

  public static final String COMPANY_PORTLET_NAME = "Company";

  public static final String ACCOUNT_PERSON_PORTLET_NAME = "AccountPerson";

  public static final String ACCOUNT_LOGIN_PORTLET_NAME = "AccountLogin";

  /**
   * セッション中のエンティティIDで示されるユーザ情報を取得する。 論理削除されたユーザを取得した場合はnullを返す。
   *
   * @param rundata
   * @param context
   * @return
   */
  public static ALBaseUser getBaseUser(RunData rundata, Context context) {
    String userid =
      ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
    try {
      if (userid == null) {
        logger.debug("Empty ID...");
        return null;
      }
      ALBaseUser user = (ALBaseUser) JetspeedSecurity.getUser(userid);
      // 削除済みユーザの取得は行わない。
      // By Haruo Kaneko
      if ("T".equals(user.getDisabled())) {
        return null;
      } else {
        return (ALBaseUser) JetspeedSecurity.getUser(userid);
      }
    } catch (UnknownUserException uex) {
      logger.error("UnknownUserException : UserID = " + userid);
      return null;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   *
   * @param rundata
   * @param context
   * @return
   */
  public static EipMCompany getEipMCompany(RunData rundata, Context context) {
    return getEipMCompany(ALEipUtils.getTemp(
      rundata,
      context,
      ALEipConstants.ENTITY_ID));
  }

  public static EipMCompany getEipMCompany(String id) {
    EipMCompany result = null;
    try {
      if (id == null || Integer.valueOf(id) == null) {
        logger.debug("Empty ID...");
        return result;
      }

      Expression exp =
        ExpressionFactory.matchDbExp(EipMCompany.COMPANY_ID_PK_COLUMN, Integer
          .valueOf(id));
      List<EipMCompany> list =
        Database.query(EipMCompany.class, exp).fetchList();
      if (list == null || list.size() == 0) {
        logger.debug("Not found ID...");
        return result;
      }
      result = list.get(0);
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }
    return result;
  }

  /**
   * セッションに格納されているIDを用いて、部署情報を取得します。
   *
   * @param rundata
   * @param context
   * @return
   */
  public static EipMPost getEipMPost(RunData rundata, Context context) {
    return getEipMPost(ALEipUtils.getTemp(
      rundata,
      context,
      ALEipConstants.ENTITY_ID));
  }

  public static EipMPost getEipMPost(String id) {
    EipMPost result = null;
    try {
      if (id == null || Integer.valueOf(id) == null) {
        logger.debug("Empty ID...");
        return result;
      }

      Expression exp =
        ExpressionFactory.matchDbExp(EipMPost.POST_ID_PK_COLUMN, Integer
          .valueOf(id));
      List<EipMPost> list = Database.query(EipMPost.class, exp).fetchList();
      if (list == null || list.size() == 0) {
        logger.debug("Not found ID...");
        return result;
      }
      result = list.get(0);
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }
    return result;
  }

  /**
   *
   * @param rundata
   * @param context
   * @return
   */
  public static EipMPosition getEipMPosition(RunData rundata, Context context) {
    EipMPosition result = null;
    String id = ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);
    try {
      if (id == null || Integer.valueOf(id) == null) {
        logger.debug("Empty ID...");
        return result;
      }

      Expression exp =
        ExpressionFactory.matchDbExp(
          EipMPosition.POSITION_ID_PK_COLUMN,
          Integer.valueOf(id));
      List<EipMPosition> list =
        Database.query(EipMPosition.class, exp).fetchList();
      if (list == null || list.size() == 0) {
        logger.debug("Not found ID...");
        return result;
      }
      result = list.get(0);
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }
    return result;
  }

  public static String[] getCsvSplitStrings(String line) {
    if (line == null || line.equals("")) {
      return null;
    }

    try {
      List<String> list = new ArrayList<String>();
      int count_comma = 0;
      char c;
      StringBuffer token = new StringBuffer("");
      int len = line.length();
      for (int i = 0; i < len; i++) {
        c = line.charAt(i);
        if (c != ',' && i == len - 1) {
          token.append(c);
          list.add(token.toString());
        } else if (c == ',') {
          list.add(token.toString());
          token = new StringBuffer("");
          count_comma++;
          continue;
        } else {
          token.append(c);
        }
        if (count_comma > AccountUtils.CSV_FILE_COL_COUNT) {
          break;
        }
      }

      if (line.endsWith(",")) {
        list.add("");
      }

      String[] strings = new String[list.size()];
      strings = list.toArray(strings);
      return strings;
    } catch (Exception e) {
      return null;
    }
  }

  /**
   * ユーザーの所属する部署の一覧を取得します。
   *
   * @param uid
   *          ユーザーID
   * @return 所属する部署リスト
   */
  public static List<UserGroupLiteBean> getPostBeanList(int uid) {
    SelectQuery<TurbineUserGroupRole> query =
      Database.query(TurbineUserGroupRole.class);
    Expression exp1 =
      ExpressionFactory.matchExp(
        TurbineUserGroupRole.TURBINE_USER_PROPERTY,
        Integer.valueOf(uid));
    Expression exp2 =
      ExpressionFactory.greaterExp(
        TurbineUserGroupRole.TURBINE_GROUP_PROPERTY,
        Integer.valueOf(3));
    Expression exp3 =
      ExpressionFactory.matchExp(TurbineUserGroupRole.TURBINE_GROUP_PROPERTY
        + "."
        + TurbineGroup.OWNER_ID_PROPERTY, Integer.valueOf(1));
    query.setQualifier(exp1);
    query.andQualifier(exp2);
    query.andQualifier(exp3);
    List<TurbineUserGroupRole> list = query.fetchList();

    if (list == null || list.size() < 0) {
      return null;
    }

    List<UserGroupLiteBean> resultList = new ArrayList<UserGroupLiteBean>();

    TurbineGroup group = null;
    UserGroupLiteBean bean = null;
    for (TurbineUserGroupRole ugr : list) {
      group = ugr.getTurbineGroup();
      bean = new UserGroupLiteBean();
      bean.initField();
      bean.setGroupId(group.getName());
      bean.setName(group.getGroupAliasName());
      resultList.add(bean);
    }

    return resultList;
  }

  /**
   * 指定した ID のユーザが削除済みかどうかを調べる。
   *
   * @param userId
   * @return
   */
  public static boolean getUserIsDisabledOrDeleted(String userId) {
    if (userId == null || userId.equals("")) {
      return true;
    }
    String disabled;
    try {
      SelectQuery<TurbineUser> query = Database.query(TurbineUser.class);
      Expression exp =
        ExpressionFactory.matchDbExp(TurbineUser.USER_ID_PK_COLUMN, Integer
          .valueOf(userId));
      query.setQualifier(exp);
      List<TurbineUser> destUserList = query.fetchList();
      if (destUserList == null || destUserList.size() <= 0) {
        return true;
      }
      disabled = (destUserList.get(0)).getDisabled();
      return ("T".equals(disabled) || "N".equals(disabled));
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return true;
    }
  }

  /**
   * 指定されたユーザーが削除／無効化されたとき、申請が来ているワークフローを全て承認します。
   *
   * @param uid
   */
  public static boolean acceptWorkflow(int uid) {
    try {
      String userId = Integer.toString(uid);

      // 申請が来ているワークフローを取得する
      SelectQuery<EipTWorkflowRequestMap> workflow_request_map_query =
        Database.query(EipTWorkflowRequestMap.class);
      Expression workflow_exp =
        ExpressionFactory.matchExp(
          EipTWorkflowRequestMap.USER_ID_PROPERTY,
          userId);
      Expression workflow_exp2 =
        ExpressionFactory.matchExp(EipTWorkflowRequestMap.STATUS_PROPERTY, "C");
      workflow_request_map_query.setQualifier(workflow_exp
        .andExp(workflow_exp2));
      List<EipTWorkflowRequestMap> workflow_request_map_list =
        workflow_request_map_query.fetchList();
      EipTWorkflowRequestMap workflow_request_map = null;
      int list_size = workflow_request_map_list.size();

      // 申請が来ているワークフローの数だけ繰り返す
      for (int j = 0; j < list_size; j++) {
        workflow_request_map = workflow_request_map_list.get(j);

        // ワークフローを最後の人まで見ていく
        int request_number = workflow_request_map.getOrderIndex();
        while (true) {
          // 次の人がいるかどうか
          SelectQuery<EipTWorkflowRequestMap> workflow_request_map_query2 =
            Database.query(EipTWorkflowRequestMap.class);
          Expression workflow_exp3 =
            ExpressionFactory.matchExp(
              EipTWorkflowRequestMap.EIP_TWORKFLOW_REQUEST_PROPERTY,
              workflow_request_map.getEipTWorkflowRequest());
          Expression workflow_exp4 =
            ExpressionFactory.matchExp(
              EipTWorkflowRequestMap.ORDER_INDEX_PROPERTY,
              Integer.valueOf(request_number + 1));
          workflow_request_map_query2.setQualifier(workflow_exp3
            .andExp(workflow_exp4));
          List<EipTWorkflowRequestMap> workflow_request_map_list2 =
            workflow_request_map_query2.fetchList();

          // 自身を自動承認状態にする
          workflow_request_map.setStatus("T");
          if (workflow_request_map_list2.size() == 1) {
            // 次の人が見つかった
            EipTWorkflowRequestMap workflow_request_map2 =
              workflow_request_map_list2.get(0);
            if (getUserIsDisabledOrDeleted(workflow_request_map2
              .getUserId()
              .toString())) {
              // 次の人が削除済み、もしくは無効化されていたら自動承認した上で次の人に回す
              workflow_request_map2.setStatus("T");
              request_number += 1;
            } else {
              // 次の人を確認状態にして終了
              workflow_request_map2.setStatus("C");
              break;
            }
          } else {
            // 次の人が見つからなければ、最後まで行ったことになるので終了
            workflow_request_map.getEipTWorkflowRequest().setProgress("A");
            break;
          }
        }
      }
      return true;
    } catch (Exception e) {
      logger.error("Exception", e);
      return false;
    }
  }

  /**
   * 管理者権限を持ったユーザを一人、管理者権限剥奪・無効化・削除しても<br/>
   * 最低限必要な管理者権限を持ったユーザ数を割らないかどうかを返します。
   *
   * @return
   */
  public static boolean isAdminDeletable() {
    return isAdminDeletable(1);
  }

  /**
   * 管理者権限を持ったユーザを指定人数、管理者権限剥奪・無効化・削除しても<br/>
   * 最低限必要な管理者権限を持ったユーザ数を割らないかどうかを返します。
   *
   * @param admin_count
   * @return
   */
  public static boolean isAdminDeletable(int admin_count) {
    try {
      int minimum_admin =
        Integer.valueOf(ALConfigService
          .get(Property.MINIMUM_ADMINISTRATOR_USER_COUNT));
      Group group = JetspeedSecurity.getGroup("LoginUser");
      Role adminrole = JetspeedSecurity.getRole("admin");
      int current_admin_count =
        Database.query(TurbineUserGroupRole.class).where(
          Operations.eq(TurbineUserGroupRole.TURBINE_ROLE_PROPERTY, adminrole
            .getId()),
          Operations.eq(TurbineUserGroupRole.TURBINE_GROUP_PROPERTY, group
            .getId()),
          Operations.ne(TurbineUserGroupRole.TURBINE_USER_PROPERTY, 1),
          Operations.eq(TurbineUserGroupRole.TURBINE_USER_PROPERTY
            + "."
            + TurbineUser.DISABLED_PROPERTY, "F")).distinct(true).getCount();
      int admin_count_will = current_admin_count - admin_count;
      if (admin_count_will < 0) {
        admin_count_will = 0;
      }
      return minimum_admin <= admin_count_will;
    } catch (JetspeedSecurityException e) {
      logger.error("Exception", e);
      return false;
    }
  }

  /**
   * 与えられたユーザー名に使われている記号が、使用できるものかを確認します。
   *
   * @return
   */
  public static boolean isValidSymbolUserName(String name) {
    List<String> symbols = Arrays.asList(USER_NAME_SYMBOLS);
    for (char c : name.toCharArray()) {
      if (ALStringUtil.isSymbol(c) && !symbols.contains(String.valueOf(c))) {
        return false;
      }
    }
    return true;
  }

  /**
   *
   * @param rundata
   * @param context
   * @return
   */
  public static OriTPayrollSystem getOriTPayrollSystem(int id) {
    OriTPayrollSystem result = null;
    try {

      Expression exp =
        ExpressionFactory.matchDbExp(OriTPayrollSystem.ID_PK_COLUMN, Integer
          .valueOf(id));
      List<OriTPayrollSystem> list =
        Database.query(OriTPayrollSystem.class, exp).fetchList();
      if (list == null || list.size() == 0) {
        logger.debug("Not found ID...");
        return result;
      }
      result = list.get(0);
    } catch (Exception ex) {
      logger.error("Exception", ex);
    }
    return result;
  }

  /***
   * 前月までの確定分の残有休（ユーザーID、年月）→ 残有休時間（54）
   *
   * @param id
   *          (ユーザーID)
   * @param yyyyMM
   *          (yyyy-MM 形式)
   * @return
   */
  public static Integer getPaidRemaining(int id, String yyyyMM) {
    try {
      int oriTpaidSum = 0;
      int timeCardSum = 0;
      List<OriTPaid> paidList = null;
      List<EipTExtTimecard> timeCardList = null;

      // 有給を取得して合計を求める
      SelectQuery<OriTPaid> oriTpaidqQuery = Database.query(OriTPaid.class);
      Expression exp =
        ExpressionFactory.matchDbExp(OriTPaid.TURBINE_USER_PROPERTY
          + "."
          + TurbineUser.USER_ID_PK_COLUMN, Integer.valueOf(id));
      Expression exp1 =
        ExpressionFactory.lessOrEqualExp(OriTPaid.GRANT_YM_PROPERTY, yyyyMM);

      oriTpaidqQuery.setQualifier(exp.andExp(exp1));

      paidList = oriTpaidqQuery.fetchList();

		  if (paidList == null || paidList.size() <= 0 ) {
			  return 0;
		  }

      // 有給
      for (int i = 0; i < paidList.size(); i++) {
        oriTpaidSum += paidList.get(i).getPaidTime();
      }

      // タイムカード情報を取得
      StringBuilder sb = new StringBuilder();
      sb.append("SELECT ");
      sb.append(" eip_t_ext_timecard.timecard_id, ");
      sb.append(" eip_t_ext_timecard.user_id, ");
      sb.append(" eip_t_ext_timecard.punch_date, ");
      sb.append(" eip_t_ext_timecard.used_time ");
      sb.append("FROM ");
      sb.append(" eip_t_ext_timecard ");
      sb.append("WHERE ");
      sb.append("eip_t_ext_timecard.user_id = '");
      sb.append(id);
      sb.append("'");
      sb.append(" and ");
      sb.append(" to_char(eip_t_ext_timecard.punch_date,'yyyy-MM') <= '");
      sb.append(yyyyMM);
      sb.append("' ");
      sb.append(" and ");
      sb.append("eip_t_ext_timecard.status = '1'"); // 承認済み

      SQLTemplate<EipTExtTimecard> eipTExtTimecardQuery =
        Database.sql(EipTExtTimecard.class, sb.toString());

      timeCardList = eipTExtTimecardQuery.fetchList();

      if (timeCardList != null) {
        // タイムカード
        for (int j = 0; j < timeCardList.size(); j++) {
          timeCardSum += Integer.valueOf(timeCardList.get(j).getUsedTime());
        }
      }

      return oriTpaidSum - timeCardSum;

	  } catch (Exception ex) {
		  logger.error("Exception [getPaidRemaining]", ex);
		  return 0;
	  }
  }

  /***
   * 現時点の残有休（ユーザーID） → 残有休時間（54）
   *
   * @param id
   *          (ユーザーID)
   * @param yyyyMM
   *          (yyyy-MM 形式)
   * @return
   */
  public static Integer getPaidRemaining(int id) {
    try {
      int oriTpaidSum = 0;
      int timeCardSum = 0;
      List<OriTPaid> paidList = null;
      List<EipTExtTimecard> timeCardList = null;

      // 有給を取得して合計を求める
      SelectQuery<OriTPaid> oriTpaidqQuery = Database.query(OriTPaid.class);
      Expression exp =
        ExpressionFactory.matchDbExp(OriTPaid.TURBINE_USER_PROPERTY
          + "."
          + TurbineUser.USER_ID_PK_COLUMN, Integer.valueOf(id));

      oriTpaidqQuery.setQualifier(exp);

      paidList = oriTpaidqQuery.fetchList();

		  if (paidList == null || paidList.size() <= 0 ) {
			  return 0;
		  }

      // 有給
      for (int i = 0; i < paidList.size(); i++) {
        oriTpaidSum += paidList.get(i).getPaidTime();
      }

      // タイムカード情報を取得
      StringBuilder sb = new StringBuilder();
      sb.append("SELECT ");
      sb.append(" eip_t_ext_timecard.timecard_id, ");
      sb.append(" eip_t_ext_timecard.user_id, ");
      sb.append(" eip_t_ext_timecard.punch_date, ");
      sb.append(" eip_t_ext_timecard.used_time ");
      sb.append("FROM ");
      sb.append(" eip_t_ext_timecard ");
      sb.append("WHERE ");
      sb.append("eip_t_ext_timecard.user_id = '");
      sb.append(id);
      sb.append("'");

      SQLTemplate<EipTExtTimecard> eipTExtTimecardQuery =
        Database.sql(EipTExtTimecard.class, sb.toString());

      timeCardList = eipTExtTimecardQuery.fetchList();

      if (timeCardList != null) {
        // タイムカード
        for (int j = 0; j < timeCardList.size(); j++) {
          timeCardSum += Integer.valueOf(timeCardList.get(j).getUsedTime());
        }
      }

		  return oriTpaidSum - timeCardSum;
	  } catch (Exception ex) {
		  logger.error("Exception [getPaidRemaining]", ex);
		  return 0;
	  }
  }

  /***
   * 各月の有給付与時間を取得
   *
   * @param id
   * @param yyyyMM
   * @return
   */
  public static Integer getOriTpaidWithMonth(int id, String yyyyMM) {
    try {
      List<OriTPaid> paidList = null;

      // 有給を取得して合計を求める
      SelectQuery<OriTPaid> oriTpaidqQuery = Database.query(OriTPaid.class);
      Expression exp =
        ExpressionFactory.matchDbExp(OriTPaid.TURBINE_USER_PROPERTY
          + "."
          + TurbineUser.USER_ID_PK_COLUMN, Integer.valueOf(id));
      Expression exp1 =
        ExpressionFactory.matchExp(OriTPaid.GRANT_YM_PROPERTY, yyyyMM);

      oriTpaidqQuery.setQualifier(exp.andExp(exp1));

      paidList = oriTpaidqQuery.fetchList();

      if (paidList == null || paidList.size() <= 0) {
        return null;
      }

      return paidList.get(0).getPaidTime();

    } catch (Exception ex) {
      logger.error("Exception [getOriTpaidWithMonth]", ex);
      return null;
    }
  }

  /***
   * 指定した年月に使用した有給時間を取得
   *
   * @param id
   * @param yyyyMM
   * @return
   */
  public static Integer getEipTExtTimeCardWasUsedTimeWithMonth(int id,
      String yyyyMM) {
    try {
      List<EipTExtTimecard> timeCardList = null;
      int timeCardSum = 0;

      // タイムカード情報を取得
      StringBuilder sb = new StringBuilder();
      sb.append("SELECT ");
      sb.append(" eip_t_ext_timecard.timecard_id, ");
      sb.append(" eip_t_ext_timecard.user_id, ");
      sb.append(" eip_t_ext_timecard.punch_date, ");
      sb.append(" eip_t_ext_timecard.used_time ");
      sb.append("FROM ");
      sb.append(" eip_t_ext_timecard ");
      sb.append("WHERE ");
      sb.append("eip_t_ext_timecard.user_id = '");
      sb.append(id);
      sb.append("'");
      sb.append(" and ");
      sb.append(" to_char(eip_t_ext_timecard.punch_date,'yyyy-MM') like '");
      sb.append(yyyyMM);
      sb.append("%' ");
      sb.append(" and ");
      sb.append("eip_t_ext_timecard.status = '1'"); // 承認済み

      SQLTemplate<EipTExtTimecard> eipTExtTimecardQuery =
        Database.sql(EipTExtTimecard.class, sb.toString());

      timeCardList = eipTExtTimecardQuery.fetchList();

      if (timeCardList == null || timeCardList.size() <= 0) {
        return 0;
      }

      // タイムカード
      for (int j = 0; j < timeCardList.size(); j++) {
        timeCardSum += Integer.valueOf(timeCardList.get(j).getUsedTime());
      }

      return timeCardSum;

    } catch (Exception ex) {
      logger.error("Exception [getEipTExtTimeCardWasUsedTimeWithMonth]", ex);
      return 0;
    }
  }

  /***
   * 有給時間のフォーマット
   *
   * @param paidValue
   * @return
   */
  public static String getPaidFormat(int paidValue) {

    if (paidValue == -1) {
      return String.valueOf(paidValue);
    }

    int day = paidValue / 8;
    int time = paidValue % 8;

	 String format = day + "日";
	 if (time > 0) {
		 format+= time + "時間";
	 }

    return format;
  }

  /***
   * 所定労働日数
   *
   * @param id
   * @param yyyyMM
   * @return
   */
  public static Integer getPrescribedWorkingDays(int id, String yyyyMM) {
    try {
      List<EipTPostEmployment> prePostEmploymentList = null;
      List<EipTPostEmployment> postEmploymentList = null;
      int holiday;
      int days;

      // ユーザーの所属部署IDを取得
      List<Integer> postIdList = ALEipUtils.getPostIdList(id);

      if (postIdList == null || postIdList.size() <= 0) {
        return 0;
      }

      // 年月を分解
      String[] ym = yyyyMM.split("-");
      int year = Integer.valueOf(ym[0]);
      int month = Integer.valueOf(ym[1]);
      int preYear = year - 1;
      int nexYear = year + 1;

      if (month != 12) {

        // 前年度の設定を取得し月文のみを取得
        SelectQuery<EipTPostEmployment> preQuery =
          Database.query(EipTPostEmployment.class);
        Expression preExp0 =
          ExpressionFactory.matchExp(EipTPostEmployment.YEAR_PROPERTY, preYear);
        Expression preExp1 =
          ExpressionFactory.inDbExp(EipTPostEmployment.EIP_MPOST_PROPERTY
            + "."
            + EipMPost.POST_ID_PK_COLUMN, postIdList);

        preQuery.setQualifier(preExp0.andExp(preExp1));
        prePostEmploymentList = preQuery.fetchList();

        if (prePostEmploymentList == null || prePostEmploymentList.size() <= 0) {
          return 0;
        }

        // 引数の年で設定を取得
        SelectQuery<EipTPostEmployment> query =
          Database.query(EipTPostEmployment.class);
        Expression exp0 =
          ExpressionFactory.matchExp(EipTPostEmployment.YEAR_PROPERTY, year);
        Expression exp1 =
          ExpressionFactory.inDbExp(EipTPostEmployment.EIP_MPOST_PROPERTY
            + "."
            + EipMPost.POST_ID_PK_COLUMN, postIdList);

        query.setQualifier(exp0.andExp(exp1));

        postEmploymentList = query.fetchList();

        if (postEmploymentList == null || postEmploymentList.size() <= 0) {
          return 0;
        }

        // 一年間の休日数合計を求める
        holiday = prePostEmploymentList.get(0).getHoliday12().intValue();

        for (int i = 1; i <= 11; i++) {
          holiday +=
            (Integer) postEmploymentList.get(0).readProperty("holiday" + i);
        }

        // 所定労働日数を計算
        days = (365 - holiday) / 12;

      } else {

        // 前年度の設定を取得し月文のみを取得
        SelectQuery<EipTPostEmployment> preQuery =
          Database.query(EipTPostEmployment.class);
        Expression preExp0 =
          ExpressionFactory.matchExp(EipTPostEmployment.YEAR_PROPERTY, year);
        Expression preExp1 =
          ExpressionFactory.inDbExp(EipTPostEmployment.EIP_MPOST_PROPERTY
            + "."
            + EipMPost.POST_ID_PK_COLUMN, postIdList);

        preQuery.setQualifier(preExp0.andExp(preExp1));
        prePostEmploymentList = preQuery.fetchList();

        if (prePostEmploymentList == null || prePostEmploymentList.size() <= 0) {
          return 0;
        }

        // 引数の年で設定を取得
        SelectQuery<EipTPostEmployment> query =
          Database.query(EipTPostEmployment.class);
        Expression exp0 =
          ExpressionFactory.matchExp(EipTPostEmployment.YEAR_PROPERTY, nexYear);
        Expression exp1 =
          ExpressionFactory.inDbExp(EipTPostEmployment.EIP_MPOST_PROPERTY
            + "."
            + EipMPost.POST_ID_PK_COLUMN, postIdList);

        query.setQualifier(exp0.andExp(exp1));

        postEmploymentList = query.fetchList();

        if (postEmploymentList == null || postEmploymentList.size() <= 0) {
          return 0;
        }

        // 一年間の休日数合計を求める
        holiday = prePostEmploymentList.get(0).getHoliday12().intValue();

        for (int i = 1; i <= 11; i++) {
          holiday +=
            (Integer) postEmploymentList.get(0).readProperty("holiday" + i);
        }

        // 所定労働日数を計算
        days = (365 - holiday) / 12;
      }

      return days;
    } catch (Exception ex) {
      logger.error("Exception [getScheduledWorkingHours]", ex);
      return 0;
    }
  }

  /***
   * 所定労働時間
   *
   * @param id
   * @param yyyyMM
   *          (yyyy-MM 形式)
   * @return
   */
  public static Float getScheduledWorkingHours(int id, String yyyyMM) {
    try {
      List<EipTPostEmployment> postEmploymentList = null;

      // ユーザーの所属部署IDを取得
      List<Integer> postIdList = ALEipUtils.getPostIdList(id);

      if (postIdList == null || postIdList.size() <= 0) {
        return 0f;
      }

      // 年月を分解
      String[] ym = yyyyMM.split("-");
      int year = Integer.valueOf(ym[0]);
      int month = Integer.valueOf(ym[1]);

      SelectQuery<EipTPostEmployment> query =
        Database.query(EipTPostEmployment.class);
      Expression exp0 =
        ExpressionFactory.matchExp(EipTPostEmployment.YEAR_PROPERTY, year);
      Expression exp1 =
        ExpressionFactory.inDbExp(EipTPostEmployment.EIP_MPOST_PROPERTY
          + "."
          + EipMPost.POST_ID_PK_COLUMN, postIdList);

      query.setQualifier(exp1.andExp(exp0));

      postEmploymentList = query.fetchList();

      if (postEmploymentList == null || postEmploymentList.size() <= 0) {
        return 0f;
      }

      return (Float) postEmploymentList.get(0).readProperty("worktime" + month);
    } catch (Exception ex) {
      logger.error("Exception [getScheduledWorkingHours]", ex);
      return null;
    }
  }

}
