/*
 * Copyright(C) 2012 orinos Co.,Ltd. All rights reserved.
 * http://www.orinos.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.account;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.eip.account.util.AccountUtils;
import com.aimluck.eip.cayenne.om.portlet.OriTPaid;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALBaseUser;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * ユーザーアカウントのフォームデータを管理するクラスです。 <BR>
 *
 */
public class AccountUserPaidFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
    .getLogger(AccountUserPaidFormData.class.getName());

  /** 日数 */
  private ALNumberField paid_days;

  /** 時間 */
  private ALNumberField paid_time;

  private String orgId;

  /** ログインしている人のユーザーID */
  private int login_uid;

  private String mode;


  private ALBaseUser user;

  /**
   * 初期化します。
   *
   * @param action
   * @param rundata
   * @param context
   *
   *
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);

    orgId = Database.getDomainName();

    login_uid = ALEipUtils.getUserId(rundata);

    user = AccountUtils.getBaseUser(rundata, context);
  }

  /**
   * 各フィールドを初期化します。 <BR>
   *
   *
   */
  @Override
  public void initField() {

    // 日数
    paid_days = new ALNumberField();
    paid_days.setFieldName("日数");

    // 時間
    paid_time = new ALNumberField();
    paid_time.setFieldName("時間");

  }

  /**
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);
    return res;
  }

  /**
   * 各フィールドに対する制約条件を設定します。 <BR>
   *
   *
   */
  @Override
  protected void setValidator() {

    // 日数
    paid_days.setNotNull(true);
    paid_days.limitMaxValue(999);

    // 時間
    paid_time.setNotNull(true);
    paid_time.limitMaxValue(999);

  }

  /**
   * フォームに入力されたデータの妥当性検証を行います。 <BR>
   *
   * @param msgList
   * @return
   *
   */
  @Override
  protected boolean validate(List<String> msgList) {
	// 日数
	paid_days.validate(msgList);
	// 時間
	paid_time.validate(msgList);

    return (msgList.size() == 0);
  }

  /**
   * 『ユーザー』を読み込みます。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // ユーザーがあるかどうかの
      if (user == null) {
        return false;
      }

      // オブジェクトを取得
      // 現在の日時情報
      Calendar cal = Calendar.getInstance();
//      // 検索する日付
//      String month = String.valueOf(user.getPaidGivingMonth());
//      if (month.length() == 1) {
//    	  month = "0" + month;
//      }

      String ym = cal.get(Calendar.YEAR) + "-" + (cal.get(Calendar.MONTH) + 1);
      SelectQuery<OriTPaid> oriTpaidQuery = Database.query(OriTPaid.class);
      Expression exp0 = ExpressionFactory.matchDbExp(OriTPaid.TURBINE_USER_PROPERTY
    		  + "."
    		  + TurbineUser.USER_ID_PK_COLUMN, user.getUserId());
      Expression exp1 = ExpressionFactory.matchExp(OriTPaid.GRANT_YM_PROPERTY, ym);

      oriTpaidQuery.setQualifier(exp0.andExp(exp1));

      // オブジェクトを代入
      if (oriTpaidQuery.getCount() == 0) {
    	  mode = "insert";
    	  return false;
      } else {
    	  mode = "update";
      }

      OriTPaid paid = oriTpaidQuery.fetchList().get(0);

      if (paid == null) {
    	  return false;
      }

      // フィールドに代入
      int def = 8;
      int paidTime = paid.getPaidTime();
      int days = paidTime / def;
      int time = paidTime % def;

      paid_days.setValue(days);
      paid_time.setValue(time);

      return true;
    } catch (Exception e) {
      logger.error("Exception", e);
      return false;
    }
  }

  /**
   * 『ユーザー』を追加します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {

    boolean res = true;
    try {

      if (res) { // オブジェクトモデルを生成

        // ユーザーを取得
        Expression exp =
          ExpressionFactory.matchDbExp(TurbineUser.USER_ID_PK_COLUMN, user
            .getUserId());
        SelectQuery<TurbineUser> query = Database.query(TurbineUser.class, exp);
        TurbineUser tuser = query.fetchList().get(0);
        if (tuser == null) {
          return false;
        }

        // 現在の日時
        Calendar cal = Calendar.getInstance();
        Date now = cal.getTime();

        // 有給テーブルオブジェクト取得
        OriTPaid paid = Database.create(OriTPaid.class);
        paid.setTurbineUser(tuser);
        // 年月を設定
//        String month = String.valueOf(user.getPaidGivingMonth());
//        if (month.length() == 1) {
//          month = "0" + month;
//        }

        String ym = cal.get(Calendar.YEAR) + "-" + (cal.get(Calendar.MONTH) + 1);
        paid.setGrantYm(ym);

        // 有給時間を設定 一日8時間計算
        int paidTime =
          ((int) paid_days.getValue() * 8) + (int) paid_time.getValue();
        paid.setPaidTime(paidTime);

        // 消失時間 どうなるか分からない コメントする
        // paid.setLapseTime(lapseTime);

        // 作成者
        paid.setCreateUserId(login_uid);
        // 作成時間または更新時間
        paid.setCreateDate(now);

        // コミット
        Database.commit();

      }
    } catch (Exception e) {
      Database.rollback();
      logger.error("Exception", e);
      res = false;
    }
    return res;
  }

  /**
   * 『ユーザー』を更新します。 <BR>
   *
   * @param rundata
   * @param context
   * @param msgList
   * @return
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    boolean res = true;
    try {

        // 現在の日時情報
        Calendar cal = Calendar.getInstance();
        Date now = cal.getTime();
        // 検索する日付
//        String month = String.valueOf(user.getPaidGivingMonth());
//        if (month.length() == 1) {
//      	  month = "0" + month;
//        }

        String ym = cal.get(Calendar.YEAR) + "-" + (cal.get(Calendar.MONTH) + 1);
        SelectQuery<OriTPaid> oriTpaidQuery = Database.query(OriTPaid.class);
        Expression exp0 = ExpressionFactory.matchDbExp(OriTPaid.TURBINE_USER_PROPERTY
      		  + "."
      		  + TurbineUser.USER_ID_PK_COLUMN, user.getUserId());
        Expression exp1 = ExpressionFactory.matchExp(OriTPaid.GRANT_YM_PROPERTY, ym);

        oriTpaidQuery.setQualifier(exp0.andExp(exp1));

        // オブジェクト判定
        if (oriTpaidQuery.getCount() == 0) {
      	  return false;
        }
        // オブジェクト代入
        OriTPaid paid = oriTpaidQuery.fetchList().get(0);

        // 有給時間計算
        int paidTime =
                ((int) paid_days.getValue() * 8) + (int) paid_time.getValue();
              paid.setPaidTime(paidTime);

        // 時間をセット
        paid.setPaidTime(paidTime);
        paid.setCreateUserId(login_uid);
        paid.setCreateDate(now);

        Database.commit();

    } catch (Exception e) {
      logger.error("Exception", e);
      res = false;
    }
    return res;
  }

  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    // TODO 自動生成されたメソッド・スタブ
    return false;
  }

  /**
   * @return paid_days
   */
  public ALNumberField getPaidDays() {
    return paid_days;
  }

  /**
   * @return paid_time
   */
  public ALNumberField getPaidTime() {
    return paid_time;
  }

/**
 * @return mode
 */
public String getMode() {
	return mode;
}

}
