/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.terasoluna.fw.message;

import java.io.InputStream;
import java.net.URL;
import java.text.MessageFormat;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.concurrent.CopyOnWriteArrayList;

import jp.terasoluna.fw.message.execption.MessageRuntimeException;

/**
 * bZ[WǗNX
 * 
 */
public class MessageManager {
    /**
     * bZ[WvpeBt@C̃x[Xl[Xg
     */
    protected final List<String> basenames = new CopyOnWriteArrayList<String>();
    /**
     * bZ[WID̃tH[}bg
     */
    protected String messageIdFormat = "[%s] ";
    /**
     * bZ[WIDȂꍇɗOX[邩ۂ
     */
    protected boolean throwIfResourceNotFound = false;
    /**
     * bZ[WtH[}b^
     */
    protected final MessageFormatter messageFormatter;
    /**
     * bZ[WID̃tH[}bg̃L[̃ftHgl
     */
    protected static final String DEFAULT_MESSAGE_ID_FORMAT_KEY = "message.id.format";
    /**
     * bZ[WvpeBt@Cx[Xl[̃L[̃ftHgl
     */
    protected static final String DEFAULT_MESSAGE_BASE_NAME_KEY = "message.basename";
    /**
     * bZ[WIDȂꍇɗOX[邩ۂ̃L[̃ftHgl
     */
    protected static final String DEFAULT_THROW_IF_RESOURCE_NOT_FOUND_KEY = "throw.if.resource.not.found";
    /**
     * bZ[WtH[}b^FQCNw肷L[̃ftHgl
     */
    protected static final String DEFAULT_MESSAGE_FORMATTER_FQCN_KEY = "message.formatter.fqcn";

    /**
     * NX[_ԋp܂B
     * 
     * <p>
     * ĂяoꂽXbhɃReLXgENX[_ݒ肳Ăꍇ͂̃ReLXgENX[_ԋp܂B<br>
     * łȂꍇ͂̃NX[hNX[_ԋp܂B
     * </p>
     * 
     * @return NX[_
     */
    protected static ClassLoader getClassLoader() {
        ClassLoader contextClassLoader = Thread.currentThread()
                .getContextClassLoader();
        if (contextClassLoader != null) {
            return contextClassLoader;
        }
        ClassLoader thisClassLoader = MessageManager.class.getClassLoader();
        return thisClassLoader;
    }

    /**
     * RXgN^
     * <p>
     * bZ[WID̃tH[}bg̃L[AbZ[WvpeBt@Cx[Xl[̃L[A bZ[WIDȂꍇɗOX[邩ۂ̃L[AbZ[WtH[}b^FQCNw肷L[̓ftHglݒ肵A {@link MessageManager#MessageManager(String, String, String, String, String)}
     * Ăяo܂B
     * </p>
     * 
     * @see MessageManager#MessageManager(String, String, String, String)
     * @param configFile ݒt@CpX(NX[_)
     */
    public MessageManager(String configFile) {
        this(configFile, DEFAULT_MESSAGE_ID_FORMAT_KEY,
                DEFAULT_MESSAGE_BASE_NAME_KEY,
                DEFAULT_THROW_IF_RESOURCE_NOT_FOUND_KEY,
                DEFAULT_MESSAGE_FORMATTER_FQCN_KEY);
    }

    /**
     * RXgN^
     * 
     * NX[_vpeBt@C擾AVbZ[W}l[W[\z܂B
     * 
     * <p>
     * Ŏ擾ݒt@CpXɊYvpeBt@C擾܂B<br>
     * vpeBt@C擾̍ۂ͈ȉ̍ڂ킹Đݒ肵܂B<br>
     * iݒlȂꍇ́AftHglgp܂Bj<br>
     * EbZ[WID̃tH[}bg̃L[<br>
     * EbZ[WvpeBt@C̃x[Xl[̃L[<br>
     * EbZ[WIDs̗OX[L<br>
     * EbZ[WtH[}b^FQCN<br>
     * 
     * vpeBt@C瓾bZ[WID̃tH[}bgAOX[tO̓oϐɊi[܂B<br>
     * bZ[Wx[Xl[̓NX[_擾ł̂SĒǉÃXg̃oϐɊi[܂B
     * </p>
     * 
     * @param configFile ݒt@CpX(NX[_)
     * @param messageIdFormatKey bZ[WID̃tH[}bg̃L[
     * @param messageBaseNameKey bZ[WvpeBt@Cx[Xl[̃L[
     * @param throwIfResourceNotFoundKey bZ[WIDȂꍇɗOX[邩ۂ̃L[
     * @param messageFormatterFqcnKey bZ[WtH[}b^FQCNw肷L[
     */
    public MessageManager(String configFile, String messageIdFormatKey,
                          String messageBaseNameKey,
                          String throwIfResourceNotFoundKey,
                          String messageFormatterFqcnKey) {
        try {
            ClassLoader cl = getClassLoader();
            {
                String format = null;
                String throwIfNotFound = null;
                String messageFormatterFqcn = null;
                // messageIdFormat,throwIfResourceNotFound,messageFormatterFqcn̓NX[_ŗDx̍
                InputStream strm = cl.getResourceAsStream(configFile);
                if (strm != null) {
                    Properties p = new Properties();
                    p.load(strm);
                    format = p.getProperty(messageIdFormatKey);
                    throwIfNotFound = p.getProperty(throwIfResourceNotFoundKey);
                    messageFormatterFqcn = p
                            .getProperty(messageFormatterFqcnKey);
                }
                if (format != null) {
                    messageIdFormat = format;
                }
                if (throwIfNotFound != null) {
                    throwIfResourceNotFound = Boolean
                            .parseBoolean(throwIfNotFound);
                }
                if (messageFormatterFqcn != null) {
                    try {
                        Class<?> clazz = Class.forName(messageFormatterFqcn,
                                true, cl);
                        messageFormatter = (MessageFormatter) clazz
                                .newInstance();
                    } catch (Exception e) {
                        StringBuilder sb = new StringBuilder(
                                "MessageFormatter[").append(
                                messageFormatterFqcn).append("] is not found.");
                        throw new MessageRuntimeException(sb.toString(), e);
                    }
                } else {
                    messageFormatter = new DefaultMessageFormatter();
                }
            }

            Enumeration<URL> urls = cl.getResources(configFile);
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (url != null) {
                    Properties p = new Properties();
                    InputStream strm = url.openStream();
                    p.load(strm);
                    // messageBasename̓NX[_ǂݍ߂̂͑SĒǉ
                    if (p.containsKey(messageBaseNameKey)) {
                        String[] basenameArray = p.getProperty(
                                messageBaseNameKey).split(",");
                        for (String s : basenameArray) {
                            String basename = s.trim();
                            if (!"".equals(basename)) {
                                basenames.add(basename);
                            }
                        }
                    }
                }
            }
        } catch (MessageRuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new MessageRuntimeException(e);
        }
    }

    /**
     * bZ[WvpeBt@C̃x[Xl[ɑΉ\[Xohԋp܂B
     * 
     * ̃P[null̏ꍇ́AftHg̃P[Xgp܂B ̃x[Xl[ɑΉ郊\[X݂Ȃꍇ́AOX[tȌꍇɌAx[Xl[𕶎o͂܂B x[Xl[̕o͂Ɠ{@link MessageRuntimeException}X[܂B OX[tȌꍇ́Anulllݒ肵\[Xohԋp܂B
     * 
     * @param basename bZ[WvpeBt@C̃x[Xl[
     * @param locale P[
     * @return \[Xoh
     */
    protected ResourceBundle getResourceBundle(String basename, Locale locale) {
        if (locale == null) {
            locale = Locale.getDefault();
        }

        ResourceBundle bundle = null;
        try {
            bundle = ResourceBundle.getBundle(basename, locale);
        } catch (MissingResourceException e) {
            if (throwIfResourceNotFound) {
                StringBuilder sb = new StringBuilder("resource[").append(
                        basename).append("] is not found");
                throw new MessageRuntimeException(sb.toString(), e);
            }
        }
        return bundle;
    }

    /**
     * vpeBt@Cɑ݂AbZ[WIDɑΉbZ[Wԋp܂B
     * 
     * <dl>
     * <dt><code>bundle</code><code>false</code>̏ꍇ</dt>
     * <dd>
     * nulllԋp܂B
     * 
     * <dt><code>key</code><code>false</code>̏ꍇ</dt> nulllԋp܂B
     * 
     * </dl>
     * 
     * <p>
     * ̃\[Xohɑ΂āÃbZ[WIDɑΉ郁bZ[W擾܂B bZ[WIDɑΉ郁bZ[W݂Ȃꍇ́AOX[tȌꍇɌA擾łȂbZ[WID𕶎o͂܂B bZ[WID̕o͂Ɠ{@link MessageRuntimeException}X[܂B
     * OX[tȌꍇ́Anulllԋp܂B
     * </p>
     * 
     * @param bundle \[Xoh
     * @param key bZ[WID
     * @return bZ[WIDɂ擾郁bZ[W
     */
    protected String getStringOrNull(ResourceBundle bundle, String key) {
        if (bundle == null) {
            return null;
        }
        if (key == null) {
            if (throwIfResourceNotFound) {
                throw new MessageRuntimeException("key is null");
            }
            return null;
        }

        try {
            return bundle.getString(key);
        } catch (MissingResourceException e) {
            return null;
        }
    }

    /**
     * bZ[Wp^[ԋp܂B
     * 
     * @param messageId bZ[WID
     * @param locale bZ[W̃P[
     * @return bZ[WIDɑ΂郁bZ[Wp^[
     */
    protected String getMessagePattern(String messageId, Locale locale) {
        String message = null;
        for (String basename : basenames) {
            ResourceBundle bundle = getResourceBundle(basename, locale);
            message = getStringOrNull(bundle, messageId);
            if (message != null) {
                break;
            }
        }
        if (message == null && throwIfResourceNotFound) {
            StringBuilder sb = new StringBuilder("key[").append(messageId)
                    .append("] is not found");
            throw new MessageRuntimeException(sb.toString());
        }
        return message;
    }

    /**
     * bZ[Wԋp܂B
     * 
     * P[nulllݒ肵A {@link MessageManager#getMessage(boolean, String, Locale, Object...)} Ăяo܂B P[ɂ̓ftHglgp܂B
     * 
     * @param resource \[X̗L
     * @param messageIdOrPattern bZ[WIDi\[XL̏ꍇj / bZ[Wp^[(\[X̏ꍇ)
     * @param args up[^
     * @return bZ[W
     * @throws MessageRuntimeException sȃp^[w肵ꍇ
     */
    public String getMessage(boolean resource, String messageIdOrPattern,
            Object... args) throws MessageRuntimeException {
        return getMessage(resource, messageIdOrPattern, null, args);
    }

    /**
     * bZ[Wԋp܂B
     * 
     * <dl>
     * <dt><code>resource</code><code>true</code>̏ꍇ</dt>
     * <dd>
     * bZ[WIDɑ΂郁bZ[Wp^[bZ[WvpeBt@Cx[Xl[XgΉevpeBt@CT܂B<br>
     * ŏɌbZ[Wp^[ɒup[^𖄂ߍ񂾃bZ[Wԋp܂Bp^[ {@link MessageFormat} ̌`ɂĂBsȃp^[w肵ꍇA{@link MessageRuntimeException}X[܂<br>
     * x[Xl[ɑΉvpeBt@C͎w肵P[Ŏ擾܂BP[nullݒ肵ꍇ {@link Locale#getDefault()}gp܂B</dd>
     * 
     * <dt><code>resource</code><code>false</code>̏ꍇ</dt>
     * <dd>
     * bZ[Wp^[(<code>messageIdOrPattern</code>)ɒup[^𖄂ߍ񂾃bZ[Wԋp܂Bp^[ {@link MessageFormat}̌`ɂĂBsȃp^[w肵ꍇA {@link MessageRuntimeException}X[܂B</dd>
     * </dl>
     * 
     * @param resource \[X̗L
     * @param messageIdOrPattern bZ[WIDi\[XL̏ꍇj / bZ[Wp^[(\[X̏ꍇ)
     * @param locale P[
     * @param args up[^
     * @return bZ[W
     * @throws MessageRuntimeException sȃp^[w肵ꍇ
     */
    public String getMessage(boolean resource, String messageIdOrPattern,
            Locale locale, Object... args) throws MessageRuntimeException {
        String pattern = null;
        StringBuilder message = new StringBuilder();

        if (resource) {
            pattern = getMessagePattern(messageIdOrPattern, locale);
            message.append(String.format(messageIdFormat, messageIdOrPattern));
        } else {
            pattern = messageIdOrPattern;
        }

        if (pattern != null) {
            try {
                String body = messageFormatter.format(pattern, args);
                message.append(body);
            } catch (IllegalArgumentException e) {
                StringBuilder sb = new StringBuilder(
                        "message pattern is illeagal. pattern=")
                        .append(pattern).append("]");
                if (resource) {
                    sb.append(" logId=");
                    sb.append(messageIdOrPattern);
                }
                throw new MessageRuntimeException(sb.toString(), e);
            }
        }
        return message.toString();
    }
}
