package jp.terasoluna.fw.collector;

import static org.junit.Assert.*;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;

import jp.terasoluna.fw.collector.exception.CollectorExceptionHandler;
import jp.terasoluna.fw.collector.exception.CollectorExceptionHandlerStatus;
import jp.terasoluna.fw.collector.file.SkipValidationErrorHandler;
import jp.terasoluna.fw.collector.validate.ExceptionValidationErrorHandler;
import jp.terasoluna.fw.collector.validate.ValidateErrorStatus;
import jp.terasoluna.fw.collector.validate.ValidationErrorException;
import jp.terasoluna.fw.collector.validate.ValidationErrorHandler;
import jp.terasoluna.fw.collector.vo.DataValueObject;
import jp.terasoluna.fw.exception.SystemException;
import jp.terasoluna.fw.file.dao.FileLineException;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;

public class AbstractCollector003Test {

    @BeforeClass
    public static void setUpBeforeClass() throws Exception {
    }

    @AfterClass
    public static void tearDownAfterClass() throws Exception {
    }

    @Before
    public void setUp() throws Exception {
    }

    @After
    public void tearDown() throws Exception {
    }

    /**
     * f[^0̏ꍇ̃eXgB
     * ̓f[^F[]
     * o̓f[^F[]
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testNormal001() throws Exception {
        int dataNum = 0;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vfƂmF
        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^1̏ꍇ
     * ̓f[^F[1]
     * o̓f[^F[1]
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testNormal002() throws Exception {
        int dataNum = 1;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^3̏ꍇ
     * ̓f[^F[1, 2, 3]
     * o̓f[^F[1, 2, 3]
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testNormal003() throws Exception {
        int dataNum = 3;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertTrue(collector.hasNext());
        assertEquals("3", collector.next().getHoge());
        assertEquals("2", collector.getPrevious().getHoge());
        assertEquals("3", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("2", collector.getPrevious().getHoge());
        assertEquals("3", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^beanƂnull܂܂ꍇ
     * ̓f[^F[null, 2, 3, 4, 5, 6]
     * o̓f[^F[null, 2, 3, 4, 5, 6]
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testContainsNull001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Arrays.asList(1);
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertNull(collector.next());
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^beanƂnull܂܂ꍇ
     * ̓f[^F[1, 2, null, 4, 5, 6]
     * o̓f[^F[1, 2, null, 4, 5, 6]
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testContainsNull002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Arrays.asList(3);
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        // 3vf
        assertTrue(collector.hasNext());
        assertNull(collector.next());
        assertEquals("2", collector.getPrevious().getHoge());
        assertNull(collector.getCurrent());
        assertEquals("4", collector.getNext().getHoge());

        // 4vf
        assertTrue(collector.hasNext());
        assertEquals("4", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("4", collector.getCurrent().getHoge());
        assertEquals("5", collector.getNext().getHoge());

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^beanƂnull܂܂ꍇ
     * ̓f[^F[1, 2, 3, 4, 5, null]
     * o̓f[^F[1, 2, 3, 4, 5, null]
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testContainsNull003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Arrays.asList(6);
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        // 6vf
        assertTrue(collector.hasNext());
        assertNull(collector.next());
        assertEquals("5", collector.getPrevious().getHoge());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.CONTINUE
     * o̓f[^F[1, 2, 3, 4, 5, 6]
     * ValidateErrorStatus.CONTINUȄꍇA̓`FbNG[̂ƂɂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorContinue001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler(ValidateErrorStatus.CONTINUE);
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.CONTINUE
     * o̓f[^F[1, 2, 3, 4, 5, 6]
     * ValidateErrorStatus.CONTINUȄꍇA̓`FbNG[̂ƂɂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorContinue002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(3);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler(ValidateErrorStatus.CONTINUE);
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.CONTINUE
     * o̓f[^F[1, 2, 3, 4, 5, 6]
     * ValidateErrorStatus.CONTINUȄꍇA̓`FbNG[̂ƂɂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorContinue003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(6);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler(ValidateErrorStatus.CONTINUE);
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.SKIP
     * o̓f[^F[2, 3, 4, 5, 6]
     * ValidateErrorStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorSkip001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "3", "2", "3", "4");

        // 3vf
        assertNextData(collector, "4", "3", "4", "5");

        // 4vf
        assertNextData(collector, "5", "4", "5", "6");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.SKIP
     * o̓f[^F[1, 2, 4, 5, 6]
     * ValidateErrorStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorSkip002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(3);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "4");

        // 3vf
        assertNextData(collector, "4", "2", "4", "5");

        // 4vf
        assertNextData(collector, "5", "4", "5", "6");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.SKIP
     * o̓f[^F[1, 2, 3, 4, 5]
     * ValidateErrorStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorSkip003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(6);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2(̓`FbNG[), 3, 4(̓`FbNG[), 5(̓`FbNG[), 6, 7(̓`FbNG[), 8(̓`FbNG[)]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.SKIP
     * o̓f[^F[3, 6]
     * ValidateErrorStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorSkip004() throws Exception {
        int dataNum = 8;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1, 2, 4, 5, 7, 8);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("3", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("3", collector.getCurrent().getHoge());
        assertEquals("6", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("3", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("3", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2(̓`FbNG[), 3(̓`FbNG[), 4(̓`FbNG[)]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.SKIP
     * o̓f[^F[]
     * ValidateErrorStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorSkip005() throws Exception {
        int dataNum = 4;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1, 2, 3, 4);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vfƂmF
        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNSKIP̑ʔeXg(BUG_B30015)
     * ̓f[^F[1, 2(̓`FbNG[), 3(̓`FbNG[), ... , 21(̓`FbNG[), 22]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.SKIP
     * o̓f[^F[1, 22]
     * ValidateErrorStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * E̓`FbNG[̃f[^20iL[̃ftHgjȏ㑱ĂAȂł邱
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     */
    @Test
    public void testOccurManyValidationErrorSkip001() throws Exception {
        int dataNum = 22;
        List<Integer> validationErrorOccurPoints = Arrays.asList(2, 3, 4, 5, 6,
                7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();

        Collector<AbstractCollectorTestBean> collector = createTestCollector(
                dataNum, validationErrorOccurPoints, argValidationErrorHandler,
                exceptionOccurPoints, argExceptionHandler, thrownException,
                nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("22", collector.getNext().getHoge());

        // 2vfɃANZXOɁASKIP20悤WAIT
        int arraySize = 0;
        int threadCount = 0;
        Thread th[] = null;
        do {
            arraySize = Thread.activeCount() + 1;
            th = new Thread[arraySize];
            threadCount = Thread.enumerate(th);
        } while (threadCount >= th.length);

        for (Thread t : th) {
            if (t != null && t.getName().startsWith("CollectorThreadFactory")) {
                // 񓯊XbhWAITTERMINATEDɂȂ܂ŁAmainXbh~
                while (t.getState() != Thread.State.TERMINATED
                        && t.getState() != Thread.State.WAITING) {
                    try {
                        Thread.sleep(1);
                    } catch (InterruptedException e) {
                        fail();
                    }
                }
            }
        }
        // Xbh̑҂ ܂
        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("22", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("22", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        collector.close();
    }

    /**
     * OSKIP̑ʔeXg(BUG_B30015)
     * ̓f[^F[1, (O), (O), ... , (O), 22]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[1,22]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA͎ɗOvf́AXLbvAo͂ȂB
     * EOSKIP̃f[^20iL[̃ftHgjȏ㑱ĂAȂł邱
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     */
    @Test
    public void testOccurManyExceptionWithSkipExceptionHandler001() throws Exception {
        int dataNum = 22;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(2, 3, 4, 5, 6, 7, 8,
                9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21);
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();

        Collector<AbstractCollectorTestBean> collector = createTestCollector(
                dataNum, validationErrorOccurPoints, argValidationErrorHandler,
                exceptionOccurPoints, argExceptionHandler, thrownException,
                nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("22", collector.getNext().getHoge());

        // 2vfɃANZXOɁASKIP20悤WAIT
        int arraySize = 0;
        int threadCount = 0;
        Thread th[] = null;
        do {
            arraySize = Thread.activeCount() + 1;
            th = new Thread[arraySize];
            threadCount = Thread.enumerate(th);
        } while (threadCount >= th.length);

        for (Thread t : th) {
            if (t != null && t.getName().startsWith("CollectorThreadFactory")) {
                // 񓯊XbhWAITTERMINATEDɂȂ܂ŁAmainXbh~
                while (t.getState() != Thread.State.TERMINATED
                        && t.getState() != Thread.State.WAITING) {
                    try {
                        Thread.sleep(1);
                    } catch (InterruptedException e) {
                        fail();
                    }
                }
            }
        }
        // Xbh̑҂ ܂
        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("22", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("22", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        collector.close();
    }


    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.END
     * o̓f[^F[]
     * ValidateErrorStatus.END̏ꍇA̓`FbNG[f[^̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorEnd001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler(ValidateErrorStatus.END);
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vfƂmF
        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.END
     * o̓f[^F[1, 2]
     * ValidateErrorStatus.END̏ꍇA̓`FbNG[f[^̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorEnd002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(3);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler(ValidateErrorStatus.END);
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ValidationErrorHandler̕ԋplFValidateErrorStatus.END
     * o̓f[^F[1, 2]
     * ValidateErrorStatus.END̏ꍇA̓`FbNG[f[^̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorEnd003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(6);
        ValidationErrorHandler argValidationErrorHandler = new SkipValidationErrorHandler(ValidateErrorStatus.END);
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ (̓`FbNtRN^̃ftHgݒ)
     * ̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ̓`FbNG[t̏o̓f[^́Anext()̂ݗOX[AgetPrevious()AgetCurrent()AgetNext()ł̓f[^擾łB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithoutExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (ValidationErrorException e) {
        }
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ (̓`FbNtRN^̃ftHgݒ)
     * ̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ̓`FbNG[t̏o̓f[^́Anext()̂ݗOX[AgetPrevious()AgetCurrent()AgetNext()ł̓f[^擾łB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithoutExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(3);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (ValidationErrorException e) {
        }
        assertEquals("2", collector.getPrevious().getHoge());
        assertEquals("3", collector.getCurrent().getHoge());
        assertEquals("4", collector.getNext().getHoge());

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ (̓`FbNtRN^̃ftHgݒ)
     * ̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ̓`FbNG[t̏o̓f[^́Anext()̂ݗOX[AgetPrevious()AgetCurrent()AgetNext()ł̓f[^擾łB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithoutExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(6);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (ValidationErrorException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ̓`FbNG[t̏o̓f[^́Anext()̂ݗOX[AgetPrevious()AgetCurrent()AgetNext()ł̓f[^擾łB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithThrowExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (ValidationErrorException e) {
        }
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ̓`FbNG[t̏o̓f[^́Anext()̂ݗOX[AgetPrevious()AgetCurrent()AgetNext()ł̓f[^擾łB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithThrowExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(3);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (ValidationErrorException e) {
        }
        assertEquals("2", collector.getPrevious().getHoge());
        assertEquals("3", collector.getCurrent().getHoge());
        assertEquals("4", collector.getNext().getHoge());

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ̓`FbNG[t̏o̓f[^́Anext()̂ݗOX[AgetPrevious()AgetCurrent()AgetNext()ł̓f[^擾łB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithThrowExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(6);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (ValidationErrorException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[2, 3, 4, 5, 6]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithSkipExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "3", "2", "3", "4");

        // 3vf
        assertNextData(collector, "4", "3", "4", "5");

        // 4vf
        assertNextData(collector, "5", "4", "5", "6");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[1, 2, 4, 5, 6]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithSkipExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(3);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "4");

        // 3vf
        assertNextData(collector, "4", "2", "4", "5");

        // 4vf
        assertNextData(collector, "5", "4", "5", "6");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[1, 2, 3, 4, 5]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithSkipExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(6);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2(̓`FbNG[), 3, 4(̓`FbNG[), 5(̓`FbNG[), 6, 7(̓`FbNG[), 8(̓`FbNG[)]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[3, 6]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithSkipExceptionHandler004() throws Exception {
        int dataNum = 8;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1, 2, 4, 5, 7, 8);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("3", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("3", collector.getCurrent().getHoge());
        assertEquals("6", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("3", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("3", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2(̓`FbNG[), 3(̓`FbNG[), 4(̓`FbNG[)]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA̓`FbNG[f[^́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithSkipExceptionHandler005() throws Exception {
        int dataNum = 4;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1, 2, 3, 4);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vfƂmF
        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1(̓`FbNG[), 2, 3, 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.END
     * o̓f[^F[]
     * CollectorExceptionHandlerStatus.END̏ꍇA̓`FbNG[f[^̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithEndExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(1);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new EndCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vfƂmF
        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3(̓`FbNG[), 4, 5, 6]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.END
     * o̓f[^F[1, 2]
     * CollectorExceptionHandlerStatus.END̏ꍇA̓`FbNG[f[^̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithEndExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(3);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new EndCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * ̓`FbNG[ꍇ
     * ̓f[^F[1, 2, 3, 4, 5, 6(̓`FbNG[)]
     * ValidationErrorHandleřʁFValidationErrorExceptionX[
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.END
     * o̓f[^F[1, 2, 3, 4, 5]
     * CollectorExceptionHandlerStatus.END̏ꍇA̓`FbNG[f[^̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurValidationErrorThrowWithEndExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Arrays.asList(6);
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Collections.emptyList();
        CollectorExceptionHandler argExceptionHandler = new EndCollectorExceptionHandler();
        Exception thrownException = null;
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[(O), 2, 3, 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[2, 3, 4, 5, 6]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA͎ɗOvf́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithSkipExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1);
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "3", "2", "3", "4");

        // 3vf
        assertNextData(collector, "4", "3", "4", "5");

        // 4vf
        assertNextData(collector, "5", "4", "5", "6");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[1, 2, (O), 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[1, 2, 4, 5, 6]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA͎ɗOvf́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithSkipExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(3);
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "4");

        // 3vf
        assertNextData(collector, "4", "2", "4", "5");

        // 4vf
        assertNextData(collector, "5", "4", "5", "6");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[1, 2, 3, 4, 5, (O)]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[1, 2, 3, 4, 5]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA͎ɗOvf́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithSkipExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(6);
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[(O), (O), 3, (O), (O), 6, (O), (O)]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[3, 6]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA͎ɗOvf́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithSkipExceptionHandler004() throws Exception {
        int dataNum = 8;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1, 2, 4, 5, 7, 8);
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("3", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("3", collector.getCurrent().getHoge());
        assertEquals("6", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("3", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("3", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[(O), (O), (O), (O)]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.SKIP
     * o̓f[^F[]
     * CollectorExceptionHandlerStatus.SKIP̏ꍇA͎ɗOvf́AXLbvAo͂ȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithSkipExceptionHandler005() throws Exception {
        int dataNum = 4;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1, 2, 3, 4);
        CollectorExceptionHandler argExceptionHandler = new SkipCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vfƂmF
        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[(O), 2, 3, 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.END
     * o̓f[^F[]
     * CollectorExceptionHandlerStatus.END̏ꍇA͎ɗOvf̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithEndExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1);
        CollectorExceptionHandler argExceptionHandler = new EndCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vfƂmF
        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertNull(collector.getPrevious());
        assertNull(collector.getCurrent());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[1, 2, (O), 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.END
     * o̓f[^F[1, 2]
     * CollectorExceptionHandlerStatus.END̏ꍇA͎ɗOvf̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithEndExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(3);
        CollectorExceptionHandler argExceptionHandler = new EndCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɗOꍇ
     * ̓f[^F[1, 2, 3, 4, 5, (O)]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.END
     * o̓f[^F[1, 2, 3, 4, 5]
     * CollectorExceptionHandlerStatus.END̏ꍇA͎ɗOvf̒Õf[^Ō̃f[^ƂȂB
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithEndExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(6);
        CollectorExceptionHandler argExceptionHandler = new EndCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎Ƀ^COꍇ
     * ̓f[^F[(^CO), 2, 3, 4, 5, 6]
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[(^CO), 2, 3, 4, 5, 6]
     * ^COvf́Anext()AgetPrevious()AgetCurrent()AgetNext()Ń^CÔ܂܃X[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurRuntimeExceptionWithoutExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1);
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertNull(collector.getPrevious());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎Ƀ^COꍇ
     * ̓f[^F[1, 2, (^CO), 4, 5, 6]
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[1, 2, (^CO), 4, 5, 6]
     * ^COvf́Anext()AgetPrevious()AgetCurrent()AgetNext()Ń^CÔ܂܃X[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurRuntimeExceptionWithoutExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(3);
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }

        // 3vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("2", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("4", collector.getNext().getHoge());

        // 4vf
        assertTrue(collector.hasNext());
        assertEquals("4", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("4", collector.getCurrent().getHoge());
        assertEquals("5", collector.getNext().getHoge());

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎Ƀ^COꍇ
     * ̓f[^F[1, 2, 3, 4, 5, (^CO)]
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[1, 2, 3, 4, 5, (^CO)]
     * ^COvf́Anext()AgetPrevious()AgetCurrent()AgetNext()Ń^CÔ܂܃X[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurRuntimeExceptionWithoutExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(6);
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }

        // 6vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎Ƀ^COꍇ
     * ̓f[^F[(^CO), 2, 3, 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[(^CO), 2, 3, 4, 5, 6]
     * ^COvf́Anext()AgetPrevious()AgetCurrent()AgetNext()Ń^CÔ܂܃X[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurRuntimeExceptionWithThrowExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1);
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertNull(collector.getPrevious());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎Ƀ^COꍇ
     * ̓f[^F[1, 2, (^CO), 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[1, 2, (^CO), 4, 5, 6]
     * ^COvf́Anext()AgetPrevious()AgetCurrent()AgetNext()Ń^CÔ܂܃X[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurRuntimeExceptionWithThrowExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(3);
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }

        // 3vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("2", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("4", collector.getNext().getHoge());

        // 4vf
        assertTrue(collector.hasNext());
        assertEquals("4", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("4", collector.getCurrent().getHoge());
        assertEquals("5", collector.getNext().getHoge());

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎Ƀ^COꍇ
     * ̓f[^F[1, 2, 3, 4, 5, (^CO)]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[1, 2, 3, 4, 5, (^CO)]
     * ^COvf́Anext()AgetPrevious()AgetCurrent()AgetNext()Ń^CÔ܂܃X[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurRuntimeExceptionWithThrowExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(6);
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = new FileLineException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }

        // 6vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (FileLineException e) {
            assertSame(thrownException, e);
        }
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɌOꍇ
     * ̓f[^F[(O), 2, 3, 4, 5, 6]
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[(SystemExceptionɃbvꂽO), 2, 3, 4, 5, 6]
     * Ovf́Anext()AgetPrevious()AgetCurrent()AgetNext()SystemExceptionɌObvX[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithoutExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1);
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = new IOException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertNull(collector.getPrevious());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɌOꍇ
     * ̓f[^F[1, 2, (O), 4, 5, 6]
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[1, 2, (SystemExceptionɃbvꂽO), 4, 5, 6]
     * Ovf́Anext()AgetPrevious()AgetCurrent()AgetNext()SystemExceptionɌObvX[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithoutExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(3);
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = new IOException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }

        // 3vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("2", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("4", collector.getNext().getHoge());

        // 4vf
        assertTrue(collector.hasNext());
        assertEquals("4", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("4", collector.getCurrent().getHoge());
        assertEquals("5", collector.getNext().getHoge());

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɌOꍇ
     * ̓f[^F[1, 2, 3, 4, 5, (O)]
     * CollectorExceptionHandlerFȂ
     * o̓f[^F[1, 2, 3, 4, 5, (SystemExceptionɃbvꂽO)]
     * Ovf́Anext()AgetPrevious()AgetCurrent()AgetNext()SystemExceptionɌObvX[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithoutExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(6);
        CollectorExceptionHandler argExceptionHandler = null;
        Exception thrownException = new IOException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }

        // 6vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɌOꍇ
     * ̓f[^F[(O), 2, 3, 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[(SystemExceptionɃbvꂽO), 2, 3, 4, 5, 6]
     * Ovf́Anext()AgetPrevious()AgetCurrent()AgetNext()SystemExceptionɌObvX[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithThrowExceptionHandler001() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(1);
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = new IOException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertNull(collector.getPrevious());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("2", collector.getCurrent().getHoge());
        assertEquals("3", collector.getNext().getHoge());

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɌOꍇ
     * ̓f[^F[1, 2, (O), 4, 5, 6]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[1, 2, (SystemExceptionɃbvꂽO), 4, 5, 6]
     * Ovf́Anext()AgetPrevious()AgetCurrent()AgetNext()SystemExceptionɌObvX[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithThrowExceptionHandler002() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(3);
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = new IOException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertTrue(collector.hasNext());
        assertEquals("2", collector.next().getHoge());
        assertEquals("1", collector.getPrevious().getHoge());
        assertEquals("2", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }

        // 3vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("2", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("4", collector.getNext().getHoge());

        // 4vf
        assertTrue(collector.hasNext());
        assertEquals("4", collector.next().getHoge());
        try {
            collector.getPrevious();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("4", collector.getCurrent().getHoge());
        assertEquals("5", collector.getNext().getHoge());

        // 5vf
        assertNextData(collector, "5", "4", "5", "6");

        // 6vf
        assertTrue(collector.hasNext());
        assertEquals("6", collector.next().getHoge());
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        assertEquals("6", collector.getCurrent().getHoge());
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * f[^͎ɌOꍇ
     * ̓f[^F[1, 2, 3, 4, 5, (O)]
     * CollectorExceptionHandleřʁFCollectorExceptionHandlerStatus.THROW
     * o̓f[^F[1, 2, 3, 4, 5, (SystemExceptionɃbvꂽO)]
     * Ovf́Anext()AgetPrevious()AgetCurrent()AgetNext()SystemExceptionɌObvX[
     * EhasNext()Anext()AgetPrevious()AgetCurrent()AgetNext()̌ʂAo̓f[^ɑ̂ł邱
     * Ẽf[^Ԃnext()sNoSuchElementExceptionX[邱
     * Ẽf[^Ԃnext()sĂAgetPrevious()AgetCurrent()AgetNext()̌ʂnext()sOƕςȂ
     */
    @Test
    public void testOccurExceptionWithThrowExceptionHandler003() throws Exception {
        int dataNum = 6;
        List<Integer> validationErrorOccurPoints = Collections.emptyList();
        ValidationErrorHandler argValidationErrorHandler = new ExceptionValidationErrorHandler();
        List<Integer> exceptionOccurPoints = Arrays.asList(6);
        CollectorExceptionHandler argExceptionHandler = new ThrowCollectorExceptionHandler();
        Exception thrownException = new IOException("test");
        List<Integer> nullBeanPoints = Collections.emptyList();
        Collector<AbstractCollectorTestBean> collector = createTestCollector(dataNum, validationErrorOccurPoints, argValidationErrorHandler, exceptionOccurPoints, argExceptionHandler, thrownException, nullBeanPoints);

        // 1vf
        assertTrue(collector.hasNext());
        assertEquals("1", collector.next().getHoge());
        assertNull(collector.getPrevious());
        assertEquals("1", collector.getCurrent().getHoge());
        assertEquals("2", collector.getNext().getHoge());

        // 2vf
        assertNextData(collector, "2", "1", "2", "3");

        // 3vf
        assertNextData(collector, "3", "2", "3", "4");

        // 4vf
        assertNextData(collector, "4", "3", "4", "5");

        // 5vf
        assertTrue(collector.hasNext());
        assertEquals("5", collector.next().getHoge());
        assertEquals("4", collector.getPrevious().getHoge());
        assertEquals("5", collector.getCurrent().getHoge());
        try {
            collector.getNext();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }

        // 6vf
        assertTrue(collector.hasNext());
        try {
            collector.next();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertNull(collector.getNext());

        assertFalse(collector.hasNext());

        try {
            collector.next();
            fail();
        } catch (NoSuchElementException e) {
        }
        assertEquals("5", collector.getPrevious().getHoge());
        try {
            collector.getCurrent();
            fail();
        } catch (SystemException e) {
            assertSame(thrownException, e.getCause());
        }
        assertNull(collector.getNext());

        collector.close();
    }

    /**
     * PɃf[^擾łꍇɎgpłȈՃAT[V\bhB<br>
     * assertTrue(collector.hasNext());<br>
     * assertEquals("2", collector.next().getHoge());<br>
     * assertEquals("1", collector.getPrevious().getHoge());<br>
     * assertEquals("2", collector.getCurrent().getHoge());<br>
     * assertEquals("3", collector.getNext().getHoge());<br>
     * <br>
     * assertNextData("2", "1", "2", "3");<br>
     * ƋLqłB<br>
     * OP[XnullԂP[Xł́Ã\bh͎gpAʂɃAT[V{邱ƁB
     * @param collector RN^
     * @param nextExpectedHoge collector.next().getHoge()̊Ғl
     * @param getPreviousExpectedHoge collector.getPrevious().getHoge()̊Ғl
     * @param getCurrentExpectedHoge collector.getCurrent().getHoge()̊Ғl
     * @param getNextExpectedHoge collector.getNext().getHoge()̊Ғl
     */
    private static void assertNextData(Collector<AbstractCollectorTestBean> collector, String nextExpectedHoge, String getPreviousExpectedHoge, String getCurrentExpectedHoge, String getNextExpectedHoge) {
        assertTrue(collector.hasNext());
        assertEquals(nextExpectedHoge, collector.next().getHoge());
        assertEquals(getPreviousExpectedHoge, collector.getPrevious().getHoge());
        assertEquals(getCurrentExpectedHoge, collector.getCurrent().getHoge());
        assertEquals(getNextExpectedHoge, collector.getNext().getHoge());
    }

    /**
     * eXgf[^񋟂RN^𐶐B
     * <p>
     * 1) ̓f[^F[1, 2, 3, 4, 5]̍<br>
     * createTestCollector(5, Collections.EMPTY_LIST, new ExceptionValidationErrorHandler(), Collections.EMPTY_LIST, null, null, Collections.EMPTY_LIST);<br>
     * <br>
     * 2) ̓f[^F[1, null, 3, null, 5]̍<br>
     * createTestCollector(5, Collections.EMPTY_LIST, new ExceptionValidationErrorHandler(), Collections.EMPTY_LIST, null, null, Arrays.asList(2, 4));<br>
     * <br>
     * 3) ̓f[^F[1, 2(̓`FbNG[), 3, 4(̓`FbNG[), 5]̍<br>
     * createTestCollector(5, Arrays.asList(2, 4), new ExceptionValidationErrorHandler(), Collections.EMPTY_LIST, null, null, Collections.EMPTY_LIST);<br>
     * <br>
     * 4) ̓f[^F[1, (O), 3, 4(O), 5]̍<br>
     * createTestCollector(5, Collections.EMPTY_LIST, new ExceptionValidationErrorHandler(), Arrays.asList(2, 4), null, O, Collections.EMPTY_LIST);<br>
     * <br>
     * ^CvZ[tɂꍇ́ACollections.EMPTY_LIST̑ɁACollections.emptyList()ListInteger^̕ϐɓĎgpB
     * @param dataNum f[^̌
     * @param validationErrorOccurPoints ̓`FbNG[|Cg
     * @param argValidationErrorHandler ValidationErrorHandler
     * @param exceptionOccurPoints O|Cg
     * @param argExceptionHandler ExceptionHandler
     * @param thrownException X[O
     * @param nullBeanPoints Bean̑nulll߂|Cg
     * @return eXgf[^񋟂RN^
     */
    private static Collector<AbstractCollectorTestBean> createTestCollector(final int dataNum, final List<Integer> validationErrorOccurPoints, final ValidationErrorHandler argValidationErrorHandler, final List<Integer> exceptionOccurPoints, final CollectorExceptionHandler argExceptionHandler, final Exception thrownException, final List<Integer> nullBeanPoints) {
        Collector<AbstractCollectorTestBean> collector = new AbstractCollector<AbstractCollectorTestBean>() {
            {
                this.validator = new Validator() {

                    public void validate(Object target, Errors errors) {
                        AbstractCollectorTestBean data = (AbstractCollectorTestBean) target;
                        if (validationErrorOccurPoints.contains(new Integer(data.getHoge()))) {
                            errors.rejectValue("hoge", "errors.required");
                        }
                    }

                    public boolean supports(Class<?> clazz) {
                        return (clazz == AbstractCollectorTestBean.class);
                    }
                };
                this.validationErrorHandler = argValidationErrorHandler;
                this.exceptionHandler = argExceptionHandler;
            }

            public Integer call() throws Exception {
                for (int count = 1; count <= dataNum; count++) {
                    if (exceptionOccurPoints.contains(count)) {
                        addQueue(new DataValueObject(thrownException, count));
                    } else if (nullBeanPoints.contains(count)) {
                        addQueue(new DataValueObject(null, count));
                    } else {
                        AbstractCollectorTestBean bean = new AbstractCollectorTestBean();
                        bean.setHoge(String.valueOf(count));
                        addQueue(new DataValueObject(bean, count));
                    }
                }

                setFinish();
                return 0;
            }

        };

        return collector;

    }

    /**
     * CollectorExceptionHandlerStatus.ENDԋpCollectorExceptionHandlerB
     */
    private static class EndCollectorExceptionHandler implements CollectorExceptionHandler {
        public CollectorExceptionHandlerStatus handleException(
                DataValueObject dataValueObject) {
            return CollectorExceptionHandlerStatus.END;
        }
    }

    /**
     * CollectorExceptionHandlerStatus.SKIPԋpCollectorExceptionHandlerB
     */
    private static class SkipCollectorExceptionHandler implements CollectorExceptionHandler {
        public CollectorExceptionHandlerStatus handleException(
                DataValueObject dataValueObject) {
            return CollectorExceptionHandlerStatus.SKIP;
        }
    }

    /**
     * CollectorExceptionHandlerStatus.THROWԋpCollectorExceptionHandlerB
     */
    private static class ThrowCollectorExceptionHandler implements CollectorExceptionHandler {
        public CollectorExceptionHandlerStatus handleException(
                DataValueObject dataValueObject) {
            return CollectorExceptionHandlerStatus.THROW;
        }
    }

}
