/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.core;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import jp.terasoluna.fw.batch.openapi.JobContext;
import jp.terasoluna.fw.batch.openapi.ReturnCode;

/**
 * Wȕ󋵂ێNXB
 * 
 */
public class JobStatus {

    /**
     * WuԂ̗񋓃NXB
     *
     * <p><code>JobState</code> ł́Aȉ̋N󋵂񋟂</p>
     *
     * <div align="center">
     *  <table width="50%" border="1">
     *   <tr>
     *    <td> <b>N</b> </td>
     *    <td> <b>N󋵔ԍ</b> </td>
     *    <td> <b>Tv</b> </td>
     *   </tr>
     *   <tr>
     *    <td> <code>SUBMITTED</code> </td>
     *    <td> <code>0</code> </td>
     *    <td align="left">
     *      NO
     *    </td>
     *   </tr>
     *   <tr>
     *    <td> <code>STARTED</code> </td>
     *    <td> <code>1</code> </td>
     *    <td align="left">
     *      N
     *    </td>
     *   </tr>
     *   <tr>
     *    <td> <code>RESTARTED</code> </td>
     *    <td> <code>2</code> </td>
     *    <td align="left">
     *      ċN
     *    </td>
     *   </tr>
     *   <tr>
     *    <td> <code>ENDING_NORMALLY</code> </td>
     *    <td> <code>3</code> </td>
     *    <td align="left">
     *      I
     *    </td>
     *   </tr>
     *   <tr>
     *    <td> <code>ENDING_ABNORMALLY</code> </td>
     *    <td> <code>4</code> </td>
     *    <td align="left">
     *      ُI
     *    </td>
     *   </tr>
     *   <tr>
     *    <td> <code>INTERRUPTED_FOR_GRACEFUL_SHUTDOWN</code> </td>
     *    <td> <code>5</code> </td>
     *    <td align="left">
     *      fI
     *    </td>
     *   </tr>
     *   <tr>
     *    <td> <code>INTERRUPTED_FOR_IMMEDIATE_SHUTDOWN</code> </td>
     *    <td> <code>6</code> </td>
     *    <td align="left">
     *      I
     *    </td>
     *   </tr>
     *   <tr>
     *    <td> <code>SUSPENDING</code> </td>
     *    <td> <code>7</code> </td>
     *    <td align="left">
     *      f/I
     *    </td>
     *   </tr>
     *  </table>
     * </div>
     * <p>
     */
    public enum STATE {

        /**
         * NOB
         */
        SUBMITTED(false),
        /**
         * NB
         */
        STARTED(false),
        /**
         * ċNB
         */
        RESTARTED(false),
        /**
         * IB
         */
        ENDING_NORMALLY(true),
        /**
         * ُIB
         */
        ENDING_ABNORMALLY(true),
        /**
         * fIB
         */
        INTERRUPTED_FOR_GRACEFUL_SHUTDOWN(false),
        /**
         * IB
         */
        INTERRUPTED_FOR_IMMEDIATE_SHUTDOWN(false),
        /**
         * f/I
         */
        SUSPENDING(true);

        /**
         * WuԂIԂł邩ǂtOB
         */
        private final boolean isEndStatus;

        /**
         * RXgN^B
         * 
         * @param isEndStatus IԂłꍇɂ́A<code>true</code>
         */
        private STATE(boolean isEndStatus) {
            this.isEndStatus = isEndStatus;
        }

        /**
         * I󋵂𔻒fB
         *
         * @return I
         */
        public boolean isEndStatus() {
            return isEndStatus;
        }
    }

    /**
     * ob`XVB
     */
    private int batchUpdateCount = 0;

    /**
     * Ώۃf[^擾B
     */
    private int collected = 0;

    /**
     * WũR~bgB
     */
    private int commitCount = 0;

    /**
     * IB
     */
    private long jobEndTime = 0;

    /**
     * WuIR[hB
     */
    private Integer jobExitCode = null;

    /**
     * JnB
     */
    private long jobStartTime = 0;

    /**
     * 󋵁B
     */
    private STATE jobState = STATE.SUBMITTED;

    /**
     * Wȕ󋵂̃XgB
     */
    private List<JobStatus> childJobStatusList;

    /**
     * X^[gLB
     */
    private boolean restartable = false;

    /**
     * X^[gB
     */
    private int restartPoint = 0;

    /**
     * BLogic̏B
     */
    private ResultCounter resultCounter = null;

    /**
     * WuNGXgԍB
     */
    private String jobRequestNo = null;

    /**
     * WuIDB
     */
    private String jobId = null;

    /**
     * Wup[eBVԍB
     */
    private int partitionNo = 0;

    /**
     * WuL[B
     */
    private String partitionKey = null;

    /**
     * RXgN^B Wȕ󋵂B
     */
    public JobStatus() {
        childJobStatusList =
            Collections.synchronizedList(new ArrayList<JobStatus>());
        resultCounter = new ResultCounter();
    }

    /**
     * 󋵂uf/Ivɐݒ肷B
     * 
     */
    public void suspend() {
        setJobState(STATE.SUSPENDING);
    }
    
    /**
     * qWȕʂXgɒǉB
     * 
     * @param jobStatus qWȕ
     */
    protected void addChildJobStatus(JobStatus jobStatus) {
        childJobStatusList.add(jobStatus);
    }

    /**
     * rWlXWbN̎sJEgB
     * 
     * @param returnCode rWlXWbÑ^[R[h
     */
    public void countBLogic(ReturnCode returnCode) {
        resultCounter.count(returnCode);
    }

    /**
     * qWu󋵂擾B
     * 
     * @param jobContext WuReLXg
     * @return qWu
     */
    public JobStatus getChild(JobContext jobContext) {
        JobStatus childJobStatus = new JobStatus();
        resetChildData(childJobStatus, jobContext);
        return childJobStatus;    
    }

    /**
     * qWu󋵂ɐeWu󋵂ݒ肷B
     * 
     * @param childJobStatus qWu
     * @param jobContext WuReLXg
     */
    protected void resetChildData(JobStatus childJobStatus ,
            JobContext jobContext) {
        childJobStatus.setJobId(this.getJobId());
        childJobStatus.setJobRequestNo(this.getJobRequestNo());
        childJobStatus.setRestartPoint(this.getRestartPoint());
        childJobStatus.setRestartable(this.isRestartable());
        childJobStatus.setJobState(this.getJobState());
        childJobStatus.setPartitionNo(jobContext.getPartitionNo());
        childJobStatus.setPartitionKey(jobContext.getPartitionKey());

        if (isShutdownImmediate() || isShutdownGraceful()) {
            childJobStatus.suspend();
        }
        addChildJobStatus(childJobStatus);
    }
    
    /**
     * CollectoȑԋpB
     * 
     * @return Collectoȑ
     */
    public int getCollected() {
        return collected;
    }

    /**
     * R~bg擾B
     * 
     * @return R~bg
     */
    public int getCommitCount() {

        int count = 0;

        if (!childJobStatusList.isEmpty()) {
            for (JobStatus result : childJobStatusList) {
                count += result.getCommitCount();
            }
        } else {
            count = commitCount;
        }

        return count;
    }

    /**
     * p擾B
     * 
     * @return p
     */
    public int getErrorContinueCount() {

        int count = 0;

        if (!childJobStatusList.isEmpty()) {
            for (JobStatus result : childJobStatusList) {
                count += result.getErrorContinueCount();
            }
        } else {
            count = resultCounter.getErrorContinueCount();
        }

        return count;
    }

    /**
     * WuIR[h擾B
     * 
     * @return WuIR[h
     */
    public Integer getJobExitCode() {
        return jobExitCode;
    }

    /**
     * ob`XV擾B
     * 
     * @return ob`XV
     */
    public int getBatchUpdateCount() {
        return batchUpdateCount;
    }

    /**
     * WuJn擾B
     * 
     * @return WuJn
     */
    public String getJobStartTime() {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
        return sdf.format(new Date(jobStartTime));
    }

    /**
     * WȕԂ擾B
     * 
     * @return 
     */
    public STATE getJobState() {
        return jobState;
    }

    /**
     * WuԂ̕擾B
     * 
     * @return Ԃ̕
     */
    public String getJobStateStr() {

        if (jobState != null) {
            return jobState.toString();
        }

        return null;
    }

    /**
     * qWȕʂێXg擾B
     * 
     * @return qWȕʃXg
     */
    public List<JobStatus> getChildJobStatusList() {
        return childJobStatusList;
    }

    /**
     * p擾B
     * 
     * @return p
     */
    public int getNormalContinueCount() {

        int count = 0;

        if (!childJobStatusList.isEmpty()) {
            for (JobStatus result : childJobStatusList) {
                count += result.getNormalContinueCount();
            }
        } else {
            count = resultCounter.getNormalContinueCount();
        }

        return count;
    }

    /**
     * WuԂ擾B(msP)
     * 
     * @return WuԂ̕(xxxxxms)
     */
    public String getProcessingTime() {
        long now = System.currentTimeMillis();
        if (jobState.isEndStatus()) {
            now = jobEndTime;
        }
        return String.valueOf(now - jobStartTime) + "ms";
    }

    /**
     * X^[g|Cg擾B
     * 
     * @return X^[g|Cg
     */
    public int getRestartPoint() {
        return restartPoint + resultCounter.getToralCount();
    }

    /**
     * ob`XVJEgB
     * 
     * @param batchUpdateCount ob`XV
     */
    public void incrementBatchUpdateCount(int batchUpdateCount) {
        this.batchUpdateCount += batchUpdateCount;
    }

    /**
     * CollecteȑCNgB
     * 
     */
    public void incrementCollected() {
        collected++;
    }

    /**
     * R~bgCNgB
     * 
     */
    public void incrementCommitCount() {
        commitCount++;
    }

    /**
     * Wup󋵂ł邩ǂ擾B
     * 
     * @return Wup󋵂łꍇ <code>true</code>
     */
    public boolean isContinue() {
        return !jobState.isEndStatus();
    }

    /**
     * WuN/ċN󋵂ł邩ǂ擾B
     * 
     * @return WuN/ċN󋵂łꍇ <code>true</code>
     */
    public boolean isExecuting() {
        return (jobState == STATE.STARTED) || (jobState == STATE.RESTARTED);
    }

    /**
     * WuX^[g\ȃWuł邩ǂ擾B
     * 
     * @return WuX^[g\ȃWułꍇ <code>true</code>
     */
    public boolean isRestartable() {
        return restartable;
    }

    /**
     * fIݒ肳Ă邩ǂ擾B
     * 
     * @return fIݒ肳Ăꍇ <code>true</code>
     */
    public boolean isShutdownGraceful() {
        return jobState == STATE.INTERRUPTED_FOR_GRACEFUL_SHUTDOWN;
    }

    /**
     * Iݒ肳Ă邩ǂ擾B
     * 
     * @return Iݒ肳Ăꍇ <code>true</code>
     */
    public boolean isShutdownImmediate() {
        return jobState == STATE.INTERRUPTED_FOR_IMMEDIATE_SHUTDOWN;
    }

    /**
     * WuN/ċN󋵁AIAfIł邩ǂ擾B
     * 
     * @return WuN/ċN󋵁AIAfIłꍇ
     * <code>true</code>
     */
    public boolean isNormallyState() {
        return isExecuting()
            || getJobState() == JobStatus.STATE.ENDING_NORMALLY
            || isShutdownGraceful();
    }

    /**
     * IR[hݒ肷B
     * 
     * @param jobExitCode IR[h
     */
    public void setJobExitCode(Integer jobExitCode) {
        this.jobExitCode = jobExitCode;
    }

    /**
     * WȕԂݒ肷B
     * 
     * @param jobState 
     */
    public void setJobState(STATE jobState) {
        this.jobState = jobState;
        if (jobState.isEndStatus()) {
            jobEndTime = System.currentTimeMillis();
        }
        if (jobState == STATE.STARTED
            || jobState == STATE.RESTARTED) {
            jobStartTime = System.currentTimeMillis();
        }
    }

    /**
     * WuX^[g\ȃWuł邩ǂݒ肷B
     * 
     * @param restartable WuX^[g\ȃWułꍇ 
     * <code>true</code>
     */
    public void setRestartable(boolean restartable) {
        this.restartable = restartable;
    }

    /**
     * X^[g|Cgݒ肷B
     * 
     * @param restartPoint X^[g|Cg
     */
    public void setRestartPoint(int restartPoint) {
        this.restartPoint = restartPoint;
    }

    /**
     * fIݒ肷B
     * 
     */
    public void shutdownGraceful() {
        if (jobState.isEndStatus()
                || jobState == STATE.INTERRUPTED_FOR_GRACEFUL_SHUTDOWN
                || jobState == STATE.INTERRUPTED_FOR_IMMEDIATE_SHUTDOWN) {
            return;
        }
        
        setJobState(STATE.INTERRUPTED_FOR_GRACEFUL_SHUTDOWN);

        if (!childJobStatusList.isEmpty()) {
            for (JobStatus result : childJobStatusList) {
                result.shutdownGraceful();
            }
        }
    }

    /**
     * Iݒ肷B
     * 
     */
    public void shutdownImmediate() {
        if (jobState.isEndStatus()
                || jobState == STATE.INTERRUPTED_FOR_IMMEDIATE_SHUTDOWN) {
            return;
        }

        setJobState(STATE.INTERRUPTED_FOR_IMMEDIATE_SHUTDOWN);

        if (!childJobStatusList.isEmpty()) {
            for (JobStatus result : childJobStatusList) {
                result.shutdownImmediate();
            }
        }
    }

    /**
     * WuID擾B
     * 
     * @return WuID
     */
    public String getJobId() {
        return jobId;
    }

    /**
     * WuIDݒ肷B
     * 
     * @param jobId WuID
     */
    public void setJobId(String jobId) {
        this.jobId = jobId;
    }

    /**
     * L[擾B
     * 
     * @return L[
     */
    public String getPartitionKey() {
        return partitionKey;
    }

    /**
     * L[ݒ肷B
     * 
     * @param partitionKey L[
     */
    public void setPartitionKey(String partitionKey) {
        this.partitionKey = partitionKey;
    }

    /**
     * ԍ擾B
     * 
     * @return ԍ
     */
    public int getPartitionNo() {
        return partitionNo;
    }

    /**
     * ԍݒ肷B
     * 
     * @param partitionNo ԍ
     */
    public void setPartitionNo(int partitionNo) {
        this.partitionNo = partitionNo;
    }

    /**
     * WuNGXgԍ擾B
     * 
     * @return WuNGXgԍ
     */
    public String getJobRequestNo() {
        return jobRequestNo;
    }

    /**
     * WuNGXgԍݒ肷B
     * 
     * @param jobRequestNo WuNGXgԍ
     */
    public void setJobRequestNo(String jobRequestNo) {
        this.jobRequestNo = jobRequestNo;
    }
}
