/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.oxm.xsd.xerces;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import jp.terasoluna.fw.oxm.xsd.SchemaValidator;
import jp.terasoluna.fw.oxm.xsd.exception.NamespaceNotFoundException;
import jp.terasoluna.fw.oxm.xsd.exception.NamespaceNotUniqueException;
import jp.terasoluna.fw.oxm.xsd.exception.ParserIOException;
import jp.terasoluna.fw.oxm.xsd.exception.ParserNotSupportedException;
import jp.terasoluna.fw.oxm.xsd.exception.ParserSAXException;
import jp.terasoluna.fw.oxm.xsd.exception.PropertyIOException;
import jp.terasoluna.fw.oxm.xsd.exception.SchemaFileNotFoundException;
import jp.terasoluna.fw.oxm.xsd.message.ErrorMessages;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xerces.impl.Constants;
import org.apache.xerces.parsers.DOMParser;
import org.apache.xerces.util.XMLGrammarPoolImpl;
import org.apache.xerces.xni.grammars.XMLGrammarPool;
import org.apache.xerces.xni.parser.XMLParserConfiguration;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;

/**
 * XMLf[^̌``FbNsSchemaValidatorNXB
 * <p>
 * ``FbNɂ̓XL[}`t@CgpB {NX̓VOgƂĎgp邱ƁB
 * </p>
 * <p>
 * XL[}`t@C͕ϊsIuWFNg̃NX pbP[WA OAgqh.xsdhŔzu邱ƁB <br>
 * jXMLϊΏۂ̃IuWFNgusample.SampleBeanvNX̏ꍇA
 * XL[}`t@C̓NXpX́usample/SampleBean.xsdv t@CƂȂB
 * </p>
 * <p>
 * XML̃p[XɂDOMp[TKvƂȂB TERASOLUNAł͏ڍׂȃG[擾邽߂ɁA g{@link jp.terasoluna.fw.oxm.xsd.xerces.XML11ConfigurationEx}A
 * {@link jp.terasoluna.fw.oxm.xsd.xerces.XMLErrorReporterEx}𗘗pB
 * </p>
 * 
 * <hr>
 * <h4>OԂ̐ݒ</h4>
 * <p>
 * {NX́AXL[}`ɖOԂgp邱ƂłB OԂgpꍇA{@link #namespace}trueݒ肷KvB
 * ftHgfalseŁAOԂgpȂݒɂȂĂB OԂgpꍇAXL[}`t@C̃LbVsƂłB
 * OԂgp邽߂ɂ́Aȉ̐ݒ肪KvłB
 * <ul>
 * <li>vpeBt@C(namespaces.properties)ɁA XMLϊΏۂ̃NXƎgp閼OԂ̑ΉtLqB
 * iڍׂ͉L̐ݒQƁj</li>
 * <li>vpeBt@C(namespaces.properties)NXpXɒuB</li>
 * <li>{NX̃CX^X𐶐inewjB</li>
 * <li>CX^X{@link #namespace}trueݒ肷B</li>
 * <li>XL[}`̃LbVݒBftHg̓LbVLłB LbV𖳌ɂꍇACX^X{@link #cache}falseɐݒ肷B</li>
 * <li>CX^X{@link #initNamespaceProperties()}\bhĂяoāA vpeBt@CǂݍށB</li>
 * </ul>
 * 
 * <br>
 * yOԂ̃vpeBt@Cinemaspace.propertiesj̐ݒz<br>
 * jXMLϊΏۂ̃IuWFNgusample.SampleBeanvNXAOԂ
 * uhttp://xxx.co.jp/sample/samplebeanv̏ꍇA vpeBt@CɉL̐ݒsB<br>
 * <br>
 * jp.terasoluna.sample2.dto.SumParam.Namespace =
 * http://xxx.co.jp/sample/samplebean
 * </p>
 * 
 * <hr>
 * <p>
 * <strong>OԂgpȂꍇ̃XL[}t@Cݒ</strong>
 * </p>
 * 
 * <p>
 * y``FbNΏۂXMLf[^z <code><pre>
 *   &lt;sample-dto&gt;
 *     &lt;user-id&gt;15&lt;/user-id&gt;
 *     &lt;user-name&gt;user1&lt;/user-name&gt;
 *     &lt;item&gt;
 *       &lt;id&gt;100&lt;/id&gt;
 *       &lt;name&gt;item1&lt;/name&gt;
 *       &lt;price&gt;1000&lt;/price&gt;
 *     &lt;/item&gt;
 *     &lt;item&gt;
 *       &lt;id&gt;101&lt;/id&gt;
 *       &lt;name&gt;item2&lt;/name&gt;
 *       &lt;price&gt;2000&lt;/price&gt;
 *     &lt;/item&gt;
 *   &lt;/sample-dto&gt;
 * </pre></code>
 * </p>
 * 
 * <p>
 * yXL[}`t@C̐ݒz<br>
 * <code><pre>
 *   &lt;?xml version=&quot;1.0&quot; encoding=&quot;UTF-8&quot;?&gt;
 *   &lt;xs:schema xmlns:xs=&quot;http://www.w3.org/2001/XMLSchema&quot;&gt;
 *     &lt;xs:element name=&quot;sample-dto&quot; type=&quot;sample-dto-type&quot;/&gt;
 *     &lt;xs:complexType name=&quot;sample-dto-type&quot;&gt;
 *       &lt;xs:sequence&gt;
 *         &lt;xs:element name=&quot;user-id&quot; type=&quot;xs:int&quot; /&gt;
 *         &lt;xs:element name=&quot;user-name&quot; type=&quot;xs:string&quot; /&gt;
 *         &lt;xs:element name=&quot;item&quot; type=&quot;item-type&quot; minOccurs=&quot;0&quot; maxOccurs=&quot;unbounded&quot; /&gt;
 *       &lt;/xs:sequence&gt;
 *     &lt;/xs:complexType&gt;
 *     &lt;xs:complexType name=&quot;item-type&quot;&gt;
 *       &lt;xs:sequence&gt;
 *          &lt;xs:element name=&quot;id&quot; type=&quot;xs:int&quot; /&gt;
 *         &lt;xs:element name=&quot;name&quot; type=&quot;xs:string&quot; /&gt;
 *         &lt;xs:element name=&quot;price&quot; type=&quot;xs:int&quot; /&gt;
 *       &lt;/xs:sequence&gt;
 *     &lt;/xs:complexType&gt;
 *   &lt;/xs:schema&gt;
 * </pre></code>
 * </p>
 * 
 * <hr>
 * <p>
 * <strong>OԂgpꍇ̃XL[}t@Cݒ</strong>
 * </p>
 * 
 * y``FbNΏۂXMLf[^Tvz
 * 
 * &lt;sample-dto xmlns=&quot;http://xxx.co.jp/sample/samplebean&quot;&gt;
 * &lt;user-id&gt;15&lt;/user-id&gt; &lt;user-name&gt;user1&lt;/user-name&gt;
 * &lt;item&gt; &lt;id&gt;100&lt;/id&gt; &lt;name&gt;item1&lt;/name&gt;
 * &lt;price&gt;1000&lt;/price&gt; &lt;/item&gt; &lt;item&gt;
 * &lt;id&gt;101&lt;/id&gt; &lt;name&gt;item2&lt;/name&gt;
 * &lt;price&gt;2000&lt;/price&gt; &lt;/item&gt; &lt;/sample-dto&gt;
 * 
 * yXL[}`t@C̐ݒTvz
 * 
 * &lt;?xml version=&quot;1.0&quot; encoding=&quot;UTF-8&quot;?&gt;
 * &lt;xs:schema xmlns:xs=&quot;http://www.w3.org/2001/XMLSchema&quot;
 * xmlns:tns=&quot;http://xxx.co.jp/sample/samplebean&quot;
 * targetNamespace=&quot;http://xxx.co.jp/sample/samplebean&quot;
 * elementFormDefault=&quot;qualified&quot;&gt; &lt;xs:element
 * name=&quot;sample-dto&quot; type=&quot;tns:sample-dto-type&quot;/&gt;
 * &lt;xs:complexType name=&quot;sample-dto-type&quot;&gt; &lt;xs:sequence&gt;
 * &lt;xs:element name=&quot;user-id&quot; type=&quot;xs:int&quot;/&gt;
 * &lt;xs:element name=&quot;user-name&quot; type=&quot;xs:string&quot;/&gt;
 * &lt;xs:element name=&quot;item&quot; type=&quot;tns:item-type&quot;
 * minOccurs=&quot;0&quot; maxOccurs=&quot;unbounded&quot;/&gt;
 * &lt;/xs:sequence&gt; &lt;/xs:complexType&gt; &lt;xs:complexType
 * name=&quot;item-type&quot;&gt; &lt;xs:sequence&gt; &lt;xs:element
 * name=&quot;id&quot; type=&quot;xs:int&quot;/&gt; &lt;xs:element
 * name=&quot;name&quot; type=&quot;xs:string&quot;/&gt; &lt;xs:element
 * name=&quot;price&quot; type=&quot;xs:int&quot;/&gt; &lt;/xs:sequence&gt;
 * &lt;/xs:complexType&gt; &lt;/xs:schema&gt;
 * 
 * @see jp.terasoluna.fw.oxm.xsd.xerces.XML11ConfigurationEx
 * @see jp.terasoluna.fw.oxm.xsd.message.ErrorMessages
 * @see jp.terasoluna.fw.oxm.xsd.xerces.XMLErrorReporterEx
 * @see jp.terasoluna.fw.web.rich.springmvc.bind.XMLServletRequestDataBinder
 * @see org.apache.xerces.util.XMLGrammarPoolImpl
 * 
 */
public class SchemaValidatorImpl implements SchemaValidator {

    /**
     * ONXB
     */
    private static Log log = LogFactory.getLog(SchemaValidatorImpl.class);

    /**
     * pbP[W̃Zp[^B
     */
    public static final String NESTED_PACKAGE_SEPARATOR = ".";

    /**
     * tH_̃Zp[^B
     */
    public static final String NESTED_FOLDER_SEPARATOR = "/";

    /**
     * XL[}`t@C̃TtBbNXB
     */
    public static final String XSD_FILE_SUFFIX = ".xsd";

    /**
     * vpeBt@Cl[Xy[X擾L[̐ڔB
     */
    private static final String NAME_SPACE_SUFFIX = ".Namespace";

    /**
     * NXƖOԂ̃}bsO`vpeBt@CB
     */
    protected String namespacePropertyFileName = "namespaces.properties";

    /**
     * XL[}`̃LbVgpݒB OԂgpꍇ̂݁ALbVsȂB
     */
    protected boolean cache = true;

    /**
     * OԂ̎gpݒB
     */
    protected boolean namespace = false;

    /**
     * OԂ̕@v[B
     */
    protected XMLGrammarPool grammarPool = new XMLGrammarPoolImpl();

    /**
     * NXƖOԂ̃}bsO`vpeBB
     */
    protected Properties namespaceProperties = null;

    /**
     * OԂ̃`FbNݒB
     */
    protected boolean namespaceCheck = true;

    /**
     * XL[}`̃LbVgpݒsB
     * 
     * @param cache
     *            XL[}`t@C̃LbVsꍇAtrue
     */
    public void setCache(boolean cache) {
        this.cache = cache;
    }

    /**
     * OԂ̎gpݒsB
     * 
     * @param namespace
     *            OԂgpꍇAtrue
     */
    public void setNamespace(boolean namespace) {
        this.namespace = namespace;
    }

    /**
     * OԂ̃`FbNݒsB
     * 
     * @param namespaceCheck
     *            OԂ`FbNꍇAtrue
     */
    public void setNamespaceCheck(boolean namespaceCheck) {
        this.namespaceCheck = namespaceCheck;
    }

    /**
     * NXƖOԂ̃}bsO`vpeBݒ肷
     * 
     * @param namespaceProperties
     *            NXƖOԂ̃}bsO`vpeB
     */
    public void setNamespaceProperties(Properties namespaceProperties) {
        this.namespaceProperties = namespaceProperties;
    }

    /**
     * NXƖOԂ̃}bsO`vpeBt@Cݒ肷B
     * 
     * @param namespacePropertyFileName
     *            NXƖOԂ̃}bsO`vpeBt@C
     */
    public void setNamespacePropertyFileName(String namespacePropertyFileName) {
        this.namespacePropertyFileName = namespacePropertyFileName;
    }

    /**
     * ɖOԂvpeBɐݒ肷B
     */
    public void initNamespaceProperties() {
        loadNamespaceProperties();

        if (this.namespaceCheck) {
            checkNamespaceProperties();
        }
    }

    /**
     * OԂ`ꂽvpeBt@Cǂݍ݁AɃZbgB
     */
    protected void loadNamespaceProperties() {
        // propertyNamenull܂͋󕶎̏ꍇAȌ̏sȂB
        if (namespacePropertyFileName == null
                || "".equals(namespacePropertyFileName)) {
            return;
        }

        // JgXbh̃ReLXgNX[_gp
        // WEB-INF/classes̃vpeBt@CǂނƂłȂꍇB
        // JNLPŃ\[X擾ɂ́ACXbh̃ReLXg
        // NX[_𗘗pȂ΂ȂȂߗ𕹗pB
        InputStream is = Thread.currentThread().getContextClassLoader()
                .getResourceAsStream(namespacePropertyFileName);
        if (is == null) {
            is = this.getClass().getResourceAsStream(
                    "/" + namespacePropertyFileName);
            if (is == null) {
                log.warn("Can not find property-file ["
                        + namespacePropertyFileName + "]");
                return;
            }
        }

        this.namespaceProperties = new Properties();

        try {
            this.namespaceProperties.load(is);
        } catch (IOException e) {
            log.error("Can not read property-file ["
                    + namespacePropertyFileName);
            throw new PropertyIOException(e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
                log.error("Failed to close inputStream.", e);
            }
        }
    }

    /**
     * OԂdĂȂ̃`FbNsB
     */
    protected void checkNamespaceProperties() {
        StringBuilder logStr = new StringBuilder();
        // namespacePropertiesnull܂͋̏ꍇAȌ̏sȂB
        if (namespaceProperties == null || namespaceProperties.isEmpty()) {
            return;
        }

        List<String> namespacePropertiesList = new ArrayList<String>();
        for (Object namespaceKey : namespaceProperties.keySet()) {
            String namespaceValue = namespaceProperties
                    .getProperty((String) namespaceKey);
            // OԂdĂꍇAG[Ȍo͂sAOX[B
            if (namespacePropertiesList.contains(namespaceValue)) {
                logStr.setLength(0);
                logStr.append("Namespace name [");
                logStr.append(namespaceValue);
                logStr.append("] is not unique. ");
                logStr.append("Namespace must be unique. ");
                logStr.append("(key = [");
                logStr.append(namespaceKey);
                logStr.append("])");
                log.error(logStr.toString());
                throw new NamespaceNotUniqueException();
            }
            namespacePropertiesList.add(namespaceValue);
        }
    }

    /**
     * XMLf[^̌``FbNs\bhB
     * <p>
     * XMLf[^DOMc[ɕϊۂɁAXMLXL[}ɂ ``FbNiÓ؁jsB<br>
     * ``FbNɂ́AXL[}`t@CpB
     * </p>
     * <p>
     * Ɍ``FbNIꍇADOMc[ԋpB<br>
     * [U̓͂ƍlf[^^̃G[ꍇAerrorMessagesɃG[i[āAnullԋpB<br>
     * ȊÕG[ɊւẮAOXMappingExceptioñTuNXɃbvăX[B
     * </p>
     * 
     * @param in
     *            XMLf[^
     * @param object
     *            ϊΏۂ̃IuWFNg
     * @param errorMessages
     *            G[bZ[Wi[CX^X
     * @return DOMc[
     */
    public Document validate(InputStream in, Object object,
            ErrorMessages errorMessages) {
        if (in == null) {
            log.error("InputStream is null.");
            throw new IllegalArgumentException("InputStream is null.");
        }
        if (errorMessages == null) {
            log.error("ErrorMessages is null.");
            throw new IllegalArgumentException("ErrorMessages is null.");
        }

        // DOMp[T̍쐬
        DOMParser parser = null;
        try {
            parser = createDomParser(object);
            setCommonParserProperty(parser, errorMessages);
            setCommonParserFeature(parser);
        } catch (SAXNotRecognizedException e) {
            // FłȂL[ݒ肳ꂽꍇɃX[O
            log.error("Schema property error.", e);
            throw new ParserNotSupportedException(e);
        } catch (SAXNotSupportedException e) {
            // T|[gĂȂlݒ肳ꂽꍇɃX[O
            log.error("Schema property error.", e);
            throw new ParserNotSupportedException(e);
        }

        // p[X
        try {
            parser.parse(new InputSource(in));
        } catch (SAXException e) {
            log.error("Schema parse error.", e);
            throw new ParserSAXException(e);
        } catch (IOException e) {
            log.error("Schema io error.", e);
            throw new ParserIOException(e);
        }

        if (errorMessages.hasErrorMessage()) {
            return null;
        }
        return parser.getDocument();
    }

    /**
     * DOMp[T𐶐B
     * 
     * @param object
     *            p[XΏۂ̃IuWFNg
     * @return DOMp[T
     * @throws SAXNotSupportedException
     *             FłȂL[ݒ肳ꂽꍇɃX[O
     * @throws SAXNotRecognizedException
     *             T|[gĂȂlݒ肳ꂽꍇɃX[O
     */
    protected DOMParser createDomParser(Object object)
            throws SAXNotRecognizedException, SAXNotSupportedException {
        DOMParser parser = new DOMParser(createXmlParserConfiguration());

        // XL[}`t@CURL
        URL schemaURL = getUrl(object);
        if (schemaURL == null) {
            log.error("Schema file is not found. Set schema file in "
                    + "[root-classpath]/" + getSchemaFilePath(object));
            throw new SchemaFileNotFoundException();
        }

        // XL[}`t@C̃P[Vݒ
        if (namespace) {
            StringBuilder key = new StringBuilder();
            key.append(Constants.XERCES_PROPERTY_PREFIX);
            key.append(Constants.SCHEMA_LOCATION);

            StringBuilder location = new StringBuilder();
            location.append(getNamespaceName(object));
            location.append(" ");
            location.append(schemaURL.toExternalForm());

            parser.setProperty(key.toString(), location.toString());
        } else {
            parser.setProperty(Constants.XERCES_PROPERTY_PREFIX
                    + Constants.SCHEMA_NONS_LOCATION, schemaURL
                    .toExternalForm());
        }
        return parser;
    }

    /**
     * XMLParserConfiguration𐶐B OԂƃLbV̎gpLɂĂꍇA
     * XL[}t@C̃LbV𗘗pB
     * 
     * @return XMLParserConfiguration XMLp[T̐ݒێIuWFNg
     */
    protected XMLParserConfiguration createXmlParserConfiguration() {
        if (namespace && cache) {
            // OԂ̕@v[ݒ肷邱ƂŃLbVLɂ
            return new XML11ConfigurationEx(grammarPool);
        }
        return new XML11ConfigurationEx();
    }

    /**
     * p[TʂPropertyݒ肷B
     * 
     * @param parser
     *            DOMp[T
     * @param errorMessages
     *            G[
     * @throws SAXNotRecognizedException
     *             FłȂL[ݒ肳ꂽꍇɃX[O
     * @throws SAXNotSupportedException
     *             T|[gĂȂlݒ肳ꂽꍇɃX[O
     */
    protected void setCommonParserProperty(DOMParser parser,
            ErrorMessages errorMessages) throws SAXNotRecognizedException,
            SAXNotSupportedException {
        // p[XɔG[CX^X
        parser.setProperty(Constants.XERCES_PROPERTY_PREFIX
                + Constants.ERROR_REPORTER_PROPERTY, new XMLErrorReporterEx(
                errorMessages));
    }

    /**
     * p[TʂFeatureݒ肷
     * 
     * @param parser
     *            DOMp[T
     * @throws SAXNotRecognizedException
     *             FłȂL[ݒ肳ꂽꍇɃX[O
     * @throws SAXNotSupportedException
     *             T|[gĂȂlݒ肳ꂽꍇɃX[O
     */
    protected void setCommonParserFeature(DOMParser parser)
            throws SAXNotRecognizedException, SAXNotSupportedException {
        // ׂĂ̑ÓG[ʒm
        parser.setFeature(Constants.SAX_FEATURE_PREFIX
                + Constants.VALIDATION_FEATURE, true);

        // ``FbNɃXL[}`t@Cgpݒ
        parser.setFeature(Constants.XERCES_FEATURE_PREFIX
                + Constants.SCHEMA_VALIDATION_FEATURE, true);
    }

    /**
     * URL擾B
     * <p>
     * ̃IuWFNgƓ̃NXpXA`t@C擾
     * </p>
     * 
     * @param object
     *            IuWFNg
     * @return \[XURLCX^X
     */
    protected URL getUrl(Object object) {
        return Thread.currentThread().getContextClassLoader().getResource(
                getSchemaFilePath(object));
    }

    /**
     * XL[}t@C̃pX擾B
     * 
     * @param object
     *            XL[}`FbNΏۂ̃IuWFNg
     * @return XL[}t@C̃pX
     */
    protected String getSchemaFilePath(Object object) {
        if (object == null) {
            log.error("Argument is null.");
            throw new IllegalArgumentException("Argument is null.");
        }

        StringBuilder retStr = new StringBuilder();
        retStr.append(object.getClass().getName().replace(
                NESTED_PACKAGE_SEPARATOR, NESTED_FOLDER_SEPARATOR));
        retStr.append(XSD_FILE_SUFFIX);
        return retStr.toString();
    }

    /**
     * vpeBt@C疼OԂ擾B OԂgpȂꍇAnullԂB KvȖOԂݒ肳ĂȂꍇAsO𓊂B
     * 
     * @param object
     *            IuWFNg
     * @return \[XURLCX^X
     */
    protected String getNamespaceName(Object object) {
        if (object == null) {
            log.error("Argument is null.");
            throw new IllegalArgumentException("Argument is null.");
        }

        // OԂgȂ
        if (!namespace) {
            return null;
        }

        // OԂ`t@CȂ
        if (this.namespaceProperties == null) {
            String message = "Namespace property is not set. " + "Put "
                    + namespacePropertyFileName + " file on your classpath, "
                    + "and call SchemaValidatorImpl initNamespaceProperties() "
                    + "after instanciate.";
            log.error(message);
            throw new IllegalStateException(message);
        }

        // OԂ擾
        StringBuilder namespaceKey = new StringBuilder(object.getClass()
                .getName());
        namespaceKey.append(NAME_SPACE_SUFFIX);
        String namespaceName = namespaceProperties.getProperty(namespaceKey
                .toString());

        if (namespaceName == null) {
            log.error("Schema namespace is not found. Set namespace key - "
                    + namespaceKey.toString() + " in "
                    + namespacePropertyFileName + " file.");
            throw new NamespaceNotFoundException();
        }
        return namespaceName;
    }
}
