﻿// <著作権>
// Copyright (c) 2007, NTT DATA Corporation.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using NTTDATA.TERASOLUNA.FW.Web;
using NTTDATA.TERASOLUNA.FW.Web.Controller.Multipart;

namespace MultipartUploadUT
{
    ///<summary>
    /// MultipartFileItem のテストクラスです。
    ///</summary>
    ///<author>
    /// 串野亮
    ///</author>
    [TestClass()]
    public class MultipartItemTest
    {
        private TestContext testContextInstance;

        private static readonly string FILE_PATH
            = @"MultipartItem.txt";

        private static readonly string BOUNDARY = "boundary";

        private static readonly Encoding ENCODING = new UTF8Encoding();

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        //[TestInitialize()]
        //public void MyTestInitialize()
        //{
        //}
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数が全て正常な値の時、正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest01.txt";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=" + name);
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor accessor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor(target);

                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; name=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(ENCODING, accessor._encoding);
                
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数boundaryがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string boundary = null;
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=ConstructorTest02.txt");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(boundary, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentNullException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数boundaryが空文字の時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string boundary = "";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=ConstructorTest03");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(boundary, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数partStreamがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest04()
        {
            // 準備
            IDictionary<string, string> headerList = new Dictionary<string, string>();
            headerList.Add(
                "CONTENT-DISPOSITION", "form-data; name=ConstructorTest03");
            headerList.Add("content-type", "application/x-www-form-urlencoded");
            headerList.Add("content-part", "MultipartItem");
            Stream pastStream = null;

            try
            {
                // 失敗
                new MultipartItemStub01(BOUNDARY, pastStream, headerList, ENCODING);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException)
            {
                // 成功
            }
            catch (Exception ex)
            {
                // 失敗
                Assert.Fail("予期せぬエラーです。" + ex.ToString());
            }
        }


        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest05()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList = null;

                try
                {
                    // 失敗
                    new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentNullException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }


        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素数が0の時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest06()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素数が1の時、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest07()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest07";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=ConstructorTest07");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor accessor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor(target);

                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(1, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; name=ConstructorTest07",
                    accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreSame(ENCODING, accessor._encoding);
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素に"Content-Disposition"が存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest08()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT", "form-data; name=ConstructorTest08");
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数headerListの要素に"Cotent-Disposition"(大文字)が存在し、nameが存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest09()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data;");
                headerList.Add(
                     "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値"name"(小文字+大文字)が定義されている時、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest10()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest10.txt";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; Name=" + name);
                headerList.Add(
                     "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor accessor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor(target);


                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; Name=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(ENCODING, accessor._encoding);
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値"name"(大文字)が定義されている時、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest11()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest10.txt";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; NAME=" + name);
                headerList.Add(
                     "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor accessor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor(target);

                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; NAME=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(ENCODING, accessor._encoding);
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値は"name"を含む文字列で定義されている時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest12()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; names=ConstructorTest11");
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)の値"name"が２つ定義されている時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest13()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION",
                    "form-data; name=ConstructorTest12; name=ConstructorTest12");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(
                        BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// "Content-Disposition"(大文字)を含む別の文字列で定義されている時、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest14()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITIONS", "form-data; name=ConstructorTest13;");
                headerList.Add(
                    "content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");

                try
                {
                    // 失敗
                    new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }


        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数encodingがnullの時、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest15()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=ConstructorTest14");
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");
                Encoding encoding = null;

                try
                {
                    // 失敗
                    new MultipartItemStub01(BOUNDARY, partStream, headerList, encoding);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (ArgumentNullException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// MultipartItem (string, Stream, IDictionary&lt;string,string&gt;, Encoding) のテスト
        ///</summary>
        ///<remarks>
        /// 引数encodingがShift-JISの時、正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void ConstructorTest16()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                string name = "ConstructorTest15";
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=" + name);
                headerList.Add("content-type", "application/x-www-form-urlencoded");
                headerList.Add("content-part", "MultipartItem");
                Encoding encoding = Encoding.GetEncoding("SHIFT-JIS");

                // 実行
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, encoding);
                NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor accessor =
                    new NTTDATA_TERASOLUNA_FW_Web_Multipart_MultipartItemAccessor(target);

                // 検証
                Assert.AreEqual(name, accessor._name);
                Assert.AreSame(headerList, accessor._headerList);
                Assert.AreEqual(3, accessor._headerList.Count);
                Assert.AreEqual(
                    "form-data; name=" + name, accessor._headerList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", accessor._headerList["content-type"]);
                Assert.AreEqual("MultipartItem", accessor._headerList["content-part"]);
                Assert.AreSame(encoding, accessor._encoding);
            }
        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListにnullを設定し、
        /// HeaderListでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void HeaderListTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target = new MultipartItemStub01(
                    BOUNDARY, partStream, headerList, ENCODING);
                target.HeaderList = null;

                // 実行 & 検証
                Assert.IsNull(target.HeaderList);
            }
        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListに要素数0を設定し、
        /// HeaderListで要素数0を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void HeaderListTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                headerList = new Dictionary<string, string>();
                target.HeaderList = headerList;

                // 実行 & 検証
                Assert.AreSame(headerList, target.HeaderList);
                Assert.AreEqual(0, target.HeaderList.Count);
            }
        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListに要素数1を設定し、
        /// HeaderListで要素数1を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void HeaderListTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                headerList = new Dictionary<string, string>();
                headerList.Add("key01", "value01");
                target.HeaderList = headerList;

                // 実行 & 検証
                Assert.AreSame(headerList, target.HeaderList);
                Assert.AreEqual(1, target.HeaderList.Count);
                Assert.AreEqual("value01", target.HeaderList["key01"]);
            }
        }

        ///<summary>
        /// HeaderList のテスト
        ///</summary>
        ///<remarks>
        /// HeaderListに要素数3を設定し、
        /// HeaderListで要素数3を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void HeaderListTest04()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=HeaderList");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                headerList = new Dictionary<string, string>();
                headerList.Add("key01", "value01");
                headerList.Add("key02", "value02");
                headerList.Add("key03", "value03");
                target.HeaderList = headerList;

                // 実行 & 検証
                Assert.AreSame(headerList, target.HeaderList);
                Assert.AreEqual(3, target.HeaderList.Count);
                Assert.AreEqual("value01", target.HeaderList["key01"]);
                Assert.AreEqual("value02", target.HeaderList["key02"]);
                Assert.AreEqual("value03", target.HeaderList["key03"]);
            }
        }

        ///<summary>
        /// Name のテスト
        ///</summary>
        ///<remarks>
        /// Nameにnullを設定し、
        /// Nameでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void NameTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Name");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Name = null;

                // 実行 & 検証
                Assert.IsNull(target.Name);
            }
        }

        ///<summary>
        /// Name のテスト
        ///</summary>
        ///<remarks>
        /// Nameに空文字を設定し、
        /// Nameで空文字を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void NameTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Name");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Name = "";

                // 実行 & 検証
                Assert.AreEqual("", target.Name);
            }
        }

        ///<summary>
        /// Name のテスト
        ///</summary>
        ///<remarks>
        /// Nameに任意の文字列"abcde"を設定し、
        /// Nameで任意の文字列"abcde"を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void NameTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Name");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Name = "abcde";

                // 実行 & 検証
                Assert.AreEqual("abcde", target.Name);
            }
        }

        ///<summary>
        /// Encoding のテスト
        ///</summary>
        ///<remarks>
        /// Encodingにnullを設定し、
        /// Encodingでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void EncodingTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Encoding");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Encoding = null;

                // 実行 & 検証
                Assert.IsNull(target.Encoding);
            }
        }

        ///<summary>
        /// Encoding のテスト
        ///</summary>
        ///<remarks>
        /// EncodingにUTF-8を設定し、
        /// EncodingでUTF-8を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void EncodingTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Encoding");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Encoding = new UTF8Encoding();

                // 実行 & 検証
                Assert.AreEqual(new UTF8Encoding(), target.Encoding);
            }
        }

        ///<summary>
        /// Encoding のテスト
        ///</summary>
        ///<remarks>
        /// EncodingにShift-JISを設定し、
        /// EncodingでShift-JISを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void EncodingTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Encoding");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.Encoding = Encoding.GetEncoding("SHIFT-JIS");

                // 実行 & 検証
                Assert.AreEqual(Encoding.GetEncoding("SHIFT-JIS"), target.Encoding);
            }
        }

        ///<summary>
        /// PositionStart のテスト
        ///</summary>
        ///<remarks>
        /// PositionStartに0を設定し、
        /// PositionStartで0を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void PositionStartTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionStart");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionStart = 0;

                // 実行 & 検証
                long num = 0;
                Assert.AreEqual(num, target.PositionStart);
            }
        }

        ///<summary>
        /// PositionStart のテスト
        ///</summary>
        ///<remarks>
        /// PositionStartに100を設定し、
        /// PositionStartで100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void PositionStartTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionStart");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionStart = 100;

                // 実行 & 検証
                long num = 100;
                Assert.AreEqual(num, target.PositionStart);
            }
        }

        ///<summary>
        /// PositionStart のテスト
        ///</summary>
        ///<remarks>
        /// PositionStartに-100を設定し、
        /// PositionStartで-100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void PositionStartTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionStart");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionStart = -100;

                // 実行 & 検証
                long num = -100;
                Assert.AreEqual(num, target.PositionStart);
            }
        }

        ///<summary>
        /// PositionEnd のテスト
        ///</summary>
        ///<remarks>
        /// PositionEndに0を設定し、
        /// PositionEndで0を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void PositionEndTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionEnd");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionEnd = 0;

                // 実行 & 検証
                long num = 0;
                Assert.AreEqual(num, target.PositionEnd);
            }
        }

        ///<summary>
        /// PositionEnd のテスト
        ///</summary>
        ///<remarks>
        /// PositionEndに100を設定し、
        /// PositionEndで100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void PositionEndTest02()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionEnd");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionEnd = 100;

                // 実行 & 検証
                long num = 100;
                Assert.AreEqual(num, target.PositionEnd);
            }
        }

        ///<summary>
        /// PositionEnd のテスト
        ///</summary>
        ///<remarks>
        /// PositionEndに-100を設定し、
        /// PositionEndで-100を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void PositionEndTest03()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=PositionEnd");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);
                target.SetPositionEnd = -100;

                // 実行 & 検証
                long num = -100;
                Assert.AreEqual(num, target.PositionEnd);
            }
        }

        ///<summary>
        /// Filename のテスト
        ///</summary>
        ///<remarks>
        /// Filenameにnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void FilenameTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList = new Dictionary<string, string>();
                headerList.Add("CONTENT-DISPOSITION", "form-data; name=Filename");
                MultipartItemStub01 target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);

                // 実行 & 検証
                Assert.IsNull(target.Filename);
            }
        }

        ///<summary>
        /// OutputStream のテスト
        ///</summary>
        ///<remarks>
        /// OutputStreamにnullを設定し、
        /// OutputStreamでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void OutputStreamTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);

                // 試験実施及び判定
                Assert.IsNull(target.OutputStream);
            }
        }

        ///<summary>
        /// Text のテスト
        ///</summary>
        ///<remarks>
        /// Textにnullを設定し、
        /// Textでnullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItem.txt")]
        public void TextTest01()
        {
            using (Stream partStream = new FileStream(FILE_PATH, FileMode.Open, FileAccess.Read))
            {
                // 準備
                IDictionary<string, string> headerList
                    = new Dictionary<string, string>();
                headerList.Add(
                    "CONTENT-DISPOSITION", "form-data; name=file1;filename=text.txt");
                MultipartItem target
                    = new MultipartItemStub01(BOUNDARY, partStream, headerList, ENCODING);

                // 試験実施及び判定
                Assert.IsNull(target.Text);
            }
        }
    }
}
