﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.  

using System;
using System.IO;
using System.Web;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Web.Controller
{
    /// <summary>
    /// <seealso cref="HttpContext"/> クラスをラップして保持するためのクラスです。
    /// </summary>
    public class HttpContextWrapper
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(HttpContextWrapper));

        /// <summary>
        /// マルチパートデータのヘッダに Content-Type ヘッダを取得する際、キーとして利用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>定数の値は "CONTENT-TYPE" です。</para>
        /// </remarks>
        protected static readonly string CONTENT_TYPE = "CONTENT-TYPE";

        /// <summary>
        /// 提供された <seealso cref="HttpContext"/> です。
        /// </summary>
        private HttpContext _httpContext = null;

        /// <summary>
        /// <see cref="HttpContextWrapper"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        protected HttpContextWrapper()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="httpContext">提供された <seealso cref="HttpContext"/> です。</param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="httpContext"/> が null 参照です。
        /// </exception>
        public HttpContextWrapper(HttpContext httpContext)
        {
            if (httpContext == null)
            {
                ArgumentNullException exception = new ArgumentNullException("httpContext");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "httpContext"), exception);
                }
                throw exception;
            }

            _httpContext = httpContext;
        }

        /// <summary>
        /// 提供された <seealso cref="HttpContext"/> から取り出した <seealso cref="Stream"/> を取得します。
        /// </summary>
        /// <value>
        /// 提供された <seealso cref="HttpContext"/> から取り出した <seealso cref="Stream"/> 。
        /// </value>
        public virtual Stream RequestStream
        {
            get
            {
                return _httpContext.Request.InputStream;
            }
        }

        /// <summary>
        /// コンストラクタで設定した <seealso cref="HttpContext"/> の <seealso cref="HttpRequest.Headers"/> から、キー
        /// <seealso cref="CONTENT_TYPE"/> の値を取得します。
        /// </summary>
        /// <value>
        /// コンストラクタで設定した <seealso cref="HttpContext"/> の <seealso cref="HttpRequest.Headers"/> から、キー
        /// <seealso cref="CONTENT_TYPE"/> の値。
        /// </value>
        public virtual string ContentType
        {
            get
            {
                return _httpContext.Request.Headers[CONTENT_TYPE];
            }
        }
    }
}
