﻿// <著作権>
// Copyright (c) 2008, NTT DATA Corporation.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Globalization;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Configuration;
using System.Resources;
using TERASOLUNA.Fw.CommonUT.MessageManagerTestData;
using System.Threading;

namespace TERASOLUNA.Fw.CommonUT
{
    /// <summary>
    ///TERASOLUNA.Fw.Common.MessageManager のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Common.MessageManager 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class MessageManagerTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}

        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            ConfigurationManager.AppSettings["MessageResource.Common"] = typeof(MessageManagerResources01).AssemblyQualifiedName;
            TERASOLUNA_Fw_Common_MessageManagerAccessor._instance = null;
        }

        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用

        [TestCleanup()]
        public void MyTestCleanup()
        {
            ConfigurationManager.AppSettings["MessageResource.Common"] = "TERASOLUNA.Fw.Common.Properties.Resources, TERASOLUNA.Fw.Common";
            ConfigurationManager.AppSettings["MessageManagerTypeName"] = string.Empty;
            TERASOLUNA_Fw_Common_MessageManagerAccessor._instance = null;
            TERASOLUNA_Fw_Common_MessageManagerAccessor.MESSAGE_MANAGER_TYPE_NAME = "MessageManagerTypeName";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.RESOURCE_NAME_PREFIX = "MessageResource.";
        }
        
        #endregion

        [TestMethod()]
        public void CultureTest01()
        {
            // 準備
            CultureInfo expectedCultureInfo = new CultureInfo("ja");

            // 実施
            MessageManager.Culture = expectedCultureInfo;
            CultureInfo actualCultureInfo = MessageManager.Culture;

            Assert.AreSame(expectedCultureInfo, actualCultureInfo);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null の場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest01()
        {
            string messageId = null;

            try
            {
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                //成功
                Assert.AreEqual("messageId", ex.ParamName);
            }
            catch (Exception ex)
            {
                Assert.Fail("到達してはいけないコードです。" + ex.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MessageManager),typeof(ArgumentNullException), "引数 \"messageId\" は null にすることはできません。"));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が空文字列の場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest02()
        {
            string messageId = string.Empty;

            try
            {
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException ex)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, ex.GetType().Name);
                Assert.AreEqual("引数 \"messageId\" の値を空文字列にすることはできません。", ex.Message);
            }
            catch (Exception ex)
            {
                Assert.Fail("到達してはいけないコードです。" + ex.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MessageManager), typeof(ArgumentException), "引数 \"messageId\" の値を空文字列にすることはできません。"));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列でない場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest03()
        {
            // 準備
            string messageId = "TERASOLUNA";

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 結果
            string expectedMessage = MessageManagerResources01.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 引数の args が null の場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest04()
        {
            string messageId = "TERASOLUNA";

            try
            {
                MessageManager.GetMessage(messageId, null);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                //成功
                Assert.AreEqual("args", ex.ParamName);
            }
            catch (Exception ex)
            {
                Assert.Fail("到達してはいけないコードです。" + ex.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MessageManager), typeof(ArgumentNullException), "引数 \"args\" は null にすることはできません。"));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 引数の args の要素数が 1 の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest05()
        {
            // 準備
            string messageId = "TERASOLUNA";
            object[] args = { "args01" };

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId, args);

            // 結果
            string expectedMessage = MessageManagerResources01.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 引数の args の要素数が 3 の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest06()
        {
            // 準備
            string messageId = "TERASOLUNA";
            object[] args = { "args01", "args02", "args03" };

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId, args);

            // 結果
            string expectedMessage = MessageManagerResources01.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// MessageManager のプロパティ Culture が null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest07()
        {
            // 準備
            string messageId = "TERASOLUNA";
            MessageManager.Culture = null;

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 結果
            string expectedMessage = MessageManagerResources01.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(accessor.ResourceManagerList[0].BaseName, typeof(MessageManagerResources01).Name);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// MessageManager のプロパティ Culture が 英語 の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest08()
        {
            // 準備
            string messageId = "TERASOLUNA";
            MessageManager.Culture = new CultureInfo("en");

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 結果
            string expectedMessage = MessageManagerResources01.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// MessageManager のプロパティ Instance が not null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest09()
        {
            // 準備
            string messageId = "TERASOLUNA";
            MessageManager cacheManager = new MessageManager();
            TERASOLUNA_Fw_Common_MessageManagerAccessor cacheAccessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(cacheManager);
            cacheAccessor.ResourceManagerList.Add(new ResourceManager(typeof(MessageManagerResources02)));
            TERASOLUNA_Fw_Common_MessageManagerAccessor._instance = cacheManager;

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 結果
            string expectedMessage = MessageManagerResources02.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor actualAccessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.AreSame(cacheManager, TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(1, actualAccessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources02).Name, actualAccessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 構成ファイルの add 要素のキー "MessageManagerTypeName" の値が空文字列の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest11()
        {
            // 準備
            string messageId = "TERASOLUNA";
            string messageTypeName = "MessageManager_GetMessageTest11";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.MESSAGE_MANAGER_TYPE_NAME = messageTypeName;
            ConfigurationManager.AppSettings[messageTypeName] = string.Empty;

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 結果
            string expectedMessage = MessageManagerResources01.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 構成ファイルの add 要素のキー "MessageManagerTypeName" の値が
        /// MessageManager派生クラスの型ではない場合、
        /// TerasolunaException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest13()
        {
            // 準備
            string messageId = "TERASOLUNA";
            string messageTypeName = "MessageManager_GetMessageTest13";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.MESSAGE_MANAGER_TYPE_NAME = messageTypeName;
            ConfigurationManager.AppSettings[messageTypeName] = typeof(Object).AssemblyQualifiedName;

            try
            {
                // 実施
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException ex)
            {
                // 確認
                string expectedMessage = string.Format("型 \"{0}\" をジェネリックパラメータで指定された型 \"{1}\" にキャストできませんでした。", typeof(Object).AssemblyQualifiedName, typeof(MessageManager).AssemblyQualifiedName);
                Assert.AreEqual(expectedMessage, ex.Message);
            }
            catch (Exception ex)
            {
                Assert.Fail("到達してはいけないコードです。" + ex.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 構成ファイルの add 要素のキー "MessageManagerTypeName" の値が
        /// MessageManager派生クラスの型である場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest14()
        {
            // 準備
            string messageId = "TERASOLUNA";
            string messageTypeName = "MessageManager_GetMessageTest14";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.MESSAGE_MANAGER_TYPE_NAME = messageTypeName;
            ConfigurationManager.AppSettings[messageTypeName] = typeof(MessageManagerImpl).AssemblyQualifiedName;

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 結果
            string expectedMessage = MessageManagerResources01.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsInstanceOfType(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance, typeof(MessageManagerImpl));
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 構成ファイルの キー "MessageResource." をプレフィックスとするadd要素数が 0 の場合、
        /// InvalidOperationException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest15()
        {
            // 準備
            string messageId = "TERASOLUNA";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.RESOURCE_NAME_PREFIX = "Test15";

            try
            {
                // 実施
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 確認
                Assert.AreEqual("アプリケーション構成ファイルの appSettings 要素に \"MessageResource.\" をプレフィックスとしたキーが見つかりません。", e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MessageManager), typeof(InvalidOperationException), "アプリケーション構成ファイルの appSettings 要素に \"MessageResource.\" をプレフィックスとしたキーが見つかりません。"));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 構成ファイルの キー "MessageResource." をプレフィックスとするadd要素数が 3 の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest16()
        {
            // 準備
            string messageId = "TERASOLUNA";
            string resourceNamePrefix = "MessageManager_GetMesssageTest16";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.RESOURCE_NAME_PREFIX = resourceNamePrefix;
            ConfigurationManager.AppSettings[resourceNamePrefix + "Client"] = typeof(MessageManagerResources03).AssemblyQualifiedName;
            ConfigurationManager.AppSettings[resourceNamePrefix + "Common"] = typeof(MessageManagerResources02).AssemblyQualifiedName;
            ConfigurationManager.AppSettings[resourceNamePrefix + "Web"] = typeof(MessageManagerResources01).AssemblyQualifiedName;

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 結果
            string expectedMessage = MessageManagerResources03.TERASOLUNA;
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(3, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources03).Name, accessor.ResourceManagerList[0].BaseName);
            Assert.AreEqual(typeof(MessageManagerResources02).Name, accessor.ResourceManagerList[1].BaseName);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[2].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 構成ファイルの キー "MessageResource.Common" の値が空文字列の場合、
        /// TerasolunaException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest17()
        {
            // 準備
            string messageId = "TERASOLUNA";
            ConfigurationManager.AppSettings["MessageResource.Common"] = string.Empty;

            try
            {
                // 実施
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual("アプリケーション構成ファイルの appSettings 要素にキー \"MessageResource.Common\" で定義された型名が空の文字列です。", e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MessageManager), typeof(TerasolunaException), "アプリケーション構成ファイルの appSettings 要素にキー \"MessageResource.Common\" で定義された型名が空の文字列です。"));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// 構成ファイルの キー "MessageResource.Common" の値が型を取得することができない文字列の場合、
        /// TerasolunaException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest18()
        {
            // 準備
            string messageId = "TERASOLUNA";
            ConfigurationManager.AppSettings["MessageResource.Common"] = "NotExistsMessageManagerTypeName";

            try
            {
                // 実施
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual("アセンブリ修飾名 \"NotExistsMessageManagerTypeName\" の型を取得できませんでした。", e.Message);
            }
            catch (Exception ex)
            {
                Assert.Fail("到達してはいけないコードです。" + ex.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// ResourceManagerListに保持しているResourceManagerが
        /// 使用できるリソースのセットを見つけることができない場合、
        /// TerasolunaException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest19()
        {
            // 準備
            string messageId = "TERASOLUNA";
            string resourceNamePrefix = "MessageManager_GetMessageTest19";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.RESOURCE_NAME_PREFIX = resourceNamePrefix;
            ConfigurationManager.AppSettings[resourceNamePrefix] = typeof(Object).AssemblyQualifiedName;

            try
            {
                // 実施
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual("リソースファイルのルート名 \"Object\" として使用できるリソースのセットが見つからず、ニュートラルカルチャリソースもありません。構成ファイルの、add要素 \"MessageResource.\" で始まるキーにリソースではないクラスを定義している可能性があります。", e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MessageManager), typeof(TerasolunaException), "リソースファイルのルート名 \"Object\" として使用できるリソースのセットが見つからず、ニュートラルカルチャリソースもありません。構成ファイルの、add要素 \"MessageResource.\" で始まるキーにリソースではないクラスを定義している可能性があります。"));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// ResourceManagerListに保持しているResourceManagerから
        /// 取得する型が文字列以外の場合、
        /// TerasolunaException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest20()
        {
            // 準備
            string messageId = "IMAGE_TERASOLUNA";

            try
            {
                // 実施
                MessageManager.GetMessage(messageId);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual("指定されたメッセージ ID \"IMAGE_TERASOLUNA\"　、カルチャ \"" + CultureInfo.CurrentCulture + "\" のリソースの値が String ではありません。", e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(MessageManager), typeof(TerasolunaException), "指定されたメッセージ ID \"IMAGE_TERASOLUNA\"　、カルチャ \""+ CultureInfo.CurrentCulture + "\" のリソースの値が String ではありません。"));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// リソースから取得する文字列が空文字列の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest21()
        {
            // 準備
            string messageId = "EMPTY";

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 確認
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.AreEqual(string.Empty, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列、かつ引数 args の要素数が1つであり、
        /// リソースの結果にプレースホルダ有り文字列（プレースホルダ1つ）の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest22()
        {
            // 準備
            string messageId = "TERASOLUNA_PRACE_HOLDER_1";
            object[] args = { "TERASOLUNA" };

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId, args);

            // 確認
            string expectedMessage = string.Format(MessageManagerResources01.TERASOLUNA_PRACE_HOLDER_1, "TERASOLUNA");
            Assert.AreEqual(expectedMessage, actualMessage);
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列、かつ引数 args の要素数が3つであり、
        /// リソースの結果にプレースホルダ有り文字列（プレースホルダ3つ）の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest23()
        {
            // 準備
            string messageId = "TERASOLUNA_PRACE_HOLDER_3";
            object[] args = { "TERASOLUNA", "for", ".NET" };

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId, args);

            // 確認
            string expectedMessage = string.Format(MessageManagerResources01.TERASOLUNA_PRACE_HOLDER_3, "TERASOLUNA", "for", ".NET");
            Assert.AreEqual(expectedMessage, actualMessage);
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列、かつ引数 args の要素数が0であり、
        /// リソースの結果にプレースホルダ有り文字列（プレースホルダ3つ）の場合、
        /// プレースホルダを含んだ文字列ごと取得できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest24()
        {
            // 準備
            string messageId = "TERASOLUNA_PRACE_HOLDER_3";

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 確認
            string expectedMessage = MessageManagerResources01.TERASOLUNA_PRACE_HOLDER_3;
            Assert.AreEqual(expectedMessage, actualMessage);
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// Culture が null の場合、
        /// デフォルトのカルチャのリソースを取得し、動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"ja\TERASOLUNA.Fw.CommonUT.resources.dll",@"ja")]
        public void GetMessageTest25()
        {
            // 準備
            string messageId = "TERASOLUNA";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.RESOURCE_NAME_PREFIX = "MessageTest24.";
            MessageManager.Culture = null;

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 確認
            string expectedMessage = "てらそるな";
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources04).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId が null ･ 空文字列ではなく、
        /// Cultureが英語の場合、
        /// 指定したカルチャのリソースを取得し、動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"en\TERASOLUNA.Fw.CommonUT.resources.dll", @"en")]
        [DeploymentItem(@"en-US\TERASOLUNA.Fw.CommonUT.resources.dll", @"en-US")]
        public void GetMessageTest26()
        {
            // 準備
            string messageId = "TERASOLUNA";
            TERASOLUNA_Fw_Common_MessageManagerAccessor.RESOURCE_NAME_PREFIX = "MessageTest25.";
            MessageManager.Culture = new CultureInfo("en");

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 確認
            string expectedMessage = "terasoluna";
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(expectedMessage, actualMessage);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources05).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の messageId で指定したキーが存在しない場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest27()
        {
            // 準備
            string messageId = "Framework";

            // 実施
            string actualMessage = MessageManager.GetMessage(messageId);

            // 確認
            Assert.IsNull(actualMessage);
            TERASOLUNA_Fw_Common_MessageManagerAccessor accessor = new TERASOLUNA_Fw_Common_MessageManagerAccessor(TERASOLUNA_Fw_Common_MessageManagerAccessor.Instance);
            Assert.IsNotNull(TERASOLUNA_Fw_Common_MessageManagerAccessor._instance);
            Assert.AreEqual(1, accessor.ResourceManagerList.Count);
            Assert.AreEqual(typeof(MessageManagerResources01).Name, accessor.ResourceManagerList[0].BaseName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }

        /// <summary>
        ///GetMessage (string, params object[]) のテスト
        ///</summary>
        /// <remarks>
        /// 取得するメッセージのプレースホルダが
        /// 指定するargsの数よりも多い場合、
        /// FormatException がスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void GetMessageTest28()
        {
            // 準備
            string messageId = "TERASOLUNA_PRACE_HOLDER_3";
            object[] args = { "TERASOLUNA" };

            // 実施
            try
            {
                string actualMessage = MessageManager.GetMessage(messageId, args);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (FormatException ex)
            {
                // 確認
                Assert.AreEqual("インデックス (0 ベース) は 0 以上で、引数リストのサイズよりも小さくなければなりません。", ex.Message);
            }
            catch (Exception ex)
            {
                Assert.Fail("到達してはいけないコードです。" + ex.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MessageManager), LogLevel.INFO));
        }
    }
}
