// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Conversion;
using TERASOLUNA.Fw.Client.Configuration.Conversion;
using System.Data;
using TERASOLUNA.Fw.ClientUT.Configuration;
using TERASOLUNA.Fw.Client.Configuration;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.ClientUT.Conversion;
using TERASOLUNA.Fw.ClientUT.Conversion.DataSetConverterTestData;
using TERASOLUNA.Fw.Common.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.ClientUT.Conversion
{
    ///<summary>
    /// DataSetConverterのテストクラスです。
    ///</summary>

    [TestClass()]
    public class DataSetConverterTest
    {


        private TestContext testContextInstance;

        ///<summary>
        /// 変換設定ファイル
        ///</summary>
        private const string CONFIG_FILE01 = "DataSetConverter_ConversionConfigSection01.config";

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            // 準備
            ConversionConfigurationSection section =
                ConfigTestHelper.GetConversionConfigurationSectionFromCustomFile(CONFIG_FILE01);
            section.CreateConvertElements();
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._conversionConfigSection = section;
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            TERASOLUNA_Fw_Client_Configuration_TerasolunaClientConfigurationAccessor._conversionConfigSection = null;
        }
        
        #endregion

        ///<summary>
        /// DataSetConverter () のテスト
        ///</summary>
        ///<remarks>
        /// デフォルトコンストラクタで正常にインスタンス化できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            DataSetConverter target = new DataSetConverter();
            Assert.IsNotNull(target);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idにnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest01()
        {
            DataSetConverter target = new DataSetConverter();

            string id = null;
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSetConverterDataSet01();

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("id", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "id"));
        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idに空文字を渡した場合、ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest02()
        {
            DataSetConverter target = new DataSetConverter();

            string id = string.Empty;
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSetConverterDataSet01();
            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "id"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentException), "E_EMPTY_STRING", "id"));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idに設定に存在しない文字列を渡した場合、TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest03()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "nothing";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSetConverterDataSet01();
            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TYPE_NOT_FOUND", id), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TYPE_NOT_FOUND", id));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idに設定に存在する文字列を渡した場合、正しく変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest04()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSetConverterDataSet01 destData = new DataSetConverterDataSet01();

            target.Convert(id, srcData, destData);

            //変換元
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", srcData.Tables["DataTable1"].Rows[0]["Column3"]);

            //変換先
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column2"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数srcDataにnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest05()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = null;
            DataSetConverterDataSet01 destData = new DataSetConverterDataSet01();

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("srcData", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "srcData"));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 空のテーブル、空の変換設定を指定した場合、変換が実施されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest06()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert3";
            DataSet srcData = new DataSet();

            DataSetConverterDataSet01 destData = new DataSetConverterDataSet01();

            target.Convert(id, srcData, destData);

            //変換先
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, destData.Tables["DataTable1"].Rows.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換元にRowが無い場合、TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest07()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();

            DataSetConverterDataSet01 destData = new DataSetConverterDataSet01();
            string srcTableName = "DataTable1";
            
            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// DataSet内にテーブルが複数存在する場合、正しく変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet02_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest08()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert4";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet02_01.xml");
            srcData.AcceptChanges();
            DataSetConverterDataSet02 destData = new DataSetConverterDataSet02();

            target.Convert(id, srcData, destData);

            //変換元
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b1", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c1", srcData.Tables["DataTable1"].Rows[0]["Column3"]);

            Assert.IsNotNull(srcData.Tables["DataTable2"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable2"].Rows.Count);
            Assert.AreEqual("a2", srcData.Tables["DataTable2"].Rows[0]["Column1"]);

            Assert.IsNotNull(srcData.Tables["DataTable3"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable3"].Rows.Count);
            Assert.AreEqual("a3", srcData.Tables["DataTable3"].Rows[0]["Column1"]);

            //変換先
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("a2", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("a3", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            Assert.IsNotNull(destData.Tables["DataTable2"].Rows);
            Assert.IsNotNull(destData.Tables["DataTable3"].Rows);
            Assert.AreEqual(0, destData.Tables["DataTable2"].Rows.Count);
            Assert.AreEqual(0, destData.Tables["DataTable3"].Rows.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// レコードの無いテーブルを変換元とした場合、変換が実施されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest09()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSetConverterDataSet01();
            DataSet destData = new DataSetConverterDataSet01();

            target.Convert(id, srcData, destData);

            //変換元
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, srcData.Tables["DataTable1"].Rows.Count);

            //変換先
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, destData.Tables["DataTable1"].Rows.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換元のレコード数が3の場合、正常に変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_03.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest10()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_03.xml");
            srcData.AcceptChanges();
            DataSetConverterDataSet01 destData = new DataSetConverterDataSet01();

            target.Convert(id, srcData, destData);

            //変換元
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(3, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b1", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c1", srcData.Tables["DataTable1"].Rows[0]["Column3"]);
            Assert.AreEqual("a2", srcData.Tables["DataTable1"].Rows[1]["Column1"]);
            Assert.AreEqual("b2", srcData.Tables["DataTable1"].Rows[1]["Column2"]);
            Assert.AreEqual("c2", srcData.Tables["DataTable1"].Rows[1]["Column3"]);
            Assert.AreEqual("a3", srcData.Tables["DataTable1"].Rows[2]["Column1"]);
            Assert.AreEqual("b3", srcData.Tables["DataTable1"].Rows[2]["Column2"]);
            Assert.AreEqual("c3", srcData.Tables["DataTable1"].Rows[2]["Column3"]);

            //変換先
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(3, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("a2", destData.Tables["DataTable1"].Rows[1]["Column2"]);
            Assert.AreEqual("a3", destData.Tables["DataTable1"].Rows[2]["Column2"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数destDataにnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest11()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSetConverterDataSet01 destData = null;

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("destData", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "destData"));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換先のテーブルが０、且つ変換設定がされていない場合は、何もされないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest12()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert2";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();

            target.Convert(id, srcData, destData);

            //変換元
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", srcData.Tables["DataTable1"].Rows[0]["Column3"]);
            //変換先
            Assert.IsNotNull(destData.Tables);
            Assert.AreEqual(0, destData.Tables.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換先にテーブルが存在しない場合、TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest13()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string destTableName = "DataTable1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, destTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, destTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// すでにレコードが登録されているテーブルを変換先とした場合、ArgumenExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest14()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string tableName = "DataTable1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_DEST_DATA", id, tableName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentException), "E_CONVERSION_INVALID_DEST_DATA", id, tableName));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換ルールにColumnが存在しない場合、変換が実施されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest15()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert2";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSetConverterDataSet01 destData = new DataSetConverterDataSet01();

            target.Convert(id, srcData, destData);

            //変換元
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", srcData.Tables["DataTable1"].Rows[0]["Column3"]);

            //変換先
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, destData.Tables["DataTable1"].Rows.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換ルールのColumnにdest属性が存在しない場合、正しく変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest17()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert5";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSetConverterDataSet01 destData = new DataSetConverterDataSet01();

            target.Convert(id, srcData, destData);

            //変換元
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", srcData.Tables["DataTable1"].Rows[0]["Column3"]);

            //変換先
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性が空文字の場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest18()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert6";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = string.Empty;

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性に.がない場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest19()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert7";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = "DataTable1Column1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性で.に前に文字がない場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest20()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert8";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = ".Column1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性で.の後に文字がない場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest21()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert9";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = "DataTable1.";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性で.が2つある場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest22()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert10";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = "DataTable1.Column1.Column1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性で該当テーブルが無い場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest23()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert11";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string srcTableName = "DataTable10";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性で該当カラムが無い場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest24()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert12";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string srcTableName = "DataTable1";
            string srcColName = "Column10";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName));

        }


        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性が空文字の場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest25()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert13";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = string.Empty;

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性に.がない場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest26()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert14";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = "DataTable1Column1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性で.に前に文字がない場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest27()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert15";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = ".Column1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性で.の後に文字がない場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest28()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert16";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = "DataTable1.";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性で.が2つある場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest29()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert17";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string value = "DataTable1.Column1.Column1";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性で該当テーブルが無い場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest30()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert18";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string srcTableName = "DataTable10";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Convert (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性で該当カラムが無い場合、TerasolunaExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_04.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ConvertTest31()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert19";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_04.xml");
            destData.Clear();
            destData.AcceptChanges();
            string srcTableName = "DataTable1";
            string srcColName = "Column10";

            try
            {
                target.Convert(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idにnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest01()
        {
            DataSetConverter target = new DataSetConverter();

            string id = null;
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("id", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "id"));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idに空文字を渡した場合、ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest02()
        {
            DataSetConverter target = new DataSetConverter();

            string id = string.Empty;
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "id"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentException), "E_EMPTY_STRING", "id"));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idに設定に存在しない文字列を渡した場合、TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_04.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest03()
        {
            // 準備
            ConversionConfigurationSection section =
                ConfigTestHelper.GetConversionConfigurationSectionFromCustomFile(CONFIG_FILE01);
            section.CreateConvertElements();
            PrivateType type = new PrivateType(typeof(TerasolunaClientConfiguration));
            type.SetStaticField("_conversionConfigSection", section);

            DataSetConverter target = new DataSetConverter();

            string id = "nothing";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TYPE_NOT_FOUND", id), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TYPE_NOT_FOUND", id));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数idに設定に存在する文字列を渡した場合、正常に逆変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_04.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest04()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_04.xml");
            destData.AcceptChanges();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", srcData.Tables["DataTable1"].Rows[0]["Column2"]);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b1", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c1", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));


        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数destDataにnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest05()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = null;
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("srcData", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "srcData"));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// column要素が0個の場合、正常に逆変換が実施されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest06()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert22";
            DataSet srcData = new DataSet();

            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            target.Reverse(id, srcData, destData);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// テーブルが0で逆変換を試みた場合、TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest07()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert3";
            DataSet srcData = new DataSet();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "DataTable1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// テーブル数が3であった場合、正常に逆変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet02_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet02_02.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest08()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert4";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet02_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet02_02.xml");
            destData.AcceptChanges();
            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("a2", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("a3", srcData.Tables["DataTable1"].Rows[0]["Column3"]);

            Assert.IsNotNull(srcData.Tables["DataTable2"].Rows);
            Assert.AreEqual(0, srcData.Tables["DataTable2"].Rows.Count);
            Assert.IsNotNull(srcData.Tables["DataTable3"].Rows);
            Assert.AreEqual(0, srcData.Tables["DataTable3"].Rows.Count);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b1", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c1", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            Assert.IsNotNull(destData.Tables["DataTable2"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable2"].Rows.Count);
            Assert.AreEqual("a2", destData.Tables["DataTable2"].Rows[0]["Column1"]);

            Assert.IsNotNull(destData.Tables["DataTable3"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable3"].Rows.Count);
            Assert.AreEqual("a3", destData.Tables["DataTable3"].Rows[0]["Column1"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換先テーブルのレコードが変換元を上回っていた場合、ConversionExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest09()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert3";
            DataSet srcData = new DataSetConverterDataSet01();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "DataTable1";
            string destTableName = "DataTable1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConversionException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_NOT_SAME_ROW_COUNT", id, srcTableName, destTableName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ConversionException), "E_CONVERSION_NOT_SAME_ROW_COUNT", id, srcTableName, destTableName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換元テーブルのレコードが変換先を上回っていた場合、ConversionExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_03.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest10()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert3";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_03.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "DataTable1";
            string destTableName = "DataTable1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConversionException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_NOT_SAME_ROW_COUNT", id, srcTableName, destTableName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ConversionException), "E_CONVERSION_NOT_SAME_ROW_COUNT", id, srcTableName, destTableName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換元テーブルのレコードと変換先のレコードが同一だった場合、正常に逆変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest11()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert3";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            target.Reverse(id, srcData, destData);

            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", srcData.Tables["DataTable1"].Rows[0]["Column3"]);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換先のレコードが0だった場合、正常に逆変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest12()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSetConverterDataSet01();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            target.Reverse(id, srcData, destData);

            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column2"]);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 変換先のレコードが3でclearview指定されていた場合、正常に逆変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_03.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest13()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_03.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column2"]);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// destDataにnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest14()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = null;

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("destData", e.ParamName);
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "destData"));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// destDataのテーブル数が0で変換設定が無い場合、逆変換が実施されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest15()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert2";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual(0, destData.Tables.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// destDataのテーブル数が0で逆変換を実施しようとした場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest16()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            string srcTableName = "DataTable1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName));

        }


        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// destDataのレコード数が0の場合、逆変換が実施されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest17()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSetConverterDataSet01();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, srcData.Tables["DataTable1"].Rows.Count);
            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, destData.Tables["DataTable1"].Rows.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// destDataのレコード数が3の場合、逆変換が実施されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_03.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest18()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert1";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_03.xml");
            destData.AcceptChanges();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(3, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("a2", srcData.Tables["DataTable1"].Rows[1]["Column2"]);
            Assert.AreEqual("a3", srcData.Tables["DataTable1"].Rows[2]["Column2"]);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(3, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a1", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b1", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c1", destData.Tables["DataTable1"].Rows[0]["Column3"]);
            Assert.AreEqual("a2", destData.Tables["DataTable1"].Rows[1]["Column1"]);
            Assert.AreEqual("b2", destData.Tables["DataTable1"].Rows[1]["Column2"]);
            Assert.AreEqual("c2", destData.Tables["DataTable1"].Rows[1]["Column3"]);
            Assert.AreEqual("a3", destData.Tables["DataTable1"].Rows[2]["Column1"]);
            Assert.AreEqual("b3", destData.Tables["DataTable1"].Rows[2]["Column2"]);
            Assert.AreEqual("c3", destData.Tables["DataTable1"].Rows[2]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// clear-viewで指定したテーブルが存在しない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest19()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert20";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "Nothing";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_CLEAR_TABLE_NOT_FOUND", srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_CLEAR_TABLE_NOT_FOUND", srcTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// column数が0の場合、逆変換が実施されないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest20()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert2";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(0, srcData.Tables["DataTable1"].Rows.Count);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// column数が3の場合、正常に逆変換されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest21()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert21";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("c", srcData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("b", srcData.Tables["DataTable1"].Rows[0]["Column3"]);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性が存在しない場合
        /// 指定したカラムがコピーされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest23()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert5";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();

            target.Reverse(id, srcData, destData);
            srcData.AcceptChanges();
            destData.AcceptChanges();
            Assert.IsNotNull(srcData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, srcData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", srcData.Tables["DataTable1"].Rows[0]["Column1"]);

            Assert.IsNotNull(destData.Tables["DataTable1"].Rows);
            Assert.AreEqual(1, destData.Tables["DataTable1"].Rows.Count);
            Assert.AreEqual("a", destData.Tables["DataTable1"].Rows[0]["Column1"]);
            Assert.AreEqual("b", destData.Tables["DataTable1"].Rows[0]["Column2"]);
            Assert.AreEqual("c", destData.Tables["DataTable1"].Rows[0]["Column3"]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性が空文字の場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest24()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert6";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = string.Empty;

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性に"."が無い場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest25()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert7";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = "DataTable1Column1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性に"."が1つで、"."の前に文字がない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest26()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert8";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = ".Column1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性に"."が1つで、"."の後ろに文字がない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest27()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert9";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = "DataTable1.";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性に "."が2つある場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest28()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert10";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = "DataTable1.Column1.Column1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性の該当テーブルがない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest29()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert11";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "DataTable10";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// src属性の該当テーブルはあるが、該当カラムがない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest30()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert12";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "DataTable1";
            string srcColName = "Column10";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性が空文字の場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest31()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert13";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = string.Empty;

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
             catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性に"."が無い場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest32()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert14";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = "DataTable1Column1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性に"."が1つで、"."の前に文字がない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest33()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert15";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = ".Column1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性に"."が1つで、"."の後ろに文字がない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest34()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert16";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = "DataTable1.";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性に "."が2つある場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest35()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert17";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string value = "DataTable1.Column1.Column1";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_INVALID_EXPRESSION", value), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_INVALID_EXPRESSION", value));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性の該当テーブルがない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest36()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert18";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "DataTable10";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_TABLE_NOT_FOUND", id, srcTableName, srcData.GetType().FullName));

        }

        ///<summary>
        /// Reverse (string, DataSet, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// dest属性の該当テーブルはあるが、該当カラムがない場合
        /// TerasolunaExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverterDataSet01_01.xml")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionExternal01.config")]
        [DeploymentItem(@"Conversion\DataSetConverterTestData\DataSetConverter_ConversionConfigSection01.config")]
        public void ReverseTest37()
        {
            DataSetConverter target = new DataSetConverter();

            string id = "convert19";
            DataSet srcData = new DataSet();
            srcData.ReadXml("DataSetConverterDataSet01_01.xml");
            srcData.AcceptChanges();
            DataSet destData = new DataSet();
            destData.ReadXml("DataSetConverterDataSet01_01.xml");
            destData.AcceptChanges();
            string srcTableName = "DataTable1";
            string srcColName = "Column10";

            try
            {
                target.Reverse(id, srcData, destData);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                   Assert.AreEqual(MessageManager.GetMessage("E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(DataSetConverter), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetConverter), typeof(TerasolunaException), "E_CONVERSION_COLUMN_NOT_FOUND", id, srcColName, srcData.GetType().FullName, srcTableName));

        }

    }
}
