﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.IO;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Multipart.MultipartValueElement のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Multipart.MultipartValueElement 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class MultipartValueElementTest
    {


        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// CreateBodyStream (Encoding) のテスト
        /// </summary>
        /// <remarks>
        /// 引数の encoding が null の場合、
        /// ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest01()
        {
            // 準備
            Encoding encoding = null;
            MultipartValueElement target = new MultipartValueElement("TERASOLUNA");

            // 実施
            try
            {
                target.CreateBodyStream(encoding);
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("encoding", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(MultipartValueElement), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "encoding"));
        }

        /// <summary>
        ///CreateBodyStream (Encoding) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の encoding が not null 、プロパティの Value が null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest02()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            MultipartValueElement target = new MultipartValueElement("TERASOLUNA");
            TERASOLUNA_Fw_Client_Communication_MultipartValueElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartValueElementAccessor(target);
            accessor._value = null;

            // 実施
            Stream actualStream = target.CreateBodyStream(encoding);

            // 結果の確認
            Assert.AreEqual(0L, actualStream.Length);
            Assert.AreEqual(0L, actualStream.Position);
            Assert.IsTrue(actualStream.CanSeek);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }

        /// <summary>
        ///CreateBodyStream (Encoding) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の encoding が not null 、プロパティの Value が空文字列の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest03()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            MultipartValueElement target = new MultipartValueElement("TERASOLUNA");
            TERASOLUNA_Fw_Client_Communication_MultipartValueElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartValueElementAccessor(target);
            accessor._value = string.Empty;

            // 実施
            Stream actualStream = target.CreateBodyStream(encoding);

            // 結果の確認
            Assert.AreEqual(0L, actualStream.Length);
            Assert.AreEqual(0L, actualStream.Position);
            Assert.IsTrue(actualStream.CanSeek);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }

        /// <summary>
        ///CreateBodyStream (Encoding) のテスト
        ///</summary>
        /// <remarks>
        /// 引数の encoding が not null 、プロパティの Value が not null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateBodyStreamTest04()
        {
            // 準備
            Encoding encoding = Encoding.UTF8;
            string value = "TERASOLUNA";
            MultipartValueElement target = new MultipartValueElement("TERASOLUNA");
            TERASOLUNA_Fw_Client_Communication_MultipartValueElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartValueElementAccessor(target);
            accessor._value = value;

            // 実施
            Stream actualStream = target.CreateBodyStream(encoding);

            // 結果の確認
            Assert.AreEqual(10L, actualStream.Length);
            Assert.AreEqual(0L, actualStream.Position);
            Assert.IsTrue(actualStream.CanSeek);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }

        /// <summary>
        /// CreateHeader () のテスト
        /// </summary>
        /// <remarks>
        /// フィールド name が not null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateHeaderTest01()
        {
            // 準備
            string name = "TERASOLUNA";
            MultipartValueElement target = new MultipartValueElement("TERASOLUNA");
            TERASOLUNA_Fw_Client_Communication_MultipartElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartElementAccessor(target);
            accessor._name = name;

            // 実施
            string actualHeaderString = target.CreateHeader();

            // 確認
            StringBuilder expectedMessage = new StringBuilder();
            expectedMessage.AppendLine(string.Format("Content-Disposition: form-data; name={0}", name));
            expectedMessage.Append("Content-Type: application/x-www-form-urlencoded");
            Assert.AreEqual(expectedMessage.ToString(), actualHeaderString);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }

        /// <summary>
        /// MultipartValueElement (string) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 name が null の場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            // 準備
            string name = null;

            // 実施
            try
            {
                MultipartValueElement target = new MultipartValueElement(name);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                Assert.AreEqual("name", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }

        /// <summary>
        ///MultipartValueElement (string) のテスト
        ///</summary>
        /// <remarks>
        /// 引数 name が空文字列の場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest02()
        {
            // 準備
            string name = string.Empty;

            // 実施
            try
            {
                MultipartValueElement target = new MultipartValueElement(name);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "name"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }

        /// <summary>
        ///MultipartValueElement (string) のテスト
        ///</summary>
        /// <remarks>
        /// 引数 name が not null の場合、
        /// 正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ConstructorTest03()
        {
            // 準備
            string name = "TERASOLUNA";

            // 実施
            MultipartValueElement target = new MultipartValueElement(name);
            
            // 結果
            TERASOLUNA_Fw_Client_Communication_MultipartElementAccessor accessor = new TERASOLUNA_Fw_Client_Communication_MultipartElementAccessor(target);
            Assert.AreSame(name, accessor._name);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }

        /// <summary>
        /// Value のテスト
        /// </summary>
        /// <remarks>
        /// not null 文字列を設定した場合、
        /// 正常に値が設定されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ValueTest01()
        {
            // 準備
            MultipartValueElement target = new MultipartValueElement("TERASOLUNA02");
            string expectedValue = "TERASOLUNA01";

            // 実施
            target.Value = expectedValue;
            string actualValue = target.Value;

            // 結果
            Assert.AreSame(expectedValue, actualValue);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(MultipartValueElement), LogLevel.INFO));
        }
    }
}
