// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Net;
using TERASOLUNA.Fw.Client.Communication;
using System.Xml;
using TERASOLUNA.Fw.Client;
using TERASOLUNA.Fw.Utlib.Logging;
using System.Data;
using TERASOLUNA.Fw.ClientUT.Communication.BinaryFileReceiverData;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
using System.IO;
using TERASOLUNA.Fw.Common.Validation;
namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Receiver.BinaryFileReceiver のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Receiver.BinaryFileReceiver 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class BinaryFileReceiverTest
    {

        private TestContext testContextInstance;

        ///<summary>
        /// 正常な通信先を表すURL文字列です。
        ///</summary>
        ///<remarks>
        ///  <para>
        ///   この定数の値は "http://localhost/TestServer/Index.aspx" です。
        ///  </para>
        ///</remarks>
        private static readonly string TEST_NORMAL_URL = "http://localhost/TestServer/Index.aspx";

        ///<summary>
        /// 存在しないサーバを表すURL文字列です。
        ///</summary>
        ///<remarks>
        ///  <para>
        ///   この定数の値は "http://errorserver/TestServer/Index.aspx" です。
        ///  </para>
        ///</remarks>
        private static readonly string TEST_NO_SERVER_URL = "http://errorserver/DummyServer/Index.aspx";

        /// <summary>
        /// 通信相手に通知するリクエスト名のキー値です。
        /// </summary>
        /// <remarks>この定数の値は "RequestName" です。</remarks>
        private static readonly string REQUEST_NAME = "RequestName";

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// requestがnullの場合、ArgumentNullExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest01()
        {   
            BinaryFileReceiver target = new BinaryFileReceiver();

            // 引数
            HttpWebRequest request = null;

            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download01Local.bmp";

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("request", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ReceiverBase), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "request"));
        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// requestのアドレスが存在しないサーバの場合、WebExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest02()
        {
            BinaryFileReceiver target = new BinaryFileReceiver();

            // 引数
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NO_SERVER_URL);

            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download01Local.bmp";

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///　リクエストがキャンセルされた場合、WebExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest03()
        {
            BinaryFileReceiver target = new BinaryFileReceiver();

            // 引数
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            request.Abort();

            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download01Local.bmp";

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// リクエストが正常に終了した場合、電文ヘッダとボディを正常に受信でき、
        /// 戻り値にResultDataのインスタンスと、レスポンスヘッダの内容とファイルパスが設定され、
        /// 進捗状況通知イベントが実行され、電文と受信ファイルの保存パスがログに出力されることの確認。
        /// ファイルパスにファイルが正しく存在することの確認も行う。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest04()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download01.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download01Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在およびサイズ確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///　reporterがnullの場合、ArgumentNullExceptionスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest05()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download01.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 引数
            IProgressChangeReporter reporter = null;

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("reporter", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "reporter"));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// ダウンロードファイルパスを設定しない場合、ファイルダウンロードが行われないことを確認する。
        /// 戻り値にResultDataのインスタンスと、レスポンスヘッダの内容が設定され、
        /// 戻り値のファイルパスにnullが格納され、電文ログが出力される。
        /// 進捗状況通知イベントが実行され、電文とダウンロードに失敗した旨のログが出力される。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest06()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            string requestName = "BinaryFileReceiver_Download01.bmp";
            SendRequest(request, requestName);

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download01Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;
            // Itemsのキーを設定しない
            reporter._notSetSaveFilePathToItems = true;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.IsNull(result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // イベント実行確認
            Assert.IsFalse(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);
            
            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // ダウンロードの失敗ログ
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FAILED", saveFilePath));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// ダウンロードファイルパスにnullを設定した場合、ファイルダウンロードが行われないことを確認する。
        /// 戻り値にResultDataのインスタンスと、レスポンスヘッダの内容が設定され、
        /// 戻り値のファイルパスにnullが格納され、電文ログが出力される。
        /// 進捗状況通知イベントが実行され、電文とダウンロードに失敗した旨のログが出力される。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest07()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            string requestName = "BinaryFileReceiver_Download01.bmp";
            SendRequest(request, requestName);

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = null;

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.IsNull(result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // イベント実行確認
            Assert.IsFalse(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // ダウンロードの失敗ログ
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FAILED", saveFilePath));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// ダウンロードファイルパスに空文字を設定した場合、ファイルダウンロードが行われないことを確認する。
        /// 戻り値にResultDataのインスタンスと、レスポンスヘッダの内容が設定され、
        /// 戻り値のファイルパスにnullが格納され、電文ログが出力される。
        /// 進捗状況通知イベントが実行され、電文とダウンロードに失敗した旨のログが出力される。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest08()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            string requestName = "BinaryFileReceiver_Download01.bmp";
            SendRequest(request, requestName);

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = string.Empty;

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.IsNull(result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // イベント実行確認
            Assert.IsFalse(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // ダウンロードの失敗ログ
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FAILED", saveFilePath));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// ダウンロードファイルパスに無効な文字が含まれた場合、TerasolunaExceptionがスローされることを確認する。
        /// 戻り値にResultDataのインスタンスと、レスポンスヘッダの内容が設定され、
        /// 戻り値のファイルパスにnullが格納され、電文ログが出力される。
        /// 進捗状況通知イベントが実行され、電文とエラーログが出力される。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest09()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            string requestName = "BinaryFileReceiver_Download01.bmp";
            SendRequest(request, requestName);

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 無効な文字が含まれるファイルパス
            string saveFilePath = "?abc.txt";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_BINARY_DOWNLOAD_ERROR", saveFilePath), e.Message);
                Assert.IsInstanceOfType(e.InnerException, typeof(ArgumentException));
            }

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);
            
            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // エラーログ確認
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(TerasolunaException), "E_BINARY_DOWNLOAD_ERROR", saveFilePath));
        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        /// </summary>
        ///<remarks>
        /// ダウンロードファイルパスに存在しないパスを設定した場合、TerasolunaExceptionがスローされることを確認する。
        /// 戻り値にResultDataのインスタンスと、レスポンスヘッダの内容が設定され、
        /// 戻り値のファイルパスにnullが格納され、電文ログが出力される。
        /// 進捗状況通知イベントが実行され、電文とエラーログが出力される。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest10()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            string requestName = "BinaryFileReceiver_Download01.bmp";
            SendRequest(request, requestName);

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 存在しないファイルパス
            string saveFilePath = @"P:\unknown.txt";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_BINARY_DOWNLOAD_ERROR", saveFilePath), e.Message);
                Assert.IsInstanceOfType(e.InnerException, typeof(UnauthorizedAccessException));
            }

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // エラーログ確認
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(TerasolunaException), "E_BINARY_DOWNLOAD_ERROR", saveFilePath));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// ダウンロードファイルパスに260字以上のパスを設定した場合、TerasolunaExceptionがスローされることを確認する。
        /// 戻り値にResultDataのインスタンスと、レスポンスヘッダの内容が設定され、
        /// 戻り値のファイルパスにnullが格納され、電文ログが出力される。
        /// 進捗状況通知イベントが実行され、電文とエラーログが出力される。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest11()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            string requestName = "BinaryFileReceiver_Download01.bmp";
            SendRequest(request, requestName);

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 260文字が含まれるファイルパス
            string saveFilePath = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_BINARY_DOWNLOAD_ERROR", saveFilePath), e.Message);
                Assert.IsInstanceOfType(e.InnerException, typeof(PathTooLongException));
            }

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // エラーログ確認
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(TerasolunaException), "E_BINARY_DOWNLOAD_ERROR", saveFilePath));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///BufferSizeが1の場合、正常にイベントが実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest13()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download01.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download01Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 1;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));
            
            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Typeにtext/xmlを設定した場合、正常に動作することを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest14()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download02.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download02Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("text/xml", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual(saveFilePath, reporter._saveFilePath);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダがない場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest15()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download03.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download03Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady); 
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダに空文字が設定された場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest17()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download04.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download04Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダに";"文字がない場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest18()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download05.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download05Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダに"="文字がない場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest19()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download06.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download06Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダに"filename"文字列がない場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest20()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download07.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download07Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダの"filename"以降が空文字の場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest21()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download08.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download08Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダの"filename"以降に"?"がない場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest22()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download09.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download09Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダの"filename"以降に"?"が1個しかない場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest23()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download10.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download10Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダに空文字のEncodingが指定された場合、例外がスローされることを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest24()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download11.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download11Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (CommunicationException e)
            {
                Assert.Equals(MessageManager.GetMessage("E_COMMUNICATION_INVALID_ENCODING", string.Empty), e.Message);
            }

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // イベント実行確認
            Assert.IsFalse(reporter._reportProgressChanged);
            Assert.IsFalse(reporter._reportProgressChangedReady);
            Assert.IsFalse(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 例外ログ確認
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(CommunicationException), "E_COMMUNICATION_INVALID_ENCODING", string.Empty));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダに存在しないEncodingが指定された場合、例外がスローされることを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest25()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download12.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download12Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (CommunicationException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_INVALID_ENCODING", "TERASOLUNA"), e.Message);
            }

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // イベント実行確認
            Assert.IsFalse(reporter._reportProgressChanged);
            Assert.IsFalse(reporter._reportProgressChangedReady);
            Assert.IsFalse(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 例外ログ確認
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(CommunicationException), "E_COMMUNICATION_INVALID_ENCODING", "TERASOLUNA"));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダにutf-8のEncodingが指定された場合、正常に動作することを確認する。
        /// サーバから返却された全角のファイル名が正しくreporterに渡されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest26()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download13.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download13Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("テラソルナ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダのエンコード方式に空文字が指定された場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest27()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download14.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download14Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダのエンコード方式に"Q"が指定された場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されないことも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest28()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download15.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download15Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.IsNull(reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダのファイル名に空文字列が指定された場合、正常に動作することを確認する。
        /// サーバから返却された空文字列のファイル名がreporterに渡されることも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest29()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download16.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download16Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual(string.Empty, reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Dispositionヘッダのエンコーディングにutf-8、ファイル名に半角ファイル名が指定された場合、正常に動作することを確認する。
        /// サーバから返却されたファイル名がreporterに渡されることも確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest30()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Download17.bmp");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Download17Local.bmp";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            DownloadResult result = actual as DownloadResult;

            // 戻り値の確認
            Assert.IsNotNull(result.ResultData);
            Assert.AreEqual(saveFilePath, result.DownloadFilePath);
            Assert.AreEqual("application/octet-stream", result.ResponseHeaders["Content-Type"]);

            // ファイルの存在確認
            Assert.IsTrue(System.IO.File.Exists(saveFilePath));
            Assert.AreEqual(result.ResponseHeaders["Content-Length"], System.IO.File.ReadAllBytes(saveFilePath).Length.ToString());

            // イベント実行確認
            Assert.IsTrue(reporter._reportProgressChanged);
            Assert.IsTrue(reporter._reportProgressChangedReady);
            Assert.IsTrue(reporter._downloadReady);
            Assert.AreEqual("ﾃﾗｿﾙﾅ.bmp", reporter._contentFileName);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 受信ファイルの保存パスログ確認
            Assert.IsTrue(LogUTUtil.CheckTraceUseMessageId(target.GetType(), "T_DOWNLOAD_FILE_PATH", saveFilePath));

            // ダウンロードファイルを削除する
            File.Delete(saveFilePath);
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///exceptionヘッダが設定された場合、ServerExceptionがスローされ、例外電文ログが出力されることを確認する
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileReceiverData\BinaryFileReceiver_Error02.xml")]
        public void ReceiveTest31()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Error02");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Error02";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(0, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
            }

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 例外ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("BinaryFileReceiver_Error02.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///exceptionヘッダが設定された場合、ServerExceptionがスローされ、例外電文ログが出力されることを確認する
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileReceiverData\BinaryFileReceiver_Error01.xml")]
        public void ReceiveTest32()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Error01");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Error01";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE01", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE01", e.Errors[0].Message);
            }

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 例外ログ確認
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));

            // 電文ログ確認
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("BinaryFileReceiver_Error01.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///exceptionヘッダが設定された場合、ServerExceptionがスローされ、例外電文ログが出力されることを確認する
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\BinaryFileReceiverData\BinaryFileReceiver_Error03.xml")]
        public void ReceiveTest33()
        {
            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            SendRequest(request, "BinaryFileReceiver_Error03");

            BinaryFileReceiver target = new BinaryFileReceiver();

            // 保存するファイルパス
            string saveFilePath = "BinaryFileReceiver_Error03";

            // 引数
            BinaryFileReceiver_IProgressChangeReporterStub01 reporter = new BinaryFileReceiver_IProgressChangeReporterStub01();

            // プロパティ
            target.BufferSize = 8192;

            // スタブのプロパティ
            reporter._saveFilePath = saveFilePath;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(3, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE11_1", e.Errors[0].Key);
                Assert.AreEqual("ERROR_CODE11_2", e.Errors[1].Key);
                Assert.AreEqual("ERROR_CODE11_3", e.Errors[2].Key);
                Assert.AreEqual("ERROR_MESSAGE11_1", e.Errors[0].Message);
                Assert.AreEqual("ERROR_MESSAGE11_2", e.Errors[1].Message);
                Assert.AreEqual("ERROR_MESSAGE11_3", e.Errors[2].Message);
            }

            // ファイルの存在確認
            Assert.IsFalse(System.IO.File.Exists(saveFilePath));

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));

            // 例外ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));

            // 電文ログ確認
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("BinaryFileReceiver_Error03.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));
        }

        /// <summary>
        /// リクエストを送信する。
        /// </summary>
        /// <param name="requestname"></param>
        private void SendRequest(HttpWebRequest request, string requestname)
        {
            DataSetXmlSender sender = new DataSetXmlSender();

            DataSet dataSet = new DataSet();

            // ヘッダ
            IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, requestname);

            sender.Send(request, dataSet, headerCollection, null);
        }


        /// <summary>
        /// 電文ログの出力チェックを行い、成功時にtrueを返却するメソッド。
        /// </summary>
        /// <param name="request">リクエスト</param>
        /// <param name="paramData">データセット</param>
        /// <returns>成功の場合true</returns>
        private bool CheckResponseLogHeader(HttpWebResponse response, Type type)
        {

            // HTTPヘッダのチェック
            StringBuilder responseHeader = new StringBuilder();
            responseHeader.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_HEADER"));
            foreach (string key in response.Headers.AllKeys)
            {
                responseHeader.AppendLine(string.Format(
                MessageManager.GetMessage("T_DICTIONARY_KEY_VALUE",
                key, response.Headers[key])));
            }

            if (!LogUTUtil.CheckTrace(type, responseHeader.ToString().Trim()))
            {
                return false;
            }

            return true;
        }

    }


}
